/*
 * xml schema utilities
 *
 * Copyright(c) 2009 olyutorskii
 * $Id: SchemaUtilities.java 510 2009-05-01 12:47:30Z olyutorskii $
 */

package jp.sourceforge.jindolf.core;

import java.io.IOException;
import java.io.InputStream;
import java.net.URISyntaxException;
import javax.xml.XMLConstants;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.stream.StreamSource;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;
import org.xml.sax.EntityResolver;
import org.xml.sax.ErrorHandler;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

/**
 * XML スキーマ関係のユーティリティ
 */
public class SchemaUtilities{

    /**
     * スキーマのURL文字列からスキーマを得る。
     * このURLはXmlResourceResolverによるリダイレクト対象となる。
     * @param xsdHref スキーマのURL文字列
     * @return スキーマ
     * @throws URISyntaxException　URIが不正
     * @throws IOException　入出力エラー
     * @throws SAXException スキーマが変
     */
    public static Schema createSchema(String xsdHref)
            throws URISyntaxException,
                   IOException,
                   SAXException {
        XmlResourceResolver resolver = new XmlResourceResolver();

        SchemaFactory xsdSchemaFactory =
            SchemaFactory.newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI);
        xsdSchemaFactory.setResourceResolver(resolver);
        xsdSchemaFactory.setErrorHandler(new StrictHandler());

        InputStream is = resolver.getXMLResourceAsStream(xsdHref);
        StreamSource source = new StreamSource(is);
        Schema schema = xsdSchemaFactory.newSchema(source);

        return schema;
    }

    /**
     * スキーマによる妥当性検証を兼用するDocumentBuilderを生成する。
     * @param schema スキーマ
     * @return DOM用ビルダ
     * @throws ParserConfigurationException パーサ準備失敗
     */
    public static DocumentBuilder createBuilder(Schema schema)
            throws ParserConfigurationException{
        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        factory.setNamespaceAware(true);
        factory.setValidating(false);
        factory.setSchema(schema);

        DocumentBuilder builder = factory.newDocumentBuilder();

        ErrorHandler handler = new StrictHandler();
        builder.setErrorHandler(handler);

        EntityResolver resolver = new XmlResourceResolver();
        builder.setEntityResolver(resolver);

        return builder;
    }

    /**
     * 隠しコンストラクタ
     */
    private SchemaUtilities(){
        super();
        return;
    }

    /**
     * スキーマ検証用の厳密なエラーハンドラ。
     */
    public static class StrictHandler implements ErrorHandler{

        /**
         * {@inheritDoc}
         * @param e {@inheritDoc} エラー情報
         * @throws org.xml.sax.SAXException {@inheritDoc} 引数と同じ物
         */
        public void error(SAXParseException e) throws SAXException{
            throw e;
        }

        /**
         * {@inheritDoc}
         * @param e {@inheritDoc} エラー情報
         * @throws org.xml.sax.SAXException {@inheritDoc} 引数と同じ物
         */
        public void fatalError(SAXParseException e) throws SAXException{
            throw e;
        }

        /**
         * {@inheritDoc}
         * @param e {@inheritDoc} エラー情報
         * @throws org.xml.sax.SAXException {@inheritDoc} 引数と同じ物
         */
        public void warning(SAXParseException e) throws SAXException{
            throw e;
        }

    }

}
