/*
 * pre-defined avatar info
 *
 * Copyright(c) 2009 olyutorskii
 * $Id: PreDefAvatar.java 647 2009-08-09 05:30:09Z olyutorskii $
 */

package jp.sourceforge.jindolf.corelib;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.ParserConfigurationException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

/**
 * プリセット済みAvatarに関する情報。
 * F国運用の時点で20キャラクタ。
 */
public class PreDefAvatar{

    private static final String faceURITemplate =
        "./plugin_wolf/img/face{0,number,#00}.jpg";
    private static final String bodyURITemplate =
        "./plugin_wolf/img/body{0,number,#00}.jpg";

    /**
     * プリセット済みAvatar一覧リストを生成する。
     * @param builder ビルダ
     * @return プリセット済みAvatar一覧リスト
     * @throws IOException
     * @throws ParserConfigurationException
     * @throws SAXException
     */
    public static List<PreDefAvatar> buildPreDefAvatarList(
            DocumentBuilder builder)
            throws IOException,
                   ParserConfigurationException,
                   SAXException {
        List<PreDefAvatar> avatarList;

        Element list = loadAvatarList(builder);
        avatarList = elemToAvatarList(list);
        avatarList = Collections.unmodifiableList(avatarList);

        return avatarList;
    }

    /**
     * プリセットAvatarに関する定義をロードする。
     * @param builder DOMビルダ
     * @return ルート要素
     * @throws javax.xml.parsers.ParserConfigurationException
     * @throws java.io.IOException
     * @throws org.xml.sax.SAXException
     */
    private static Element loadAvatarList(DocumentBuilder builder)
            throws ParserConfigurationException,
                   IOException,
                   SAXException {
        InputStream is = XmlResource.I_URL_AVATARDEF.openStream();

        Document document = builder.parse(is);

        Element root = document.getDocumentElement();
        String tagName = root.getTagName();
        if( ! tagName.equals("preDefinedAvatarList") ){
            throw new SAXException("illegal root " + tagName);
        }

        return root;
    }

    /**
     * 要素内部を探索し、プリセットAvatarを登録する。
     * @param list ルート要素
     * @return プリセットAvatarが登録されたList
     * @throws org.xml.sax.SAXException
     */
    private static List<PreDefAvatar> elemToAvatarList(Element list)
            throws SAXException {
        NodeList elems = list.getElementsByTagName("preDefinedAvatar");
        int avatarNum = elems.getLength();
        if(avatarNum <= 0){
            throw new SAXException("there is no <preDefinedAvatar>");
        }
        List<PreDefAvatar> avatarList = new ArrayList<PreDefAvatar>(avatarNum);

        for(int index=0; index < avatarNum; index++){
            Node node = elems.item(index);
            Element elem = (Element) node;
            PreDefAvatar avatar = buildAvatar(elem);
            avatarList.add(avatar);
        }

        return avatarList;
    }

    /**
     * 個々のプリセットAvatar定義をオブジェクトに変換する。
     * @param avatarDef プリセットAvatar定義要素
     * @return プリセットAvatar定義オブジェクト
     * @throws org.xml.sax.SAXException
     */
    private static PreDefAvatar buildAvatar(Element avatarDef)
            throws SAXException {
        String avatarId  = avatarDef.getAttribute("avatarId");
        String jobTitle  = avatarDef.getAttribute("jobTitle");
        String shortName = avatarDef.getAttribute("shortName");
        String serialNum = avatarDef.getAttribute("serialNum");

        String fullName = jobTitle + "\u0020" + shortName;

        int serialNo;
        try{
            serialNo = Integer.parseInt(serialNum);
        }catch(NumberFormatException e){
            throw new SAXException("illegal number form", e);
        }

        PreDefAvatar avatar;
        try{
            avatar = new PreDefAvatar(avatarId,
                                      fullName,
                                      jobTitle,
                                      shortName,
                                      serialNo );
        }catch(URISyntaxException e){
            throw new SAXException("illegal URL", e);
        }

        return avatar;
    }

    private final String avatarId;
    private final String fullName;
    private final String jobTitle;
    private final String shortName;
    private final int serialNo;
    private final URI faceIconURI;
    private final URI bodyIconURI;

    /**
     * コンストラクタ
     * @param avatarId Avatar識別子
     * @param fullName フルネーム
     * @param jobTitle 職業名
     * @param shortName 省略名
     * @param serialNo 通し番号
     * @throws URISyntaxException 不正なURI
     */
    private PreDefAvatar(String avatarId,
                           String fullName,
                           String jobTitle,
                           String shortName,
                           int serialNo )
        throws URISyntaxException {

        super();

        if(   avatarId  == null
           || fullName  == null
           || jobTitle  == null
           || shortName == null ){
            throw new NullPointerException();
        }

        this.avatarId  = avatarId.intern();
        this.fullName  = fullName.intern();
        this.jobTitle  = jobTitle.intern();
        this.shortName = shortName.intern();
        this.serialNo  = serialNo;

        String urlString;

        urlString = MessageFormat.format(faceURITemplate, this.serialNo);
        this.faceIconURI = new URI(urlString);

        urlString = MessageFormat.format(bodyURITemplate, this.serialNo);
        this.bodyIconURI = new URI(urlString);

        return;
    }

    /**
     * Avatar識別子を返す。
     * @return Avatar識別子
     */
    public String getAvatarId(){
        return this.avatarId;
    }

    /**
     * フルネームを返す。
     * @return フルネーム
     */
    public String getFullName(){
        return this.fullName;
    }

    /**
     * 職業名を返す。
     * @return 職業名
     */
    public String getJobTitle(){
        return this.jobTitle;
    }

    /**
     * 省略名を返す。
     * @return 省略名
     */
    public String getShortName(){
        return this.shortName;
    }

    /**
     * 通し番号を返す。
     * @return 通し番号
     */
    public int getSerialNo(){
        return this.serialNo;
    }

    /**
     * 顔アイコン画像のURIを返す。
     * ※ 相対URI
     * @return 画像URI
     */
    public URI getFaceIconURI(){
        return this.faceIconURI;
    }

    /**
     * 全身アイコン画像のURIを返す。
     * ※ 相対URI
     * @return 画像URI
     */
    public URI getBodyIconURI(){
        return this.bodyIconURI;
    }

    /**
     * このオブジェクトの文字列化
     * @return 文字列
     */
    @Override
    public String toString(){
        StringBuilder result = new StringBuilder();
        result.append(getAvatarId()).append(' ');
        result.append(getFullName()).append(' ');
        result.append(getJobTitle()).append(' ');
        result.append(getShortName()).append(' ');
        result.append(getSerialNo()).append(' ');
        result.append(getFaceIconURI()).append(' ');
        result.append(getBodyIconURI());
        return result.toString();
    }

}
