/*
 * XML resources
 *
 * Copyright(c) 2009 olyutorskii
 * $Id: XmlResource.java 647 2009-08-09 05:30:09Z olyutorskii $
 */

package jp.sourceforge.jindolf.corelib;

import java.io.FileNotFoundException;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

/**
 * このパッケージに含まれるXML関連の各種リソース、およびWeb上の原本。
 */
public class XmlResource{
    public static final String O_XSDBASE =
            "http://jindolf.sourceforge.jp/xml/xsd/301/";
    public static final String I_XSDBASE = "resources/xsd/";
    public static final String I_XMLBASE = "resources/xml/";

    public static final String O_RES_EXTXML =
            "http://www.w3.org/2001/xml.xsd";
    public static final String O_RES_CORETYPE =
            O_XSDBASE + "coreType.xsd";
    public static final String O_RES_COREXML =
            O_XSDBASE + "coreXML.xsd";

    public static final String I_RES_EXTXML =
            I_XSDBASE + "ext/xml-2009-01.xsd";
    public static final String I_RES_CORETYPE =
            I_XSDBASE + "private/coreType.xsd";
    public static final String I_RES_COREXML =
            I_XSDBASE + "private/coreXML.xsd";

    public static final String I_RES_LANDDEF =
            I_XMLBASE + "landDefList.xml";
    public static final String I_RES_AVATARDEF =
            I_XMLBASE + "preDefAvatarList.xml";

    public static final URI O_URI_EXTXML;
    public static final URI O_URI_CORETYPE;
    public static final URI O_URI_COREXML;

    public static final URI I_URI_EXTXML;
    public static final URI I_URI_CORETYPE;
    public static final URI I_URI_COREXML;

    public static final URI I_URI_LANDDEF;
    public static final URI I_URI_AVATARDEF;


    public static final URL I_URL_COREXML;
    public static final URL I_URL_LANDDEF;
    public static final URL I_URL_AVATARDEF;


    /**
     * Web上の原本からリソースへの変換を記述したマップ。
     */
    public static final Map<URI, URI> RESOLVE_MAP;

    static{
        O_URI_EXTXML   = loadOuter(O_RES_EXTXML);
        O_URI_CORETYPE = loadOuter(O_RES_CORETYPE);
        O_URI_COREXML  = loadOuter(O_RES_COREXML);

        try{
            I_URI_EXTXML    = loadInner(I_RES_EXTXML);
            I_URI_CORETYPE  = loadInner(I_RES_CORETYPE);
            I_URI_COREXML   = loadInner(I_RES_COREXML);
            I_URI_LANDDEF   = loadInner(I_RES_LANDDEF);
            I_URI_AVATARDEF = loadInner(I_RES_AVATARDEF);
        }catch(FileNotFoundException e){
            throw new ExceptionInInitializerError(e);
        }catch(URISyntaxException e){
            throw new ExceptionInInitializerError(e);
        }

        try{
            I_URL_COREXML   = I_URI_COREXML.toURL();
            I_URL_LANDDEF   = I_URI_LANDDEF.toURL();
            I_URL_AVATARDEF = I_URI_AVATARDEF.toURL();
        }catch(MalformedURLException e){
            throw new ExceptionInInitializerError(e);
        }

        Map<URI, URI> map = new HashMap<URI, URI>();
        map.put(O_URI_EXTXML,   I_URI_EXTXML);
        map.put(O_URI_CORETYPE, I_URI_CORETYPE);
        map.put(O_URI_COREXML,  I_URI_COREXML);
        RESOLVE_MAP = Collections.unmodifiableMap(map);
    }

    /**
     * 内部リソースのURIを得る。
     * @param res リソース名
     * @return URI
     * @throws FileNotFoundException リソースが見つからない。
     */
    private static URI loadInner(String res)
            throws FileNotFoundException,
                   URISyntaxException {
        URL url = XmlResource.class.getResource(res);
        if(url == null) throw new FileNotFoundException(res);
        URI result = url.toURI().normalize();
        return result;
    }

    /**
     * 外部リソースのURIを得る。
     * @param http 外部リソース名
     * @return URI
     */
    private static URI loadOuter(String http){
        URI uri = URI.create(http).normalize();
        return uri;
    }

    /**
     * 隠れコンストラクタ
     */
    private XmlResource(){
        super();
        return;
    }

}
