/*
 * basic handler for XHTML
 * 
 * Copyright(c) 2009 olyutorskii
 * $Id: BasicHandler.java 651 2009-08-09 06:19:25Z olyutorskii $
 */

package jp.sourceforge.jindolf.parser;

import jp.sourceforge.jindolf.corelib.PeriodType;
import jp.sourceforge.jindolf.corelib.VillageState;

/**
 * 人狼BBSの各種XHTMLの基本的な構造をパースするためのハンドラ。
 * このハンドラの全メソッドはパーサ{@link HtmlParser}により呼ばれる。
 *
 * パーサはパース開始時に{@link #startParse(DecodedContent)}を呼び、
 * パース終了直前に{@link #endParse()}を呼ぶ。
 * その間に他の様々なメソッドが呼び出される。
 *
 * 一部のメソッドに渡される{@link DecodedContent}オブジェクトはmutableである。
 * 後々で内容が必要になるならば、ハンドラはこの内容の必要な箇所をコピーして
 * 保存しなければならない。
 *
 * フラグメントや属性値中の文字参照記号列の解釈はハンドラ側の責務とする。
 *
 * 各メソッドは{@link HtmlParseException}をスローすることにより、
 * パース作業を中断させることができる。
 */
public interface BasicHandler{

    /**
     * パース開始の通知を受け取る。
     * @param content これからパースを始めるXHTML文字列
     * @throws HtmlParseException パースエラー
     */
    public void startParse(DecodedContent content) throws HtmlParseException;

    /**
     * titleタグの内容の通知を受け取る。
     * 例：「人狼BBS:F F2019 新緑の村」。
     * @param content パース対象文字列
     * @param start タイトル開始位置
     * @param end タイトル終了位置
     * @throws HtmlParseException パースエラー
     */
    public void pageTitle(DecodedContent content, int start, int end)
            throws HtmlParseException;

    /**
     * ログイン名(ID)の通知を受け取る。
     * ログインせずに得られたページがパース対象であるなら、呼ばれない。
     * F国のみで動作確認。
     * @param content パース対象文字列
     * @param start ログイン名開始位置
     * @param end ログイン名終了位置
     * @throws HtmlParseException パースエラー
     */
    public void loginName(DecodedContent content, int start, int end)
            throws HtmlParseException;

    /**
     * 村の名前の通知を受け取る。
     * 国名と番号と愛称に分解するのはハンドラ側の責務。
     * 例：「F2019 新緑の村」。
     * @param content パース対象文字列
     * @param start 村名開始位置
     * @param end 村名終了位置
     * @throws HtmlParseException パースエラー
     */
    public void villageName(DecodedContent content, int start, int end)
            throws HtmlParseException;

    /**
     * 次回更新時刻の通知を受け取る。
     * 既に終了した村がパース対象の場合、あまり月日に意味はないかも。
     * @param month 更新月
     * @param day 更新日
     * @param hour 更新時
     * @param minute 更新分
     * @throws HtmlParseException パースエラー
     */
    public void commitTime(int month, int day, int hour, int minute)
            throws HtmlParseException;

    /**
     * 他の日へのリンクの通知を受け取る。
     * 複数回呼ばれる場合がある。
     * @param content パース対象文字列
     * @param anchorStart aタグhref属性値の開始位置。無ければ負の値。
     * @param anchorEnd aタグhref属性値の終了位置。無ければ負の値。
     * @param periodType 日のタイプ。「終了」ならnull。
     * @param day 日にち。「プロローグ」、「エピローグ」、「終了」では-1。
     * @throws HtmlParseException パースエラー
     */
    public void periodLink(DecodedContent content,
                            int anchorStart, int anchorEnd,
                            PeriodType periodType, int day)
            throws HtmlParseException;

    /**
     * 村一覧リスト内の個別の村情報の通知を受け取る。
     * @param content パース対象文字列
     * @param urlStart URL開始位置
     * @param urlEnd URL終了位置
     * @param vnameStart 村名開始位置
     * @param vnameEnd 村名終了位置
     * @param hh 更新時。不明なら負の数。
     * @param mm 更新分。不明なら負の数。
     * @param state 村の状態
     */
    public void villageRecord(DecodedContent content,
                                int urlStart, int urlEnd,
                                int vnameStart, int vnameEnd,
                                int hh, int mm,
                                VillageState state )
            throws HtmlParseException;

    /**
     * パースの終了の通知を受け取る。
     * @throws HtmlParseException パースエラー
     */
    public void endParse() throws HtmlParseException;

    // TODO 「全て表示」リンクの検出メソッドは必要？
    // TODO 表示モード切り替え(人狼墓全)リンクの検出メソッドは必要？
    // TODO 「次の日へ」リンクの検出メソッドは必要？
    // TODO 投票先、襲撃先プルダウンリストの検出メソッドは必要？
    // TODO 霊能結果の検出メソッドは必要？
    // TODO 発言フォームの検出メソッドは必要？

}
