/*
 * content builder
 *
 * Copyright(c) 2009 olyutorskii
 * $Id: ContentBuilder.java 578 2009-08-03 17:31:06Z olyutorskii $
 */

package jp.sourceforge.jindolf.parser;

import java.nio.charset.CharsetDecoder;

/**
 * {@link StreamDecoder}からの通知に従い、
 * {@link DecodedContent}へとデコードする。
 */
public class ContentBuilder implements DecodeHandler{

    protected DecodedContent content;

    protected boolean hasByte1st;
    protected byte byte1st;

    /**
     * コンストラクタ。
     * 長さ0で空の{@link DecodedContent}がセットされる。
     */
    public ContentBuilder(){
        super();
        setDecodedContent(new DecodedContent());
        return;
    }

    /**
     * 任意の{@link DecodedContent}をデコード先対象にセットする。
     * @param content デコード先
     * @throws NullPointerException 引数がnullだった場合
     */
    public void setDecodedContent(DecodedContent content)
            throws NullPointerException{
        if(content == null) throw new NullPointerException();
        this.content = content;
        init();
        return;
    }

    /**
     * デコード先対象の{@link DecodedContent}を取得する。
     * @return デコード先対象
     */
    public DecodedContent getContent(){
        return this.content;
    }

    /**
     * デコード処理の初期化。
     */
    protected void init(){
        this.content.init();
        this.hasByte1st = false;
        this.byte1st = 0x00;
        return;
    }

    /**
     * エラー情報をフラッシュする。
     */
    protected void flushError(){
        if(this.hasByte1st){
            this.content.addDecodeError(this.byte1st);
            this.hasByte1st = false;
        }
        return;
    }

    /**
     * {@inheritDoc}
     * @param decoder {@inheritDoc}
     * @throws DecodeException {@inheritDoc}
     */
    public void startDecoding(CharsetDecoder decoder)
            throws DecodeException{
        init();
        return;
    }

    /**
     * {@inheritDoc}
     * @throws DecodeException {@inheritDoc}
     */
    public void endDecoding()
            throws DecodeException{
        flushError();
        return;
    }

    /**
     * {@inheritDoc}
     * @param seq {@inheritDoc}
     * @throws DecodeException {@inheritDoc}
     */
    public void charContent(CharSequence seq)
            throws DecodeException{
        flushError();
        this.content.append(seq);
        return;
    }

    /**
     * {@inheritDoc}
     * @param errorArray {@inheritDoc}
     * @param offset {@inheritDoc}
     * @param length {@inheritDoc}
     * @throws DecodeException {@inheritDoc}
     */
    public void decodingError(byte[] errorArray, int offset, int length)
            throws DecodeException{
        int limit = offset + length;
        for(int bpos = offset; bpos < limit; bpos++){
            byte bval = errorArray[bpos];
            if( ! this.hasByte1st){
                if(ShiftJis.isShiftJIS1stByte(bval)){
                    this.byte1st = bval;
                    this.hasByte1st = true;
                }else{
                    this.content.addDecodeError(bval);
                }
            }else{
                if(ShiftJis.isShiftJIS2ndByte(bval)){
                    this.content.addDecodeError(this.byte1st, bval);
                }else{
                    this.content.addDecodeError(this.byte1st);
                    this.content.addDecodeError(bval);
                }
                this.hasByte1st = false;
            }
        }

        return;
    }

}
