/*
 * invalid Shift_JIS decoding information
 *
 * Copyright(c) 2009 olyutorskii
 * $Id: DecodeErrorInfo.java 578 2009-08-03 17:31:06Z olyutorskii $
 */

package jp.sourceforge.jindolf.parser;

/**
 * 不正な Shift_JIS デコードの情報。
 * 1バイトもしくは2バイトで構成される。
 * 1バイトの場合はエンコーディングに関するエラー。
 * 2バイトの場合は文字集合に関するエラー。
 */
public class DecodeErrorInfo{

    private final int charPos;
    private final boolean has2nd;
    private final byte rawByte1st;
    private final byte rawByte2nd;

    /**
     * コンストラクタ
     * @param charPos デコードエラーで置き換えられた文字列の開始位置
     * @param rawByte1st デコードエラーを引き起こした最初のバイト値
     * @param rawByte2nd デコードエラーを引き起こした2番目のバイト値
     */
    public DecodeErrorInfo(int charPos,
                             byte rawByte1st,
                             byte rawByte2nd){
        super();

        this.charPos = charPos;
        this.has2nd = true;
        this.rawByte1st = rawByte1st;
        this.rawByte2nd = rawByte2nd;

        return;
    }

    /**
     * コンストラクタ
     * @param charPos デコードエラーで置き換えられた文字列の開始位置
     * @param rawByte1st デコードエラーを引き起こしたバイト値
     */
    public DecodeErrorInfo(int charPos,
                             byte rawByte1st){
        super();

        this.charPos = charPos;
        this.has2nd = false;
        this.rawByte1st = rawByte1st;
        this.rawByte2nd = 0x00;

        return;
    }

    /**
     * デコードエラーで置き換えられた文字列の開始位置を返す。
     * @return デコードエラーで置き換えられた文字列の開始位置
     */
    public int getCharPosition(){
        return this.charPos;
    }

    /**
     * 2バイト目の情報を持つか判定する。
     * @return 2バイト目の情報を持つならtrue
     */
    public boolean has2nd(){
        return this.has2nd;
    }

    /**
     * 1バイト目の値を返す。
     * @return 1バイト目の値
     */
    public byte getRawByte1st(){
        return this.rawByte1st;
    }

    /**
     * 2バイト目の値を返す。
     * @return 2バイト目の値
     * @throws IllegalStateException 2バイト目の情報を把持していないとき
     */
    public byte getRawByte2nd() throws IllegalStateException{
        if( ! this.has2nd ) throw new IllegalStateException();
        return this.rawByte2nd;
    }

    /**
     * 出現位置のみが違う複製オブジェクトを生成する。
     * @param gap 出現位置から引きたい値。正の値なら文字開始位置に向かう。
     * @return 複製オブジェクト
     */
    public DecodeErrorInfo createGappedClone(int gap){
        DecodeErrorInfo result;

        int newPos = this.charPos - gap;
        if(has2nd){
            result = new DecodeErrorInfo(newPos,
                                         this.rawByte1st, this.rawByte2nd);
        }else{
            result = new DecodeErrorInfo(newPos, this.rawByte1st);
        }

        return result;
    }

    /**
     * {@inheritDoc}
     * @return {@inheritDoc}
     */
    @Override
    public String toString(){
        StringBuilder result = new StringBuilder();

        result.append("start:").append(this.charPos).append(' ');

        String hex;
        hex = Integer.toHexString(this.rawByte1st & 0xff);
        result.append(hex);

        if(this.has2nd){
            hex = Integer.toHexString(this.rawByte2nd & 0xff);
            result.append(':').append(hex);
        }

        return result.toString();
    }

    // TODO Comparatorの用意

}
