/*
 * decoded content
 *
 * Copyright(c) 2009 olyutorskii
 * $Id: DecodedContent.java 578 2009-08-03 17:31:06Z olyutorskii $
 */

package jp.sourceforge.jindolf.parser;

import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

/**
 * ShiftJISデコードエラー情報を含む文字列。
 * デコードエラーを起こした箇所は代替文字{@link #ALTCHAR}で置き換えられる。
 * マルチスレッドには非対応。
 */
public class DecodedContent implements CharSequence{

    /** 代替文字 */
    public static final char ALTCHAR = '?';
    // HTMLで使うなら < や > や & や " や ' はやめて！

    private final StringBuilder rawContent = new StringBuilder();

    private List<DecodeErrorInfo> decodeError;
    private List<DecodeErrorInfo> unmodList;

    /**
     * コンストラクタ
     */
    public DecodedContent(){
        super();
        init();
        return;
    }

    /**
     * 初期化。
     * 長さ0の文字列＆デコードエラー無しの状態になる。
     * コンストラクタで新インスタンスを作るより低コスト。
     */
    protected void init(){
        this.rawContent.setLength(0);
        this.decodeError = null;
        return;
    }

    /**
     * デコードエラー格納オブジェクトが無ければ用意する。
     */
    private void assignDecodeError(){
        if(this.decodeError == null){
            this.decodeError = new LinkedList<DecodeErrorInfo>();
        }
        return;
    }

    /**
     * 変更不可ビューオブジェクトが無ければ用意する。
     */
    private void assignUnmodList(){
        assignDecodeError();
        if(this.unmodList == null){
            this.unmodList = Collections.unmodifiableList(this.decodeError);
        }
        return;
    }

    /**
     * デコードエラーを含むか判定する。
     * @return デコードエラーを含むならtrue
     */
    public boolean hasDecodeError(){
        if(this.decodeError == null) return false;
        if(this.decodeError.isEmpty()) return false;
        return true;
    }

    /**
     * デコードエラーの一覧を取得する。
     * @return デコードエラーの一覧
     */
    public List<DecodeErrorInfo> getDecodeErrors(){
        assignUnmodList();
        return this.unmodList;
    }

    /**
     * 生の文字列を得る。
     * 高速なCharSequenceアクセス用途。
     * @return 生の文字列。
     */
    public CharSequence getRawContent(){
        return this.rawContent;
    }

    /**
     * {@inheritDoc}
     * @param index {@inheritDoc}
     * @return {@inheritDoc}
     */
    public char charAt(int index){
        return this.rawContent.charAt(index);
    }

    /**
     * {@inheritDoc}
     * @return {@inheritDoc}
     */
    public int length(){
        return this.rawContent.length();
    }

    /**
     * {@inheritDoc}
     * @param start {@inheritDoc}
     * @param end {@inheritDoc}
     * @return {@inheritDoc}
     */
    public CharSequence subSequence(int start, int end){
        return this.rawContent.subSequence(start, end);
    }

    /**
     * 文字列を追加する。
     * @param seq 追加する文字列
     * @return thisオブジェクト
     */
    protected DecodedContent append(CharSequence seq){
        this.rawContent.append(seq);
        return this;
    }

    /**
     * 文字列を追加する。
     * @param seq 追加する文字列
     * @param startPos 開始位置
     * @param endPos 終了位置
     * @return thisオブジェクト
     */
    protected DecodedContent append(CharSequence seq,
                                     int startPos, int endPos){
        this.rawContent.append(seq, startPos, endPos);
        return this;
    }

    /**
     * デコードエラーを追加する
     * @param errorInfo デコードエラー
     */
    protected void addPlainDecodeError(DecodeErrorInfo errorInfo){
        assignDecodeError();
        this.decodeError.add(errorInfo);
        return;
    }

    /**
     * 代替文字とともにデコードエラーを追加する
     * @param errorInfo デコードエラー
     */
    protected void addDecodeError(DecodeErrorInfo errorInfo){
        addPlainDecodeError(errorInfo);
        this.rawContent.append(ALTCHAR);
        return;
    }

    /**
     * 代替文字とともにデコードエラーを追加する
     * @param b1 1バイト目の値
     */
    protected void addDecodeError(byte b1){
        DecodeErrorInfo errInfo =
                new DecodeErrorInfo(this.rawContent.length(), b1);
        addDecodeError(errInfo);
        return;
    }

    /**
     * 代替文字とともにデコードエラーを追加する
     * @param b1 1バイト目の値
     * @param b2 2バイト目の値
     */
    protected void addDecodeError(byte b1, byte b2){
        DecodeErrorInfo errInfo =
                new DecodeErrorInfo(this.rawContent.length(), b1, b2);
        addDecodeError(errInfo);
        return;
    }

    /**
     * {@inheritDoc}
     * @return {@inheritDoc}
     */
    @Override
    public String toString(){
        return this.rawContent.toString();
    }

}
