/*
 * XHTML parser
 *
 * Copyright(c) 2009 olyutorskii
 * $Id: HtmlParser.java 651 2009-08-09 06:19:25Z olyutorskii $
 */

package jp.sourceforge.jindolf.parser;

import java.util.regex.Pattern;
import jp.sourceforge.jindolf.corelib.PeriodType;
import jp.sourceforge.jindolf.corelib.VillageState;

/**
 * 人狼BBS各種XHTML文字列のパースを行いハンドラに通知する。
 */
public class HtmlParser extends AbstractParser{

    private BasicHandler handler;
    private final TalkParser     talkParser     = new TalkParser(this);
    private final SysEventParser sysEventParser = new SysEventParser(this);

    /**
     * コンストラクタ
     */
    public HtmlParser(){
        super();
        return;
    }

    /**
     * {@link BasicHandler}ハンドラを登録する。
     * @param handler
     */
    public void setBasicHandler(BasicHandler handler){
        this.handler = handler;
        return;
    }

    /**
     * {@link TalkHandler}ハンドラを登録する。
     * @param handler ハンドラ
     */
    public void setTalkHandler(TalkHandler handler){
        this.talkParser.setTalkHandler(handler);
        return;
    }

    /**
     * {@link SysEventHandler}ハンドラを登録する。
     * @param handler ハンドラ
     */
    public void setSysEventHandler(SysEventHandler handler){
        this.sysEventParser.setSysEventHandler(handler);
        return;
    }

    private static final Pattern O_HTML_PATTERN =
            compile("<html\u0020");
    private static final Pattern TITLE_PATTERN =
            compile("<title>([^<]*)</title>");
    private static final Pattern O_BODY_PATTERN =
            compile("<body>");
    private static final Pattern O_TABLE_PATTERN =
            compile("<table\u0020");

    /**
     * XHTML先頭部分のパース
     * @throws HtmlParseException パースエラー
     */
    private void parseHead() throws HtmlParseException{
        setErrorMessage("lost head part");

        findAffirm(O_HTML_PATTERN);
        shrinkRegion();

        findAffirm(TITLE_PATTERN);
        int titleStart = matchStart(1);
        int titleEnd   = matchEnd(1);
        shrinkRegion();

        this.handler.pageTitle(getContent(), titleStart, titleEnd );

        findAffirm(O_BODY_PATTERN);
        shrinkRegion();

        findAffirm(O_TABLE_PATTERN);
        shrinkRegion();

        findAffirm(O_TABLE_PATTERN);
        shrinkRegion();

        return;
    }

    private static final Pattern LOGINFORM_PATTERN =
            compile(
                  "("
                    +"<form"
                    +"\u0020" + "action=\"index\\.rb\""
                    +"\u0020" + "method=\"post\""
                    +"\u0020" + "class=\"login_form\""
                    +">"
                + ")|("
                    +"<div"
                    +"\u0020" + "class=\"login_form\""
                    +">"
                + ")"
            );
    private static final Pattern C_EDIV_PATTERN =
            compile(
                  SP_I
                + "<a\u0020href=\"[^\"]*\">[^<]*</a>"
                + SP_I
                + "</div>"
            );
    private static final Pattern USERID_PATTERN =
            compile(
                  "name=\"user_id\""
                + "\u0020"
                + "value=\"([^\"]*)\""
            );
    private static final Pattern C_FORM_PATTERN =
            compile("</form>");
    
    /**
     * ログインフォームのパース。
     * ログイン名までの認識を確認したのはF国のみ。
     * @throws HtmlParseException パースエラー
     */
    private void parseLoginForm() throws HtmlParseException{
        setErrorMessage("lost login form");

        boolean isLand_E_Form;
        findAffirm(LOGINFORM_PATTERN);
        if(isGroupMatched(1)){
            isLand_E_Form = false;
        }else{                         // E国ログインフォーム検出
            isLand_E_Form = true;
        }
        shrinkRegion();

        if(isLand_E_Form){
            lookingAtAffirm(C_EDIV_PATTERN);
            shrinkRegion();
            return;
        }else{
            findAffirm(USERID_PATTERN);
            int loginStart = matchStart(1);
            int loginEnd   = matchEnd(1);
            shrinkRegion();

            if(loginStart < loginEnd){
                this.handler.loginName(getContent(), loginStart, loginEnd);
            }

            findAffirm(C_FORM_PATTERN);
            shrinkRegion();
        }

        return;
    }

    private static final Pattern VILLAGEINFO_PATTERN =
            compile(
                 "([^<]+?)" +SP_I          // 最短一致数量子
                +"<strong>"
                    +"\uff08"
                    +"([0-9]+)"                       // 月
                    +"/"
                    +"([0-9]+)"                       // 日
                    +"\u0020"
                    +"(?:(?:(午前)|(午後))\u0020)?"  // AMPM
                    +"([0-9]+)"                       // 時
                    +"(?:時\u0020|\\:)"
                    +"([0-9]+)"                       // 分
                    +"分?\u0020に更新"
                    +"\uff09"
                +"</strong>"
            );

    /**
     * 村に関する各種情報をパース。
     * @throws HtmlParseException パースエラー
     */
    private void parseVillageInfo() throws HtmlParseException{
        setErrorMessage("lose village information");

        sweepSpace();

        lookingAtAffirm(VILLAGEINFO_PATTERN);
        int vnameStart = matchStart(1);
        int vnameEnd   = matchEnd(1);

        int month = parseGroupedInt(2);
        int day   = parseGroupedInt(3);
        int hh    = parseGroupedInt(6);
        int mm    = parseGroupedInt(7);
        if(isGroupMatched(5)){  // 午後指定
            hh = (hh + 12) % 24;
        }
        shrinkRegion();

        this.handler.villageName(getContent(), vnameStart, vnameEnd);
        this.handler.commitTime(month, day, hh, mm);

        return;
    }

    private static final Pattern O_PARAG_PATTERN = compile("<p>");
    private static final Pattern PERIODLINK_PATTERN =
            compile(
            "("
                + "<span\u0020class=\"time\">"
            +")|(?:"
                + "<a\u0020href=\"([^\"]*)\">"
            +")|("
                + "</p>"
            +")"
            );
    private static final Pattern PERIOD_PATTERN =
            compile(
                "(プロローグ)" +
            "|"+
                "(エピローグ)" +
            "|"+
                "(終了)" +
            "|"+
                "([0-9]+)日目"
            );
    private static final Pattern C_SPAN_PATTERN   = compile("</span>");
    private static final Pattern C_ANCHOR_PATTERN = compile("</a>");

    /**
     * Period間リンクをパース。
     * @throws HtmlParseException パースエラー
     */
    private void parsePeriodLink() throws HtmlParseException{
        setErrorMessage("lost period link");

        findAffirm(O_PARAG_PATTERN);
        shrinkRegion();

        for(;;){
            Pattern closePattern;
            int anchorStart = -1;
            int anchorEnd   = -1;
            
            sweepSpace();
            lookingAtAffirm(PERIODLINK_PATTERN);
            if(isGroupMatched(1)){
                closePattern = C_SPAN_PATTERN;
            }else if(isGroupMatched(2)){
                closePattern = C_ANCHOR_PATTERN;
                anchorStart = matchStart(2);
                anchorEnd   = matchEnd(2);
            }else if(isGroupMatched(3)){
                shrinkRegion();
                break;
            }else{
                assert false;
                throw buildParseException();
            }
            shrinkRegion();

            int day = -1;
            PeriodType periodType = null;
            lookingAtAffirm(PERIOD_PATTERN);
            if(isGroupMatched(1)){
                periodType = PeriodType.PROLOGUE;
            }else if(isGroupMatched(2)){
                periodType = PeriodType.EPILOGUE;
            }else if(isGroupMatched(3)){
                periodType = null;
            }else if(isGroupMatched(4)){
                periodType = PeriodType.PROGRESS;
                day = parseGroupedInt(4);
            }else{
                assert false;
                throw buildParseException();
            }
            shrinkRegion();

            lookingAtAffirm(closePattern);
            shrinkRegion();

            this.handler.periodLink(getContent(),
                                    anchorStart, anchorEnd,
                                    periodType, day );
        }

        return;
    }

    private static final Pattern O_MESSAGE_PATTERN =
            compile("<div\u0020class=\"message\">");
    private static final Pattern O_MSGKIND_PATTERN =
            compile(
             "(?:"
                +"<div\u0020class=\"(?:(announce)|(order)|(extra))\">"
            +")|(?:"
                +"<a\u0020name=\"([^\"]*)\">"
            +")"
            );
    private static final Pattern C_DIV_PATTERN = compile("</div>");

    /**
     * 各種メッセージをパース
     * @throws HtmlParseException パースエラー
     */
    private void parseMessage() throws HtmlParseException{
        setErrorMessage("lost message");

        boolean skipGarbage = true;

        for(;;){
            sweepSpace();

            boolean matched;
            if(skipGarbage){
                skipGarbage = false;
                matched = findProbe(O_MESSAGE_PATTERN); // 最初の1回のみ
            }else{
                matched = lookingAtProbe(O_MESSAGE_PATTERN);
            }
            if( ! matched ){
                break;
            }
            shrinkRegion();

            sweepSpace();

            lookingAtAffirm(O_MSGKIND_PATTERN);
            if(isGroupMatched(1)){
                shrinkRegion();
                this.sysEventParser.parseAnnounce();
            }else if(isGroupMatched(2)){
                shrinkRegion();
                this.sysEventParser.parseOrder();
            }else if(isGroupMatched(3)){
                shrinkRegion();
                this.sysEventParser.parseExtra();
            }else if(isGroupMatched(4)){
                int nameStart = matchStart(4);
                int nameEnd   = matchEnd(4);
                shrinkRegion();
                this.talkParser.parseTalk(nameStart, nameEnd);
            }else{
                assert false;
                throw buildParseException();
            }

            lookingAtAffirm(C_DIV_PATTERN);
            shrinkRegion();
        }

        return;
    }

    private static final Pattern O_LISTTABLE_PATTERN =
            compile("<table\u0020class=\"list\">");
    private static final Pattern ACTIVEVILLAGE =
            compile(
             "("
                +"</table>"
            +")|(?:"
                +"<tr><td>"
                +"<a\u0020href=\"([^\"]*)\">([^<]*)</a>"
                +"\u0020<strong>\uff08"
                    +"(?:(?:(午前)|(午後))\u0020)?"  // AMPM
                    +"([0-9]+)"                       // 時
                    +"(?:時\u0020|\\:)"
                    +"([0-9]+)"                       // 分
                    +"分?\u0020更新"
                +"\uff09</strong>"
                +"</td><td>(?:"
                +"[^<]*"
                    + "(参加者募集中です。)"
                    +"|(開始待ちです。)"
                    +"|(進行中です。)"
                    +"|(勝敗が決定しました。)"
                    +"|(終了・ログ公開中。)"
                +")</td></tr>"
            +")"
            );

    /**
     * トップページの村一覧表のパース
     * @throws HtmlParseException パースエラー
     */
    private void parseTopList() throws HtmlParseException{
        setErrorMessage("lost village list");

        if( ! findProbe(O_LISTTABLE_PATTERN) ) return;
        shrinkRegion();

        for(;;){
            lookingAtAffirm(ACTIVEVILLAGE);
            if(isGroupMatched(1)) break;
            int urlStart = matchStart(2);
            int urlEnd   = matchEnd(2);
            int vnameStart = matchStart(3);
            int vnameEnd   = matchEnd(3);
            int hh = parseGroupedInt(6);
            if(isGroupMatched(5)){
                hh = (hh + 12) % 24;
            }
            int mm = parseGroupedInt(7);

            VillageState state;
            if(isGroupMatched(8)){
                state = VillageState.PROLOGUE;
            }else if(isGroupMatched(9)){
                state = VillageState.PROLOGUE;
            }else if(isGroupMatched(10)){
                state = VillageState.PROGRESS;
            }else if(isGroupMatched(11)){
                state = VillageState.EPILOGUE;
            }else if(isGroupMatched(12)){
                state = VillageState.GAMEOVER;
            }else{
                assert false;
                throw buildParseException();
            }

            shrinkRegion();

            sweepSpace();

            this.handler.villageRecord(getContent(),
                                       urlStart, urlEnd,
                                       vnameStart, vnameEnd,
                                       hh, mm,
                                       state );
        }

        return;
    }

    private static final Pattern O_LISTLOG_PATTERN =
            compile(
            "<a\u0020href=\"(index[^\"]*ready_0)\">([^<]*)</a><br\u0020/>"
            );

    /**
     * 村一覧ページのパース
     * @throws HtmlParseException パースエラー
     */
    private void parseLogList() throws HtmlParseException{
        setErrorMessage("lost village list");

        boolean is1st = true;
        for(;;){
            boolean matched;
            if(is1st){
                matched = findProbe(O_LISTLOG_PATTERN);
                is1st = false;
            }else{
                matched = lookingAtProbe(O_LISTLOG_PATTERN);
            }
            if( ! matched ) break;

            int urlStart = matchStart(1);
            int urlEnd   = matchEnd(1);
            int vnameStart = matchStart(2);
            int vnameEnd   = matchEnd(2);

            shrinkRegion();

            this.handler.villageRecord(getContent(),
                                       urlStart, urlEnd,
                                       vnameStart, vnameEnd,
                                       -1, -1,
                                       VillageState.GAMEOVER );
        }

        return;
    }

    private static final Pattern C_BODY_PATTERN =
            compile("</body>");
    private static final Pattern C_HTML_PATTERN =
            compile(SP_I+ "</html>" +SP_I);

    /**
     * XHTML末尾のパース
     * @throws HtmlParseException パースエラー
     */
    private void parseTail() throws HtmlParseException{
        setErrorMessage("lost last part");

        findAffirm(C_BODY_PATTERN);
        shrinkRegion();

        matchesAffirm(C_HTML_PATTERN);
        shrinkRegion();

        return;
    }

    /**
     * 人狼BBSのトップページをパースする。
     * 現在アクティブに進行中の村一覧情報が通知される。
     * 古国の場合は過去村一覧情報が通知される。
     * @param content パース対象の文字列
     * @throws HtmlParseException パースエラー
     */
    public void parseTopPage(DecodedContent content)
            throws HtmlParseException{
        setContent(content);

        this.handler.startParse(getContent());

        parseHead();
        parseLoginForm();
        parseTopList();
        parseTail();

        this.handler.endParse();

        reset();

        return;
    }

    /**
     * 人狼BBSの村一覧ページをパースする。
     * 過去村一覧情報が通知される。
     * ※ 古国に村一覧ページはない。
     * @param content パース対象の文字列
     * @throws HtmlParseException パースエラー
     */
    public void parseLogPage(DecodedContent content)
            throws HtmlParseException{
        setContent(content);

        this.handler.startParse(getContent());

        parseHead();
        parseLogList();
        parseTail();

        this.handler.endParse();

        reset();

        return;
    }

    /**
     * 人狼BBSの個別の村の個別の日ページをパースする。
     * @param content パース対象の文字列
     * @throws HtmlParseException パースエラー
     */
    public void parsePeriodPage(DecodedContent content)
            throws HtmlParseException{
        setContent(content);

        this.handler.startParse(getContent());

        parseHead();
        parseLoginForm();
        parseVillageInfo();
        parsePeriodLink();
        parseMessage();
        parseTail();

        this.handler.endParse();

        reset();

        return;
    }

}
