/*
 * stream decoder
 *
 * Copyright(c) 2009 olyutorskii
 * $Id: StreamDecoder.java 578 2009-08-03 17:31:06Z olyutorskii $
 */

package jp.sourceforge.jindolf.parser;

import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.channels.Channels;
import java.nio.channels.ReadableByteChannel;
import java.nio.charset.CharsetDecoder;
import java.nio.charset.CoderResult;
import java.nio.charset.CodingErrorAction;

/**
 * バイトストリームからの文字デコーダ。
 * 入力バイトストリームをデコードし、デコード結果およびデコードエラーを
 * 文字デコードハンドラ{@link DecodeHandler}に通知する。
 * このクラスは、
 * デコードエラー詳細を察知できない{@link java.io.InputStreamReader}の
 * 代替品として設計された。
 * マルチスレッド対応はしていない。
 */
public class StreamDecoder{

    public static final int BYTEBUF_DEFSZ = 4 * 1024;
    public static final int CHARBUF_DEFSZ = 4 * 1024;

    protected final CharsetDecoder decoder;

    protected ReadableByteChannel channel;
    protected final ByteBuffer byteBuffer;
    protected final CharBuffer charBuffer;

    protected boolean isEndOfInput = false;
    protected boolean isFlushing = false;

    protected DecodeHandler handler;
    protected byte[] errorData = new byte[16];

    /**
     * コンストラクタ
     * @param decoder デコーダ
     */
    public StreamDecoder(CharsetDecoder decoder){
        this(decoder, BYTEBUF_DEFSZ, CHARBUF_DEFSZ);
        return;
    }

    /**
     * コンストラクタ
     * @param decoder デコーダ
     * @param inbuf_sz 入力バッファサイズ
     * @param outbuf_sz 出力バッファサイズ
     */
    public StreamDecoder(CharsetDecoder decoder,
                           int inbuf_sz,
                           int outbuf_sz ){
        super();

        this.decoder = decoder;
        this.byteBuffer = ByteBuffer.allocate(inbuf_sz);
        this.charBuffer = CharBuffer.allocate(outbuf_sz);
        this.channel = null;

        initDecoder();

        return;
    }

    /**
     * デコーダの初期化。
     */
    protected void initDecoder(){
        this.byteBuffer.clear().flip();
        this.charBuffer.clear();

        this.decoder.onMalformedInput     (CodingErrorAction.REPORT);
        this.decoder.onUnmappableCharacter(CodingErrorAction.REPORT);
        this.decoder.reset();

        this.isEndOfInput = false;
        this.isFlushing = false;

        return;
    }

    /**
     * デコードハンドラの設定。
     * @param handler デコードハンドラ
     */
    public void setDecodeHandler(DecodeHandler handler){
        this.handler = handler;
        return;
    }

    /**
     * デコードエラー格納配列の再アサイン。
     * 決して縮小することは無い。
     * @param size 再アサイン量。バイト長。
     */
    protected void reassignErrorData(int size){
        int oldLength = this.errorData.length;
        if(oldLength >= size) return;
        if(oldLength * 2 > size) size = oldLength * 2;
        byte[] newData = new byte[size];
        System.arraycopy(this.errorData, 0, newData, 0, oldLength);
        this.errorData = newData;
        return;
    }

    /**
     * デコードハンドラに文字列を渡す。
     * @throws DecodeException デコードエラー
     */
    protected void putContent() throws DecodeException{
        if(this.charBuffer.position() <= 0){
            return;
        }

        this.charBuffer.flip();
        this.handler.charContent(this.charBuffer);
        this.charBuffer.clear();

        return;
    }

    /**
     * デコードハンドラにデコードエラーを渡す。
     * @param result デコード結果
     * @throws DecodeException デコードエラー
     * @throws java.io.IOException 入力エラー
     */
    protected void putDecodeError(CoderResult result)
            throws IOException,
                   DecodeException{
        int length = chopErrorSequence(result);
        this.handler.decodingError(this.errorData, 0, length);
        return;
    }

    /**
     * デコードエラーの原因バイト列を抽出する。
     * @param result デコード結果
     * @return 原因バイト列の長さ
     * @throws java.io.IOException 入力エラー
     */
    protected int chopErrorSequence(CoderResult result) throws IOException{
        int errorLength = result.length();
        reassignErrorData(errorLength);
        this.byteBuffer.get(this.errorData, 0, errorLength);
        return errorLength;
    }

    /**
     * 入力を読み進める。
     * @return 入力バイト数。
     * @throws java.io.IOException 入出力エラー
     */
    protected int readByteBuffer() throws IOException{
        this.byteBuffer.compact();

        int length = this.channel.read(this.byteBuffer);
        if(length <= 0){
            this.isEndOfInput = true;
        }

        this.byteBuffer.flip();

        return length;
    }

    /**
     * バイトストリームのデコードを開始する。
     * @param is 入力ストリーム
     * @throws java.io.IOException 入出力エラー
     * @throws DecodeException デコードエラー
     */
    public void decode(InputStream is)
            throws IOException,
                   DecodeException {
        this.channel = Channels.newChannel(is);

        try{
            decodeChannel();
        }finally{
            this.channel.close();
            this.channel = null;
        }

        return;
    }

    /**
     * 内部チャネルのデコードを開始する。
     * @throws java.io.IOException 入出力エラー
     * @throws DecodeException デコードエラー
     */
    protected void decodeChannel()
            throws IOException,
                   DecodeException {
        initDecoder();

        this.handler.startDecoding(this.decoder);

        for(;;){
            CoderResult result;
            if( ! this.isFlushing ){
                result = this.decoder.decode(this.byteBuffer,
                                             this.charBuffer,
                                             this.isEndOfInput);
            }else{
                result = this.decoder.flush(this.charBuffer);
            }

            if(result.isError()){
                putContent();
                putDecodeError(result);
                continue;
            }

            if(result.isOverflow()){
                putContent();
                continue;
            }

            if(result.isUnderflow()){
                if(this.isEndOfInput){
                    if( ! this.isFlushing ){
                        this.isFlushing = true;
                        continue;
                    }
                    putContent();
                    break;
                }
                readByteBuffer();
                continue;
            }
        }

        this.handler.endDecoding();

        return;
    }

}
