/*
 * System event handler
 * 
 * Copyright(c) 2009 olyutorskii
 * $Id: SysEventHandler.java 651 2009-08-09 06:19:25Z olyutorskii $
 */

package jp.sourceforge.jindolf.parser;

import jp.sourceforge.jindolf.corelib.EventFamily;
import jp.sourceforge.jindolf.corelib.GameRole;
import jp.sourceforge.jindolf.corelib.SysEventType;
import jp.sourceforge.jindolf.corelib.Team;

/**
 * システムイベントのパース通知用のハンドラ。
 *
 * このハンドラの全メソッドはパーサ{@link SysEventParser}から呼び出される。
 *
 * パーサがシステムイベントを発見すると、まず最初に
 * {@link #startSysEvent(EventFamily)}がファミリ種別と共に呼び出される。
 * 次にシステムイベントのイベント種別が判明すると、
 * {@link #sysEventType(SysEventType)}が呼び出される。
 * イベント種別に従い、このハンドラの様々なメソッドが0回以上呼び出される。
 * 最後に{@link #endSysEvent()}が呼び出される。
 * その後パーサは次のシステムイベントを探し始める。
 *
 * 一部のメソッドに渡される{@link DecodedContent}オブジェクトはmutableである。
 * 後々で内容が必要になるならば、ハンドラはこの内容の必要な箇所をコピーして
 * 保存しなければならない。
 *
 * フラグメントや属性値中の文字参照記号列の解釈はハンドラ側の責務とする。
 *
 * 各メソッドは{@link HtmlParseException}をスローすることにより、
 * パース作業を中断させることができる。
 */
public interface SysEventHandler{

    /**
     * システムイベントのパース開始の通知を受け取る。
     * @param eventFamily イベントファミリ種別
     * @throws HtmlParseException パースエラー
     */
    public void startSysEvent(EventFamily eventFamily)
        throws HtmlParseException;

    /**
     * システムイベント種別の通知を受け取る。
     * @param type イベント種別
     * @throws HtmlParseException パースエラー
     */
    public void sysEventType(SysEventType type)
        throws HtmlParseException;

    /**
     * システムイベントのパース処理終了の通知を受け取る。
     * @throws HtmlParseException パースエラー
     */
    public void endSysEvent()
        throws HtmlParseException;

    /**
     * ONSTAGEイベントの詳細の通知を受け取る。
     * @param content パース対象の文字列
     * @param entryNo エントリ番号
     * @param avatarStart Avatar名開始位置
     * @param avatarEnd Avatar名終了位置
     * @throws HtmlParseException パースエラー
     * @see jp.sourceforge.jindolf.corelib.SysEventType#ONSTAGE
     */
    public void sysEventOnStage(DecodedContent content,
                                  int entryNo,
                                  int avatarStart, int avatarEnd )
        throws HtmlParseException;

    /**
     * OPENROLEイベントの詳細の通知を受け取る。
     * 複数回連続して呼ばれる。
     * @param role 役職
     * @param num 役職の人数
     * @throws HtmlParseException パースエラー
     * @see jp.sourceforge.jindolf.corelib.SysEventType#OPENROLE
     */
    public void sysEventOpenRole(GameRole role, int num)
        throws HtmlParseException;

    /**
     * SURVIVORイベントの詳細の通知を受け取る。
     * 複数回連続して呼ばれる。
     * @param content パース対象の文字列
     * @param avatarStart Avatar名開始位置
     * @param avatarEnd Avatar名終了位置
     * @throws HtmlParseException パースエラー
     * @see jp.sourceforge.jindolf.corelib.SysEventType#SURVIVOR
     */
    public void sysEventSurvivor(DecodedContent content,
                                   int avatarStart, int avatarEnd)
        throws HtmlParseException;

    /**
     * COUNTINGイベントの詳細の通知を受け取る。
     * 複数回連続して呼ばれる。
     * 最後の呼び出しで投票元Avatar名の位置情報が負だった場合、
     * 投票先Avatar名は処刑が実行されたAvatarを表す。
     * @param content パース対象の文字列。
     * @param voteByStart 投票元Avatar名開始位置
     * @param voteByEnd 投票元Avatar名終了位置
     * @param voteToStart 投票先Avatar名開始位置
     * @param voteToEnd 投票先Avatar名終了位置
     * @throws HtmlParseException パースエラー
     * @see jp.sourceforge.jindolf.corelib.SysEventType#COUNTING
     */
    public void sysEventCounting(DecodedContent content,
                                 int voteByStart, int voteByEnd,
                                 int voteToStart, int voteToEnd )
        throws HtmlParseException;

    /**
     * SUDDENDEATHイベントの詳細の通知を受け取る。
     * @param content パース対象の文字列
     * @param avatarStart Avatar名開始位置
     * @param avatarEnd Avatar名終了位置
     * @throws HtmlParseException パースエラー
     * @see jp.sourceforge.jindolf.corelib.SysEventType#SUDDENDEATH
     */
    public void sysEventSuddenDeath(DecodedContent content,
                                      int avatarStart, int avatarEnd )
        throws HtmlParseException;

    /**
     * MURDEREDイベントの詳細の通知を受け取る。
     * ハム溶けの時など、連続して複数回呼ばれる事がある。
     * @param content パース対象の文字列
     * @param avatarStart Avatar名開始位置
     * @param avatarEnd Avatar名終了位置
     * @throws HtmlParseException パースエラー
     * @see jp.sourceforge.jindolf.corelib.SysEventType#MURDERED
     */
    public void sysEventMurdered(DecodedContent content,
                                   int avatarStart, int avatarEnd)
        throws HtmlParseException;

    /**
     * PLAYERLISTイベントの詳細の通知を受け取る。
     * 複数回連続して呼ばれる。
     * @param content パース対象の文字列
     * @param avatarStart Avatar名開始位置
     * @param avatarEnd Avatar名終了位置
     * @param urlStart URL開始位置
     * @param urlEnd URL終了位置
     * @param idStart ID開始位置
     * @param idEnd ID終了位置
     * @param isLiving 生存していればtrue
     * @param role 役職
     * @throws HtmlParseException パースエラー
     * @see jp.sourceforge.jindolf.corelib.SysEventType#PLAYERLIST
     */
    public void sysEventPlayerList(DecodedContent content,
                                     int avatarStart, int avatarEnd,
                                     int urlStart, int urlEnd,
                                     int idStart, int idEnd,
                                     boolean isLiving,
                                     GameRole role )
        throws HtmlParseException;

    /**
     * JUDGEイベントの詳細の通知を受け取る。
     * @param content パース対象の文字列。
     * @param judgeByStart 占師Avatar名開始位置
     * @param judgeByEnd 占師Avatar名終了位置
     * @param judgeToStart 占われたAvatar名開始位置
     * @param judgeToEnd 占われたAvatar名終了位置
     * @throws HtmlParseException パースエラー
     * @see jp.sourceforge.jindolf.corelib.SysEventType#JUDGE
     */
    public void sysEventJudge(DecodedContent content,
                               int judgeByStart, int judgeByEnd,
                               int judgeToStart, int judgeToEnd )
        throws HtmlParseException;

    /**
     * GUARDイベントの詳細の通知を受け取る。
     * @param content パース対象の文字列。
     * @param guardByStart 狩人Avatar名開始位置
     * @param guardByEnd 狩人Avatar名終了位置
     * @param guardToStart 護られたAvatar名開始位置
     * @param guardToEnd 護られたAvatar名終了位置
     * @throws HtmlParseException パースエラー
     * @see jp.sourceforge.jindolf.corelib.SysEventType#GUARD
     */
    public void sysEventGuard(DecodedContent content,
                               int guardByStart, int guardByEnd,
                               int guardToStart, int guardToEnd )
        throws HtmlParseException;

    /**
     * ASKENTRYイベントの詳細の通知を受け取る。
     * @param hh 時間
     * @param mm 分
     * @param minLimit 最小構成人数
     * @param maxLimit 最大定員
     * @throws HtmlParseException パースエラー
     * @see jp.sourceforge.jindolf.corelib.SysEventType#ASKENTRY
     */
    public void sysEventAskEntry(int hh, int mm,
                                   int minLimit, int maxLimit)
        throws HtmlParseException;

    /**
     * ASKCOMMITイベントの詳細の通知を受け取る。
     * @param hh 時間(24時間制)
     * @param mm 分
     * @throws HtmlParseException パースエラー
     * @see jp.sourceforge.jindolf.corelib.SysEventType#ASKCOMMIT
     */
    public void sysEventAskCommit(int hh, int mm)
        throws HtmlParseException;

    /**
     * NOCOMMENTイベントの詳細の通知を受け取る。
     * 複数回連続して呼ばれる可能性がある。
     * @param content パース対象文字列
     * @param avatarStart Avatar名開始位置
     * @param avatarEnd Avatar名終了位置
     * @throws HtmlParseException パースエラー
     * @see jp.sourceforge.jindolf.corelib.SysEventType#NOCOMMENT
     */
    public void sysEventNoComment(DecodedContent content,
                                    int avatarStart, int avatarEnd)
        throws HtmlParseException;

    /**
     * STAYEPILOGUEイベントの詳細の通知を受け取る。
     * @param winner 勝利陣営
     * @param hh 時間(24時間制)
     * @param mm 分
     * @throws HtmlParseException パースエラー
     * @see jp.sourceforge.jindolf.corelib.SysEventType#STAYEPILOGUE
     */
    public void sysEventStayEpilogue(Team winner, int hh, int mm)
        throws HtmlParseException;

    /**
     * イベントの内容(DIV要素)の一般文字列出現の通知を受け取る。
     * イベント種別は問わない。
     * @param content パース対象文字列
     * @param contentStart 内容開始位置
     * @param contentEnd 内容終了位置
     * @throws HtmlParseException パースエラー
     */
    public void sysEventContent(DecodedContent content,
                                  int contentStart, int contentEnd)
        throws HtmlParseException;

    /**
     * イベントの内容(DIV要素)のBRタグ出現の通知を受け取る。
     * イベント種別は問わない。
     * @throws HtmlParseException パースエラー
     */
    public void sysEventContentBreak()
        throws HtmlParseException;

    /**
     * イベントの内容(DIV要素)のAタグ出現の通知を受け取る。
     * イベント種別は問わない。
     * href属性によるURL記述も通知される。
     * @param content パース対象文字列
     * @param urlStart URL開始位置
     * @param urlEnd URL終了位置
     * @param contentStart 内容開始位置
     * @param contentEnd 内容終了位置
     * @throws HtmlParseException パースエラー
     */
    public void sysEventContentAnchor(DecodedContent content,
                                         int urlStart, int urlEnd,
                                         int contentStart, int contentEnd)
        throws HtmlParseException;

}
