/*
 * handler for parse talk-part
 *
 * Copyright(c) 2009 olyutorskii
 * $Id: TalkHandler.java 651 2009-08-09 06:19:25Z olyutorskii $
 */

package jp.sourceforge.jindolf.parser;

import jp.sourceforge.jindolf.corelib.TalkType;

/**
 * 人狼BBSの発言部XHTML断片をパースするためのハンドラ。
 *
 * このハンドラの全メソッドはパーサ{@link HtmlParser}から呼び出される。
 *
 * パーサが発言箇所を発見すると、まず最初に
 * {@link #startTalk()}が呼び出される。
 * 発言内容に従い、このハンドラの様々なメソッドが0回以上呼び出される。
 * 最後に{@link #endTalk()}が呼び出される。
 * その後パーサは次の発言を探し始める。
 *
 * 一部のメソッドに渡される{@link DecodedContent}オブジェクトはmutableである。
 * 後々で内容が必要になるならば、ハンドラはこの内容の必要な箇所をコピーして
 * 保存しなければならない。
 *
 * フラグメントや属性値中の文字参照記号列の解釈はハンドラ側の責務とする。
 *
 * 各メソッドは{@link HtmlParseException}をスローすることにより、
 * パース作業を中断させることができる。
 */
public interface TalkHandler{

    /**
     * 発言部パース開始の通知を受け取る。
     * @throws HtmlParseException パースエラー
     */
    public void startTalk()
            throws HtmlParseException;

    /**
     * 発言部パース終了の通知を受け取る。
     * @throws HtmlParseException パースエラー
     */
    public void endTalk()
            throws HtmlParseException;

    /**
     * 発言部ID(Aタグのname属性)の通知を受け取る。
     * @param content パース対象文字列
     * @param start ID開始位置
     * @param end ID終了位置
     * @throws HtmlParseException パースエラー
     */
    public void talkId(DecodedContent content, int start, int end)
            throws HtmlParseException;

    /**
     * 発言したAvatar名の通知を受け取る。
     * @param content パース対象文字列
     * @param start Avatar名開始位置
     * @param end Avatar名終了位置
     * @throws HtmlParseException パースエラー
     */
    public void talkAvatar(DecodedContent content, int start, int end)
            throws HtmlParseException;

    /**
     * 発言時刻の通知を受け取る。
     * @param hh 時間(24時間制)
     * @param mm 分
     * @throws HtmlParseException パースエラー
     */
    public void talkTime(int hh, int mm)
            throws HtmlParseException;

    /**
     * 発言者の顔アイコンURLの通知を受け取る。
     * @param content パース対象文字列
     * @param start URL開始位置
     * @param end URL終了位置
     * @throws HtmlParseException パースエラー
     */
    public void talkIconUrl(DecodedContent content, int start, int end)
            throws HtmlParseException;

    /**
     * 発言種別の通知を受け取る。
     * @param type 発言種別
     * @throws HtmlParseException パースエラー
     */
    public void talkType(TalkType type)
            throws HtmlParseException;

    /**
     * 発言テキスト内容の通知を受け取る。
     * 1発言のパース中に複数回呼ばれる事もありうる。
     * @param content パース対象文字列
     * @param start テキスト開始位置
     * @param end テキスト終了位置
     * @throws HtmlParseException パースエラー
     */
    public void talkText(DecodedContent content, int start, int end)
            throws HtmlParseException;

    /**
     * 発言テキスト内のBRタグ出現の通知を受け取る。
     * 1発言のパース中に複数回呼ばれる事もありうる。
     * @throws HtmlParseException パースエラー
     */
    public void talkBreak()
            throws HtmlParseException;

    // TODO 「発言取り消し」リンクの検出メソッドは必要？

}
