/*
 * talk-part parser
 *
 * Copyright(c) 2009 olyutorskii
 * $Id: TalkParser.java 651 2009-08-09 06:19:25Z olyutorskii $
 */

package jp.sourceforge.jindolf.parser;

import java.util.regex.Pattern;
import jp.sourceforge.jindolf.corelib.TalkType;

/**
 * 人狼BBSシステムが出力する各発言箇所のパーサ。
 * パース進行に従い{@link TalkHandler}の各種メソッドが呼び出される。
 */
public class TalkParser extends AbstractParser{

    private TalkHandler handler;

    /**
     * コンストラクタ
     * @param parent 親パーサ
     */
    public TalkParser(AbstractParser parent){
        super(parent);
        return;
    }

    /**
     * {@link TalkHandler}ハンドラを登録する。
     * @param handler ハンドラ
     */
    public void setTalkHandler(TalkHandler handler){
        this.handler = handler;
        return;
    }

    /**
     * 各Avatarの個別の発言をパースする。
     * 最初のAタグは既にパース済みとする。
     * @param idStart Aタグのname属性値開始位置
     * @param idEnd Aタグのname属性終了位置
     * @throws HtmlParseException パースエラー
     */
    public void parseTalk(int idStart, int idEnd) throws HtmlParseException{
        this.handler.startTalk();

        this.handler.talkId(getContent(), idStart, idEnd);

        parseName();
        parseTime();
        parseIcon();
        parseType();
        parseText();
        parseTail();

        this.handler.endTalk();

        return;
    }

    private static final Pattern AVATARNAME_PATTERN =
            compile("([^<]*)");

    /**
     * 発言者名をパースする。
     * @throws HtmlParseException パースエラー
     */
    private void parseName() throws HtmlParseException{
        setErrorMessage("lost dialog avatar-name");

        lookingAtAffirm(AVATARNAME_PATTERN);
        int avatarStart = matchStart(1);
        int avatarEnd   = matchEnd(1);
        shrinkRegion();

        this.handler.talkAvatar(getContent(), avatarStart, avatarEnd);

        return;
    }

    private static final Pattern TALKTIME_PATTERN =
            compile(
                 "</a>"
                +SP_I
                +"<span class=\"time\">"
                +"(?:(午前\u0020)|(午後\u0020))?"
                +"([0-9][0-9]?)(?:時\u0020|\\:)"
                +"([0-9][0-9]?)分?\u0020"
                +"</span>"
                +SP_I
                +"<table\u0020[^>]*>"
                +SP_I
                +"(?:<tbody>)?"
                +SP_I
                +"<tr>"
            );

    /**
     * 発言時刻をパースする。
     * @throws HtmlParseException パースエラー
     */
    private void parseTime() throws HtmlParseException{
        setErrorMessage("lost dialog time");

        lookingAtAffirm(TALKTIME_PATTERN);
        int hh = parseGroupedInt(3);
        int mm = parseGroupedInt(4);
        if(isGroupMatched(2)){  // 午後指定
            hh = (hh + 12) % 24;
        }
        shrinkRegion();
        sweepSpace();

        this.handler.talkTime(hh, mm);

        return;
    }

    private static final Pattern IMGSRC_PATTERN =
            compile(
                  "<td\u0020[^>]*><img\u0020src=\"([^\"]*)\"></td>"
                 +SP_I
                 +"<td\u0020[^>]*><img\u0020[^>]*></td>"
            );

    /**
     * アイコンのURLをパースする。
     * @throws HtmlParseException パースエラー
     */
    private void parseIcon() throws HtmlParseException{
        setErrorMessage("lost icon url");

        lookingAtAffirm(IMGSRC_PATTERN);
        int imageStart = matchStart(1);
        int imageEnd   = matchEnd(1);
        shrinkRegion();
        sweepSpace();

        this.handler.talkIconUrl(getContent(), imageStart, imageEnd );

        return;
    }

    private static final Pattern TALKDIC_PATTERN =
            compile(
                 "<td>" +SP_I+ "<div(?:\u0020[^>]*)?>"
                +SP_I
                +"<div class=\"mes_"
                +"(?:(say)|(think)|(whisper)|(groan))"
                +"_body1\">"
            );

    /**
     * 発言種別をパースする。
     * @throws HtmlParseException パースエラー
     */
    private void parseType() throws HtmlParseException{
        setErrorMessage("lost dialog type");

        lookingAtAffirm(TALKDIC_PATTERN);
        TalkType type;
        if(isGroupMatched(1)){
            type = TalkType.PUBLIC;
        }else if(isGroupMatched(2)){
            type = TalkType.PRIVATE;
        }else if(isGroupMatched(3)){
            type = TalkType.WOLFONLY;
        }else if(isGroupMatched(4)){
            type = TalkType.GRAVE;
        }else{
            assert false;
            throw buildParseException();
        }
        shrinkRegion();

        this.handler.talkType(type);

        return;
    }

    private static final Pattern TEXT_PATTERN =
            compile("([^<>]+)|(<br />)");

    /**
     * 発言テキストをパースする。
     * 前後のホワイトスペースは無視しない。
     * @throws HtmlParseException パースエラー
     */
    private void parseText() throws HtmlParseException{
        setErrorMessage("lost dialog text");

        while(lookingAtProbe(TEXT_PATTERN)){
            if(isGroupMatched(1)){
                int textStart = matchStart(1);
                int textEnd   = matchEnd(1);
                this.handler.talkText(getContent(), textStart, textEnd );
            }else if(isGroupMatched(2)){
                this.handler.talkBreak();
            }else{
                assert false;
                throw buildParseException();
            }
            shrinkRegion();
        }

        return;
    }

    private static final Pattern TAIL_PATTERN =
            compile(
                       "</div>"
                +SP_I+ "</div>"
                +SP_I+ "</td>"
                +SP_I+ "</tr>"
                +SP_I+ "(?:</tbody>)?"
                +SP_I+ "</table>"
            );

    /**
     * 発言末尾をパースする。
     * @throws HtmlParseException パースエラー
     */
    private void parseTail() throws HtmlParseException{
        setErrorMessage("lost dialog termination");

        lookingAtAffirm(TAIL_PATTERN);
        shrinkRegion();
        sweepSpace();

        return;
    }

}
