/*
 * application settings
 *
 * Copyright(c) 2008 olyutorskii
 * $Id: AppSetting.java 836 2009-09-09 15:58:36Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.io.File;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * アプリケーション及び実行環境の各種設定。
 */
public final class AppSetting{

    private static final Map<String, String> propertyMap =
            new TreeMap<String, String>();
    private static final String osName;
    private static final String osVersion;
    private static final String osArch;
    private static final String javaVendor;
    private static final String javaVersion;

    private static final String classpath;
    private static final String[] classpaths;

    private static final Map<String, String> environmentMap =
            new TreeMap<String, String>();
    private static final String envLANG;
    private static final String envDISPLAY;
//  public static final String envPATH;
//  public static final String[] paths;

    private static final String proxyHost;
    private static final String proxyPort;
    private static final String nonProxyHosts;

    private static boolean needHelp = false;
    private static boolean needVersion = false;
    private static boolean needVMInfo = false;
    private static boolean boldMetal = false;
    private static boolean noSplash = false;
    private static boolean consolelog = false;

    private static int frameWidth = 800;
    private static int frameHeight = 600;
    private static int frameXpos = Integer.MIN_VALUE;
    private static int frameYpos = Integer.MIN_VALUE;

    private static String initfont = null;
    private static Boolean antialias = null;
    private static Boolean fractional = null;

    private static final List<String> invokeArgs = new LinkedList<String>();

    private static final Pattern geomPattern =
            Pattern.compile(
                 "([1-9][0-9]*)x([1-9][0-9]*)"
                +"(?:(\\+|\\-)([1-9][0-9]*)(\\+|\\-)([1-9][0-9]*))?"
                );

    static{
        osName      = getSecureProperty("os.name");
        osVersion   = getSecureProperty("os.version");
        osArch      = getSecureProperty("os.arch");
        javaVendor  = getSecureProperty("java.vendor");
        javaVersion = getSecureProperty("java.version");
        classpath   = getSecureProperty("java.class.path");

        proxyHost     = getSecureProperty("http.proxyHost");
        proxyPort     = getSecureProperty("http.proxyPort");
        nonProxyHosts = getSecureProperty("http.nonProxyHosts");

        envLANG      = getSecureEnvironment("LANG");
        envDISPLAY   = getSecureEnvironment("DISPLAY");

        if(classpath != null){
            classpaths = classpath.split(File.pathSeparator);
        }else{
            classpaths = new String[0];
        }
/*
        envPATH      = getSecureEnvironment("PATH");
        if(envPATH != null){
            paths = envPATH.split(File.pathSeparator);
        }else{
            paths = new String[0];
        }
*/
    }

    /**
     * OS名を返す。
     * @return OS名
     */
    public static String getOsName(){
        return osName;
    }

    /**
     * アーキテクチャ名を返す。
     * @return アーキテクチャ名
     */
    public static String getOsArch(){
        return osArch;
    }

    /**
     * OSバージョン名を返す。
     * @return OSバージョン名
     */
    public static String getOsVersion(){
        return osVersion;
    }

    /**
     * Java実行系のベンダ名を返す。
     * @return Java実行系のベンダ名
     */
    public static String getJavaVendor(){
        return javaVendor;
    }

    /**
     * Java実行系のバージョン名を返す。
     * @return Java実行系のバージョン名
     */
    public static String getJavaVersion(){
        return javaVersion;
    }

    /**
     * クラスパス集合を返す。
     * @return クラスパス集合
     */
    public static String getClasspath(){
        return classpath;
    }

    /**
     * トークンで分解されたクラスパスを返す。
     * @return トークンで分解されたクラスパス
     */
    public static String[] getClasspaths(){
        return classpaths;
    }

    /**
     * 環境変数LANGの値を返す。
     * @return 環境変数LANGの値
     */
    public static String getEnvLANG(){
        return envLANG;
    }

    /**
     * 環境変数DISPLAYの値を返す。
     * @return 環境変数DISPLAYの値
     */
    public static String getEnvDISPLAY(){
        return envDISPLAY;
    }

    /**
     * プロキシのホスト名を返す。
     * @return プロキシのホスト名
     */
    public static String getProxyHost(){
        return proxyHost;
    }

    /**
     * プロキシのポート番号を返す。
     * @return プロキシのポート番号
     */
    public static String getProxyPort(){
        return proxyPort;
    }

    /**
     * プロキシ対象外のホスト名を返す。
     * @return プロキシ対象外のホスト名
     */
    public static String getNonProxyHosts(){
        return nonProxyHosts;
    }

    /**
     * ヘルプ出力が必要か否か判定する。
     * @return 必要ならtrue
     */
    public static boolean needHelp(){
        return needHelp;
    }

    /**
     * バージョン表示出力が必要か否か判定する。
     * @return 必要ならtrue
     */
    public static boolean needVersion(){
        return needVersion;
    }

    /**
     * VM情報出力が必要か否か判定する。
     * @return 必要ならtrue
     */
    public static boolean needVMInfo(){
        return needVMInfo;
    }

    /**
     * boldMetal設定を使うか否か判定する。
     * @return 使うならtrue
     */
    public static boolean useBoldMetal(){
        return boldMetal;
    }

    /**
     * スプラッシュ画面を消すか否か判定する。
     * @return 消すならtrue
     */
    public static boolean noSplash(){
        return noSplash;
    }

    /**
     * ログ表示にコンソールを使うか否か判定する。
     * @return コンソールを使うならtrue
     */
    public static boolean useConsolelog(){
        return consolelog;
    }

    /**
     * 初期のフレーム幅を返す。
     * @return 初期のフレーム幅
     */
    public static int initialFrameWidth(){
        return frameWidth;
    }

    /**
     * 初期のフレーム高を返す。
     * @return 初期のフレーム高
     */
    public static int initialFrameHeight(){
        return frameHeight;
    }

    /**
     * 初期のフレーム位置のX座標を返す。
     * 特に指示されていなければInteger.MIN_VALUEを返す。
     * @return 初期のフレーム位置のX座標
     */
    public static int initialFrameXpos(){
        return frameXpos;
    }

    /**
     * 初期のフレーム位置のY座標を返す。
     * 特に指示されていなければInteger.MIN_VALUEを返す。
     * @return 初期のフレーム位置のY座標
     */
    public static int initialFrameYpos(){
        return frameYpos;
    }

    /**
     * 発言表示用フォントのファミリ名を返す。
     * 特に指示されていなければnullを返す。
     * @return 発言表示用フォントのファミリ名
     */
    public static String getInitfont(){
        return initfont;
    }

    /**
     * 文字アンチエイリアスを行うか否か判定する。
     * 特に指示されていなければnullを返す。
     * @return アンチエイリアスを行うならBoolean.TRUE
     */
    public static Boolean useAntialias(){
        return antialias;
    }

    /**
     * 文字のサブピクセル描画を行うか否か判定する。
     * 特に指示されていなければnullを返す。
     * @return サブピクセル描画を行うならBoolean.TRUE
     */
    public static Boolean useFractional(){
        return fractional;
    }

    /**
     * 可能ならシステムプロパティを読み込む。
     * @param key キー
     * @return プロパティ値。セキュリティ上読み込み禁止の場合はnull。
     */
    private static String getSecureProperty(String key){
        String result;
        try{
            result = System.getProperty(key);
            if(result != null) propertyMap.put(key, result);
        }catch(SecurityException e){
            result = null;
        }
        return result;
    }

    /**
     * 可能なら環境変数を読み込む。
     * @param name 環境変数名
     * @return 環境変数値。セキュリティ上読み込み禁止の場合はnull。
     */
    private static String getSecureEnvironment(String name){
        String result;
        try{
            result = System.getenv(name);
            if(result != null) environmentMap.put(name, result);
        }catch(SecurityException e){
            result = null;
        }
        return result;
    }

    /**
     * VM詳細情報を文字列化する。
     * @return VM詳細情報
     */
    public static String getVMInfo(){
        StringBuilder result = new StringBuilder();

        result.append("最大ヒープメモリ量: "
                + Jindolf.RUNTIME.maxMemory() + " bytes\n");

        result.append("\n");

        result.append("起動時引数:\n");
        for(String arg : invokeArgs){
            result.append("  ").append(arg).append("\n");
        }

        result.append("\n");

        result.append("主要システムプロパティ:\n");
        Set<String> propKeys = propertyMap.keySet();
        for(String propKey : propKeys){
            if(propKey.equals("java.class.path")) continue;
            String value = propertyMap.get(propKey);
            result.append("  ");
            result.append(propKey).append("=").append(value).append("\n");
        }

        result.append("\n");

        result.append("主要環境変数:\n");
        Set<String> envKeys = environmentMap.keySet();
        for(String envKey : envKeys){
//          if(envKey.equals("PATH")) continue;
            String value = environmentMap.get(envKey);
            result.append("  ");
            result.append(envKey).append("=").append(value).append("\n");
        }

        result.append("\n");
/*
        result.append("execパス:\n");
        for(String path : paths){
            result.append("  ");
            result.append(path).append("\n");
        }

        result.append("\n");
*/
        result.append("クラスパス:\n");
        for(String path : classpaths){
            result.append("  ");
            result.append(path).append("\n");
        }

        return result.toString();
    }

    /**
     * オプション文字列を解析する。
     * @param args main()に渡されるオプション文字列
     */
    public static void parseOptions(String[] args){
        invokeArgs.clear();
        for(String arg : args){
            if(arg == null) continue;
            if(arg.length() <= 0) continue;
            invokeArgs.add(arg);
        }
        ListIterator<String> iterator = invokeArgs.listIterator();

        while(iterator.hasNext()){
            String arg = iterator.next();
            if(   arg.equals("-h")
               || arg.equals("-help")
               || arg.equals("--help")
               || arg.equals("-?")  ){
                needHelp = true;
                needVersion = false;
                continue;
            }
            if( arg.equals("-version") ){
                needVersion = true;
                needHelp = false;
                continue;
            }
            if( arg.equals("-boldMetal") ){
                boldMetal = true;
                continue;
            }
            if( arg.equals("-nosplash") ){
                noSplash = true;
                continue;
            }
            if( arg.equals("-geometry") ){
                if( ! iterator.hasNext() ){
                    optionError(
                            "起動オプション["
                            +arg
                            +"]に引数がありません。");
                    return;
                }
                String geometry = iterator.next();
                Matcher matcher = geomPattern.matcher(geometry);
                if( ! matcher.matches() ){
                    optionError(
                            "起動オプション["
                            +arg
                            +"]の引数形式["
                            +geometry
                            +"]が不正です。" );
                    return;
                }
                String width = matcher.group(1);
                String height = matcher.group(2);
                String xSign = matcher.group(3);
                String xPos = matcher.group(4);
                String ySign = matcher.group(5);
                String yPos = matcher.group(6);
                try{
                    frameWidth = Integer.parseInt(width);
                    frameHeight = Integer.parseInt(height);
                    if(xPos != null && xPos.length() > 0){
                        frameXpos = Integer.parseInt(xPos);
                        if(xSign.equals("-")) frameXpos = -frameXpos;
                    }
                    if(yPos != null && yPos.length() > 0){
                        frameYpos = Integer.parseInt(yPos);
                        if(ySign.equals("-")) frameYpos = -frameYpos;
                    }
                }catch(NumberFormatException e){
                    optionError(
                            "起動オプション["
                            +arg
                            +"]の引数形式["
                            +geometry
                            +"]が不正です。" );
                    return;
                }

                continue;
            }
            if( arg.equals("-vminfo") ){
                needVMInfo = true;
                continue;
            }
            if( arg.equals("-consolelog") ){
                consolelog = true;
                continue;
            }
            if( arg.equals("-initfont") ){
                checkNextArg(arg, iterator);
                initfont = iterator.next();
                continue;
            }
            if( arg.equals("-antialias") ){
                antialias = parseBooleanSwitch(arg, iterator);
                continue;
            }
            if( arg.equals("-fractional") ){
                fractional = parseBooleanSwitch(arg, iterator);
                continue;
            }

            optionError(
                    "未定義の起動オプション["
                    +arg
                    +"]が指定されました。");
            assert false;
        }

        return;
    }

    /**
     * 真偽二値をとるオプション解析の下請け。
     * @param option オプション名
     * @param iterator 引数並び
     * @return 真偽
     */
    private static Boolean parseBooleanSwitch(
            String option,
            ListIterator<String> iterator ){
        Boolean result = null;
        checkNextArg(option, iterator);
        String onoff = iterator.next();
        if(   onoff.compareToIgnoreCase("on"  ) == 0
           || onoff.compareToIgnoreCase("yes" ) == 0
           || onoff.compareToIgnoreCase("true") == 0){
            result = Boolean.TRUE;
        }else if(   onoff.compareToIgnoreCase("off"  ) == 0
                 || onoff.compareToIgnoreCase("no"   ) == 0
                 || onoff.compareToIgnoreCase("false") == 0){
            result = Boolean.FALSE;
        }else{
            optionError(
                    "起動オプション["
                    +option
                    +"]の引数形式["
                    +onoff
                    +"]が不正です。"
                    +"on, off, yes, no, true, false"
                    +"のいずれかを指定してください。");
            assert false;
        }
        return result;
    }

    /**
     * 追加引数を持つオプションのチェック。
     * @param option オプション名
     * @param iterator 引数並び
     */
    private static void checkNextArg(CharSequence option,
                                       ListIterator<String> iterator ){
        if( ! iterator.hasNext() ){
            optionError(
                    "起動オプション["
                    +option
                    +"]に引数がありません。");
            assert false;
            throw new Error();
        }
        return;
    }

    /**
     * 起動オプションエラー共通処理。
     * @param message エラーメッセージ
     */
    private static void optionError(CharSequence message){
        System.err.println(message);
        System.err.println(
                "起動オプション一覧は、"
                +"起動オプションに「-help」を指定すると確認できます。" );
        Jindolf.RUNTIME.exit(1);
        assert false;
        throw new AssertionError();
    }

    /**
     * 隠しコンストラクタ。
     */
    private AppSetting(){
        super();
        return;
    }

}
