/*
 * Jindolf main class
 *
 * Copyright(c) 2008 olyutorskii
 * $Id: Jindolf.java 846 2009-09-10 14:31:11Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.awt.AWTEvent;
import java.awt.Dimension;
import java.awt.EventQueue;
import java.awt.GraphicsEnvironment;
import java.awt.Toolkit;
import java.awt.Window;
import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.io.Reader;
import java.net.URL;
import java.nio.charset.Charset;
import java.security.Permission;
import java.util.Date;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.ConsoleHandler;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.logging.LoggingPermission;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JWindow;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;

/**
 * Jindolf スタートアップクラス。
 *
 * コンストラクタは無いよ。
 * アプリ開始はstaticメソッド{@link #main(String[])}呼び出しから。
 */
public final class Jindolf{

    /** クラスロード時のナノカウント。 */
    public static final long NANOCT_LOADED = System.nanoTime();
    /** クラスロード時刻(エポックmsec)。 */
    public static final long EPOCHMS_LOADED  = System.currentTimeMillis();
    private static      long nanoct_entry;
    private static      long epochms_entry;

    /** バージョン。 */
    public static final String VERSION = "2.20.2";
    /** タイトル。 */
    public static final String TITLE;
    /** 作者名。 */
    public static final String AUTHOR;
    /** 著作権表記。 */
    public static final String COPYRIGHT;
    /** ライセンス表記。 */
    public static final String LICENSE = "The MIT License";
    /** 連絡先。 */
    public static final String CONTACT = "http://jindolf.sourceforge.jp/";
    /** クレジット。 */
    public static final String ID;

    /** このClass。 */
    public static final Class<?>        SELF_KLASS;
    /** このPackage。 */
    public static final Package         SELF_PACKAGE;
    /** ランタイムPackage。 */
    public static final Package         JRE_PACKAGE;
    /** 実行環境。 */
    public static final Runtime         RUNTIME;
    /** セキュリティマネージャ。 */
    public static final SecurityManager SEC_MANAGER;
    /** 共通ロガー。 */
    public static final Logger          logger;
    /** クラスローダ。 */
    public static final ClassLoader     LOADER;

    private static final Charset CHARSET_UTF8 = Charset.forName("UTF-8");

    private static final AtomicBoolean INVOKE_FLAG = new AtomicBoolean(false);

    static{
        SELF_KLASS   = Jindolf.class;
        SELF_PACKAGE = SELF_KLASS.getPackage();
        JRE_PACKAGE  = java.lang.Object.class.getPackage();
        RUNTIME      = Runtime.getRuntime();
        SEC_MANAGER  = System.getSecurityManager();
        logger       = Logger.getLogger(SELF_PACKAGE.getName());

        ClassLoader thisLoader;
        try{
            thisLoader = SELF_KLASS.getClassLoader();
        }catch(SecurityException e){
            thisLoader = null;
        }
        LOADER = thisLoader;

        String implTitle = SELF_PACKAGE.getImplementationTitle();
        if(implTitle != null){
            TITLE = implTitle;
        }else{
            TITLE = "Jindolf";
        }

        String implVendor = SELF_PACKAGE.getImplementationVendor();
        if(implVendor != null){
            AUTHOR = implVendor;
        }else{
            AUTHOR = "olyutorskii";
        }

        COPYRIGHT = "Copyright(c) 2008 " + AUTHOR;
        ID = TITLE + " Ver." + VERSION +" "+ COPYRIGHT +" ("+ LICENSE +")";

        new Jindolf();
    }

    /**
     * Jindolf のスタートアップエントリ。
     *
     * @param args コマンドライン引数
     */
    public static void main(final String[] args){
        // 二重起動チェック
        boolean hasInvoked = ! INVOKE_FLAG.compareAndSet(false, true);
        if(hasInvoked){
            System.err.println("二度目以降の起動がキャンセルされました。");
            return;
        }

        nanoct_entry = System.nanoTime();
        epochms_entry      = System.currentTimeMillis();

        checkCompileError();
        checkEnvironment();
        checkPackageDefinition();

        AppSetting.parseOptions(args);

        if(AppSetting.needHelp()){
            showHelpMessage();
            RUNTIME.exit(0);
        }

        if(AppSetting.needVersion()){
            System.out.println(ID);
            RUNTIME.exit(0);
        }

        if(AppSetting.needVMInfo()){
            System.out.println(AppSetting.getVMInfo());
        }

        if( hasRuntime6() && AppSetting.noSplash() ){
            System.out.println(
                      "JRE1.6以降では、"
                    + "Java実行系の方でスプラッシュ画面の非表示を"
                    + "指示してください(おそらく空の-splash:オプション)" );
        }

        if(AppSetting.useBoldMetal()){
            // もの凄く日本語が汚くなるかもよ！注意
            UIManager.put("swing.boldMetal", Boolean.TRUE);
        }else{
            UIManager.put("swing.boldMetal", Boolean.FALSE);
        }

        initLogging();        // ここからロギング解禁
                              // Jindolf.exit()もここから解禁

        logger.info(ID + " は "
                    + new Date(epochms_entry)
                    + " に実行を開始しました。 " );
        logger.info("Initial Nano-Count:"+nanoct_entry);
        logger.info(
                "Max-heap:"
                + RUNTIME.maxMemory()   / 1024 / 1024 + "MByte"
                + " Total-heap:"
                + RUNTIME.totalMemory() / 1024 / 1024 + "MByte");
        logger.info("\n" + AppSetting.getVMInfo());

        if(LOADER == null){
            logger.log(Level.WARNING,
                        "セキュリティ設定により、"
                       +"クラスローダを取得できませんでした");
        }

        RUNTIME.addShutdownHook(new Thread(){
            @Override
            public void run(){
                logger.info("シャットダウン処理に入ります…");
                System.out.flush();
                System.err.flush();
                RUNTIME.runFinalization();
                Thread.yield();
                return;
            }
        });

        replaceEventQueue();

        Window splashWindow = null;
        if( ! hasRuntime6() && ! AppSetting.noSplash() ){
            splashWindow = createSplashWindow();
            splashWindow.setVisible(true);
            GUIUtils.dispatchEmptyAWTEvent();
        }

        preInitClass();

        boolean hasError = false;
        try{
            SwingUtilities.invokeAndWait(new Runnable(){
                public void run(){
                    startGUI();
                    return;
                }
            });
        }catch(Throwable e){
            logger.log(Level.SEVERE,
                       "アプリケーション初期化に失敗しました",
                       e);
            e.printStackTrace(System.err);
            hasError = true;
        }finally{
            if(splashWindow != null){
                splashWindow.setVisible(false);
                splashWindow.dispose();
                splashWindow = null;
                GUIUtils.dispatchEmptyAWTEvent();
            }
        }

        if(hasError) exit(1);

        return;
    }

    /**
     * AWTイベントディスパッチスレッド版スタートアップエントリ。
     */
    private static void startGUI(){
        LandsModel model = new LandsModel();
        model.loadLandList();

        JFrame topFrame = buildMVC(model);

        GUIUtils.modifyWindowAttributes(topFrame, true, false, true);

        topFrame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        topFrame.pack();

        Dimension initGeometry =
                new Dimension(AppSetting.initialFrameWidth(),
                              AppSetting.initialFrameHeight());
        topFrame.setSize(initGeometry);

        if(   AppSetting.initialFrameXpos() <= Integer.MIN_VALUE
           || AppSetting.initialFrameYpos() <= Integer.MIN_VALUE ){
            topFrame.setLocationByPlatform(true);
        }else{
            topFrame.setLocation(AppSetting.initialFrameXpos(),
                                 AppSetting.initialFrameYpos() );
        }

        topFrame.setVisible(true);

        return;
    }

    /**
     * 任意のクラス群に対して一括ロード／初期化を単一スレッドで順に行う。
     * どーしてもクラス初期化の順序に依存する障害が発生する場合や
     * クラス初期化のオーバーヘッドでGUIの操作性が損なわれるときなどにどうぞ。
     *
     * @throws java.lang.LinkageError クラス間リンケージエラー。
     * @throws java.lang.ExceptionInInitializerError クラス初期化で異常
     */
    private static void preInitClass()
            throws LinkageError,
                   ExceptionInInitializerError {
        Object[] classes = {            // Class型 または String型
            "java.lang.Object",
            TabBrowser.class,
            Discussion.class,
            GlyphDraw.class,
            java.net.HttpURLConnection.class,
            java.text.SimpleDateFormat.class,
            Void.class,
        };

        for(Object obj : classes){
            String className;
            if(obj instanceof Class){
                className = ((Class<?>)obj).getName();
            }else if(obj instanceof String){
                className = obj.toString();
            }else{
                continue;
            }

            try{
                if(LOADER != null){
                    Class.forName(className, true, LOADER);
                }else{
                    Class.forName(className);
                }
            }catch(ClassNotFoundException e){
                logger.log(Level.WARNING,
                           "クラスの明示的ロードに失敗しました",
                           e);
                continue;
            }
        }

        return;
    }

    /**
     * 独自ロガーにエラーや例外を吐く、
     * カスタム化されたイベントキューに差し替える。
     */
    private static void replaceEventQueue(){
        final String message = "イベントディスパッチ中に異常が起きました。";
        final Toolkit kit = Toolkit.getDefaultToolkit();

        EventQueue oldQueue = kit.getSystemEventQueue();
        EventQueue newQueue = new EventQueue(){
            @Override
            protected void dispatchEvent(AWTEvent event){
                try{
                    super.dispatchEvent(event);
                }catch(RuntimeException e){
                    logger.log(Level.SEVERE, message, e);
                    throw e;
                }catch(Exception e){
                    logger.log(Level.SEVERE, message, e);
                }catch(Error e){
                    logger.log(Level.SEVERE, message, e);
                    throw e;
                }catch(Throwable e){
                    logger.log(Level.SEVERE, message, e);
                }
                // TODO Toolkit#beep()もするべきか
                // TODO モーダルダイアログを出すべきか
                // TODO 標準エラー出力抑止オプションを用意すべきか
                // TODO セキュリティバイパス
                return;
            }
        };
        oldQueue.push(newQueue);
    }

    /**
     * JRE1.5相当のランタイムライブラリ環境が提供されているか判定する。
     *
     * @return JRE1.5相当ならtrueを返す。
     */
    public static boolean hasRuntime5(){
        boolean result;
        try{
            result = JRE_PACKAGE.isCompatibleWith("1.5");
        }catch(NumberFormatException e){
            assert false;
            return false;
        }
        return result;
    }

    /**
     * JRE1.6相当のランタイムライブラリ環境が提供されているか判定する。
     *
     * @return JRE1.6相当ならtrueを返す。
     */
    public static boolean hasRuntime6(){
        boolean result;
        try{
            result = JRE_PACKAGE.isCompatibleWith("1.6");
        }catch(NumberFormatException e){
            assert false;
            return false;
        }
        return result;
    }

    /**
     * VMごとプログラムを終了する。
     * ※おそらく随所でシャットダウンフックが起動されるはず。
     *
     * @param exitCode 終了コード
     * @throws java.lang.SecurityException セキュリティ違反
     */
    public static void exit(int exitCode) throws SecurityException{
        logger.info(
                "終了コード["
                + exitCode
                + "]でVMごとアプリケーションを終了します。" );
        RUNTIME.runFinalization();
        System.out.flush();
        System.err.flush();
        try{
            RUNTIME.exit(exitCode);
        }catch(SecurityException e){
            logger.log(
                    Level.WARNING,
                     "セキュリティ設定により、"
                    +"VMを終了させることができません。", e);
            throw e;
        }
        return;
    }

    /**
     * コンパイル時のエラーを判定する。
     */
    private static void checkCompileError(){
        if(   '狼' != 0x72fc
           || '　' != 0x3000
           || '~'  != 0x007e
           || '\\' != 0x005c  // バックスラッシュ
           || '¥'  != 0x00a5  // 半角円通貨
           || '～' != 0xff5e
           || '�' != 0xfffd  // Unicode専用特殊文字
           ){
            System.err.println(
                    "ソースコードの文字コードが"
                   +"正しくコンパイルされていないかも。"
                   +"あなたは今、オリジナル開発元の意図しない文字コード環境で"
                   +"コンパイルされたプログラムを起動しようとしているよ。"
                   +"ソースコードの入手に際して"
                   +"どのような文字コード変換が行われたか認識しているかな？"
                   +"コンパイルオプションで正しい文字コードを指定したかな？"
                    );
            RUNTIME.exit(1);
        }
        return;
    }

    /**
     * Jindolf の実行が可能な環境でなければ、即時VM実行を終了する。
     */
    private static void checkEnvironment(){
        if(!hasRuntime5()){
            System.err.println(
                    Jindolf.TITLE
                    + " は JRE1.5 以降の実行環境を検出できませんでした");
            RUNTIME.exit(1);
        }

        if(GraphicsEnvironment.isHeadless()){
            System.err.println(
                    Jindolf.TITLE
                    + " はGUI環境と接続できませんでした");
            RUNTIME.exit(1);
        }

        return;
    }

    /**
     * パッケージ定義エラーの検出。
     */
    private static void checkPackageDefinition(){
        String implVersion = SELF_PACKAGE.getImplementationVersion();
        if(   implVersion != null
           && ! implVersion.equals(VERSION) ){
            System.err.println(
                     "ビルドエラー："
                    +"パッケージ定義とバージョン番号が一致しません。"
                    +"["+ implVersion +"]≠["+ VERSION +"]");
            RUNTIME.exit(1);
        }

        return;
    }

    /**
     * ログ操作のアクセス権があるか否か判定する。
     * @return アクセス権があればtrue
     */
    public static boolean hasLoggingPermission(){
        if(SEC_MANAGER == null) return true;

        Permission logPermission = new LoggingPermission("control", null);
        try{
            SEC_MANAGER.checkPermission(logPermission);
        }catch(SecurityException e){
            return false;
        }

        return true;
    }

    /**
     * ロギング初期化。
     */
    private static void initLogging(){
        boolean hasPermission = hasLoggingPermission();

        if( ! hasPermission){
            System.out.println(
                      "セキュリティ設定により、"
                    + "ログ設定を変更できませんでした" );
        }

        if(hasPermission){
            logger.setUseParentHandlers(false);
            Handler pileHandler = new PileHandler();
            logger.addHandler(pileHandler);
        }

        if(hasPermission && AppSetting.useConsolelog()){
            Handler consoleHandler = new ConsoleHandler();
            logger.addHandler(consoleHandler);
        }

        return;
    }

    /**
     * スプラッシュウィンドウを作成する。
     *
     * @return 未表示のスプラッシュウィンドウ。
     */
    private static Window createSplashWindow(){
        JWindow splashWindow = new JWindow();
        JLabel splashLabel = new JLabel(GUIUtils.getLogoIcon());
        splashWindow.add(splashLabel);

        splashWindow.pack();
        splashWindow.setLocationRelativeTo(null); // locate center

        return splashWindow;
    }

    /**
     * モデル・ビュー・コントローラの結合。
     *
     * @param model 最上位のデータモデル
     * @return アプリケーションのトップフレーム
     */
    private static JFrame buildMVC(LandsModel model){
        ActionManager actionManager = new ActionManager();
        TopView topView = new TopView();

        Controller controller = new Controller(actionManager, topView, model);

        JFrame topFrame = controller.createTopFrame();

        return topFrame;
    }

    /**
     * 標準出力端末にヘルプメッセージ（オプションの説明）を表示する。
     */
    private static void showHelpMessage(){
        System.out.flush();
        System.err.flush();

        CharSequence helpText;
        try{
            helpText = loadResourceText("resources/help.txt");
        }catch(IOException e){
            return;
        }

        System.out.print(helpText);

        System.out.flush();
        System.err.flush();

        return;
    }

    /**
     * リソースからテキストデータをロードする。
     * @param resourceName リソース名
     * @return テキスト文字列
     * @throws java.io.IOException 入出力の異常。おそらくビルドミス。
     */
    public static CharSequence loadResourceText(String resourceName)
            throws IOException{
        InputStream is;
        is = getResourceAsStream(resourceName);
        is = new BufferedInputStream(is);
        Reader reader = new InputStreamReader(is, CHARSET_UTF8);
        LineNumberReader lineReader = new LineNumberReader(reader);

        StringBuilder result = new StringBuilder();
        try{
            for(;;){
                String line = lineReader.readLine();
                if(line == null) break;
                if(line.startsWith("#")) continue;
                result.append(line).append('\n');
            }
        }finally{
            lineReader.close();
        }

        return result;
    }

    /**
     * クラスローダを介してリソースからの入力を生成する。
     * @param name リソース名
     * @return リソースからの入力
     */
    public static InputStream getResourceAsStream(String name){
        return SELF_KLASS.getResourceAsStream(name);
    }

    /**
     * クラスローダを介してリソース読み込み用URLを生成する。
     * @param name リソース名
     * @return URL
     */
    public static URL getResource(String name){
        return SELF_KLASS.getResource(name);
    }

    /**
     * 隠れコンストラクタ。
     */
    private Jindolf(){
        super();
        assert this.getClass() == SELF_KLASS;
        return;
    }

}
