/*
 * content builder
 *
 * Copyright(c) 2009 olyutorskii
 * $Id: ContentBuilder.java 869 2009-10-21 17:24:26Z olyutorskii $
 */

package jp.sourceforge.jindolf.parser;

import java.nio.charset.CharsetDecoder;

/**
 * {@link SjisDecoder}からの通知に従い、
 * {@link DecodedContent}へとデコードする。
 */
public class ContentBuilder implements DecodeHandler{

    private DecodedContent content;

    private boolean hasByte1st;
    private byte byte1st;

    /**
     * コンストラクタ。
     * 長さ0で空の{@link DecodedContent}がセットされる。
     */
    public ContentBuilder(){
        this(128);
        return;
    }

    /**
     * コンストラクタ。
     * 長さ0で空の{@link DecodedContent}がセットされる。
     * @param capacity 初期容量
     * @throws NegativeArraySizeException 容量指定が負。
     */
    public ContentBuilder(int capacity) throws NegativeArraySizeException{
        super();
        this.content = new DecodedContent(capacity);
        initImpl();
        return;
    }

    /**
     * デコード結果の{@link DecodedContent}を取得する。
     * @return デコード結果文字列
     */
    public DecodedContent getContent(){
        return this.content;
    }

    /**
     * デコード処理の初期化下請。
     */
    private void initImpl(){
        this.content.init();
        this.hasByte1st = false;
        this.byte1st = 0x00;
        return;
    }

    /**
     * デコード処理の初期化。
     */
    protected void init(){
        initImpl();
        return;
    }

    /**
     * エラー情報をフラッシュする。
     */
    protected void flushError(){
        if(this.hasByte1st){
            this.content.addDecodeError(this.byte1st);
            this.hasByte1st = false;
        }
        return;
    }

    /**
     * {@inheritDoc}
     * @param decoder {@inheritDoc}
     * @throws DecodeException {@inheritDoc}
     */
    public void startDecoding(CharsetDecoder decoder)
            throws DecodeException{
        init();
        return;
    }

    /**
     * {@inheritDoc}
     * @throws DecodeException {@inheritDoc}
     */
    public void endDecoding()
            throws DecodeException{
        flushError();
        return;
    }

    /**
     * {@inheritDoc}
     * @param seq {@inheritDoc}
     * @throws DecodeException {@inheritDoc}
     */
    public void charContent(CharSequence seq)
            throws DecodeException{
        flushError();
        this.content.append(seq);
        return;
    }

    /**
     * {@inheritDoc}
     * @param errorArray {@inheritDoc}
     * @param offset {@inheritDoc}
     * @param length {@inheritDoc}
     * @throws DecodeException {@inheritDoc}
     */
    public void decodingError(byte[] errorArray, int offset, int length)
            throws DecodeException{
        int limit = offset + length;
        for(int bpos = offset; bpos < limit; bpos++){
            byte bval = errorArray[bpos];
            if( ! this.hasByte1st){
                if(ShiftJis.isShiftJIS1stByte(bval)){
                    this.byte1st = bval;
                    this.hasByte1st = true;
                }else{
                    this.content.addDecodeError(bval);
                }
            }else{
                if(ShiftJis.isShiftJIS2ndByte(bval)){   // 文字集合エラー
                    this.content.addDecodeError(this.byte1st, bval);
                    this.hasByte1st = false;
                }else if(ShiftJis.isShiftJIS1stByte(bval)){
                    this.content.addDecodeError(this.byte1st);
                    this.byte1st = bval;
                    this.hasByte1st = true;
                }else{
                    this.content.addDecodeError(this.byte1st);
                    this.content.addDecodeError(bval);
                    this.hasByte1st = false;
                }
            }
        }

        return;
    }

}
