/*
 * Land information panel
 *
 * Copyright(c) 2009 olyutorskii
 * $Id: LandInfoPanel.java 888 2009-11-04 06:23:35Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.awt.Frame;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.text.DateFormat;
import java.util.Date;
import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;
import jp.sourceforge.jindolf.corelib.LandDef;
import jp.sourceforge.jindolf.corelib.LandState;

/**
 * 国情報表示パネル。
 */
@SuppressWarnings("serial")
public class LandInfoPanel extends JPanel implements ActionListener{

    private static final String COMMAND_SHOWBASE    = "SHOWBASE";
    private static final String COMMAND_SHOWCONTACT = "SHOWCONTACT";

    /**
     * 国の状態を文字列化する。
     * @param state 国状態
     * @return 文字列化された国状態
     */
    private static String getStatusMark(LandState state){
        String result;

        switch(state){
        case CLOSED:     result = "サービス終了";     break;
        case HISTORICAL: result = "過去ログ提供のみ"; break;
        case ACTIVE:     result = "稼動中";           break;
        default:
            assert false;
            result = "";
            break;
        }

        return result;
    }

    private final JLabel landName       = new JLabel();
    private final JLabel landIdentifier = new JLabel();
    private final JLabel baseURL        = new JLabel();
    private final JLabel startDate      = new JLabel();
    private final JLabel endDate        = new JLabel();
    private final JLabel landState      = new JLabel();
    private final JLabel locale         = new JLabel();
    private final JLabel timezone       = new JLabel();
    private final JLabel contact        = new JLabel();
    private final JLabel description    = new JLabel();

    /**
     * コンストラクタ。
     */
    public LandInfoPanel(){
        super();

        GUIUtils.forceMonoPitchFont(this.baseURL);
        GUIUtils.forceMonoPitchFont(this.contact);
        GUIUtils.forceMonoPitchFont(this.landIdentifier);
        GUIUtils.forceMonoPitchFont(this.locale);

        design();

        return;
    }

    /**
     * 一行分レイアウトする。
     * @param item 項目名
     * @param comp コンポーネント
     */
    private void layoutRow(String item, JComponent comp){
        GridBagConstraints constraints = new GridBagConstraints();
        constraints.insets = new Insets(2, 2, 2, 2);

        String itemCaption = item + " : ";
        JLabel itemLabel = new JLabel(itemCaption);

        constraints.anchor = GridBagConstraints.EAST;
        constraints.gridwidth = 1;
        add(itemLabel, constraints);

        constraints.anchor = GridBagConstraints.WEST;
        constraints.gridwidth = GridBagConstraints.REMAINDER;
        add(comp, constraints);

    }

    /**
     * Web表示用パネルの生成。
     * @param label URLを載せるラベル
     * @param command アクションコマンド名
     * @return Web表示用パネル
     */
    private JComponent buildShowWeb(JLabel label, String command){
        JPanel panel = new JPanel();
        panel.setBorder(BorderFactory.createEmptyBorder(0, 0, 0, 0));

        JButton button = new JButton("Web");
        button.setIcon(GUIUtils.getWWWIcon());
        button.setMargin(new Insets(1, 1, 1, 1));
        button.setActionCommand(command);
        button.addActionListener(this);
        button.setToolTipText("Webブラウザで表示");

        GridBagLayout layout = new GridBagLayout();
        panel.setLayout(layout);

        GridBagConstraints constraints = new GridBagConstraints();
        constraints.fill      = GridBagConstraints.NONE;
        constraints.gridwidth = GridBagConstraints.RELATIVE;
        constraints.insets    = new Insets(0, 0, 0, 10);

        panel.add(label,  constraints);
        panel.add(button, constraints);

        return panel;
    }

    /**
     * レイアウトを行う。
     */
    private void design(){
        GridBagLayout layout = new GridBagLayout();
        setLayout(layout);

        layoutRow("国名",      this.landName);
        layoutRow("識別名",    this.landIdentifier);

        JComponent showBase =
                buildShowWeb(this.baseURL, COMMAND_SHOWBASE);
        layoutRow("ベースURL", showBase);

        layoutRow("建国",      this.startDate);
        layoutRow("亡国",      this.endDate);
        layoutRow("状態",      this.landState);
        layoutRow("ロケール",  this.locale);
        layoutRow("時間帯",    this.timezone);

        JComponent showContact =
                buildShowWeb(this.contact, COMMAND_SHOWCONTACT);
        layoutRow("連絡先", showContact);

        layoutRow("説明",      this.description);

        GridBagConstraints constraints = new GridBagConstraints();
        constraints.fill = GridBagConstraints.BOTH;
        constraints.weightx = 1.0;
        constraints.weighty = 1.0;
        constraints.gridwidth = GridBagConstraints.REMAINDER;
        constraints.gridheight = GridBagConstraints.REMAINDER;
        add(new JPanel(), constraints);  // ダミー詰め物

        return;
    }

    /**
     * 国情報を更新する。
     * @param land 国
     */
    public void update(Land land){
        LandDef landDef = land.getLandDef();

        DateFormat dform =
            DateFormat.getDateTimeInstance(DateFormat.FULL,
                                           DateFormat.FULL);

        long start = landDef.getStartDateTime();
        String startStr = dform.format(new Date(start));
        if(start < 0){
            startStr = "(不明)";
        }

        long end   = landDef.getEndDateTime();
        String endStr = dform.format(new Date(end));
        if(end < 0){
            endStr = "まだまだ";
        }

        String status = getStatusMark(land.getLandDef().getLandState());

        String base = land.getServerAccess().getBaseURL().toString();

        this.landName       .setText(landDef.getLandName());
        this.landIdentifier .setText(landDef.getLandId());
        this.baseURL        .setText(base);
        this.startDate      .setText(startStr);
        this.endDate        .setText(endStr);
        this.landState      .setText(status);
        this.locale         .setText(landDef.getLocale().toString());
        this.timezone       .setText(landDef.getTimeZone().getDisplayName());
        this.contact        .setText(landDef.getContactInfo());
        this.description    .setText(landDef.getDescription());

        revalidate();

        return;
    }

    /**
     * ボタン押下処理。
     * @param event ボタン押下イベント
     */
    public void actionPerformed(ActionEvent event){
        Window ancestor = SwingUtilities.getWindowAncestor(this);
        Frame frame = null;
        if(ancestor instanceof Frame){
            frame = (Frame) ancestor;
        }

        String command = event.getActionCommand();
        if(command.equals(COMMAND_SHOWBASE)){
            WebIPCDialog.showDialog(frame, this.baseURL.getText());
        }else if(command.equals(COMMAND_SHOWCONTACT)){
            WebIPCDialog.showDialog(frame, this.contact.getText());
        }

        return;
    }

}
