/*
 * アンカー描画
 *
 * Copyright(c) 2008 olyutorskii
 * $Id: AnchorDraw.java 933 2009-12-02 09:07:01Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.Point;
import java.awt.Rectangle;
import java.io.IOException;
import java.util.regex.Pattern;
import jp.sourceforge.jindolf.corelib.TalkType;

/**
 * アンカー描画。
 */
public class AnchorDraw extends AbstractTextRow{

    private static final Color COLOR_ANCHOR   = new Color(0xffff99);
    private static final int UNDER_MARGIN = 5;

    private final Talk talk;
    private final GlyphDraw caption;
    private final GlyphDraw dialog;
    private final Image faceImage;
    private final Point imageOrigin   = new Point();
    private final Point captionOrigin = new Point();
    private final Point dialogOrigin  = new Point();

    /**
     * コンストラクタ。
     * @param talk 発言
     */
    public AnchorDraw(Talk talk){
        this(talk, FontInfo.DEFAULT_FONTINFO);
        return;
    }

    /**
     * コンストラクタ。
     * @param talk 発言
     * @param fontInfo フォント設定
     */
    public AnchorDraw(Talk talk, FontInfo fontInfo){
        super(fontInfo);

        this.talk = talk;

        Period period = this.talk.getPeriod();
        Village village = period.getVillage();
        Avatar avatar = this.talk.getAvatar();

        Image image;
        if(this.talk.getTalkType() == TalkType.GRAVE){
            image = village.getGraveImage();
        }else{
            image = village.getAvatarFaceImage(avatar);
        }
        this.faceImage = image;

        this.caption = new GlyphDraw(getCaptionString(), this.fontInfo);
        this.dialog  = new GlyphDraw(this.talk.getDialog(), this.fontInfo);

        this.caption.setColor(COLOR_ANCHOR);
        this.dialog .setColor(COLOR_ANCHOR);

        return;
    }

    /**
     * キャプション文字列の取得。
     * @return キャプション文字列
     */
    private CharSequence getCaptionString(){
        StringBuilder result = new StringBuilder();
        Avatar avatar = this.talk.getAvatar();
        result.append(avatar.getFullName())
              .append(' ')
              .append(this.talk.getAnchorNotation());
        return result;
    }

    /**
     * {@inheritDoc}
     * @param newWidth {@inheritDoc}
     * @return {@inheritDoc}
     */
    public Rectangle setWidth(int newWidth){
        int imageWidth  = this.faceImage.getWidth(null);
        int imageHeight = this.faceImage.getHeight(null);

        this.caption.setWidth(newWidth - imageWidth);
        int captionWidth  = this.caption.getWidth();
        int captionHeight = this.caption.getHeight();

        this.dialog.setWidth(newWidth);
        int dialogWidth  = this.dialog.getWidth();
        int dialogHeight = this.dialog.getHeight();

        int headerHeight = Math.max(imageHeight, captionHeight);

        int totalWidth = Math.max(imageWidth + captionWidth, dialogWidth);

        int totalHeight = headerHeight;
        totalHeight += dialogHeight;

        int imageYpos;
        int captionYpos;
        if(imageHeight > captionHeight){
            imageYpos = 0;
            captionYpos = (imageHeight - captionHeight) / 2;
        }else{
            imageYpos = (captionHeight - imageHeight) / 2;
            captionYpos = 0;
        }

        this.imageOrigin  .setLocation(0,          imageYpos);
        this.captionOrigin.setLocation(imageWidth, captionYpos);
        this.dialogOrigin .setLocation(0,          headerHeight);

        this.bounds.width  = totalWidth;
        this.bounds.height = totalHeight + UNDER_MARGIN;

        return this.bounds;
    }

    /**
     * {@inheritDoc}
     * @param fontInfo {@inheritDoc}
     */
    @Override
    public void setFontInfo(FontInfo fontInfo){
        super.setFontInfo(fontInfo);

        this.caption.setFontInfo(this.fontInfo);
        this.dialog .setFontInfo(this.fontInfo);

        int width = getWidth();
        setWidth(width);

        return;
    }

    /**
     * {@inheritDoc}
     * @param from {@inheritDoc}
     * @param to {@inheritDoc}
     */
    public void drag(Point from, Point to){
        this.caption.drag(from, to);
        this.dialog.drag(from, to);
        return;
    }

    /**
     * {@inheritDoc}
     * @param appendable {@inheritDoc}
     * @return {@inheritDoc}
     * @throws java.io.IOException {@inheritDoc}
     */
    public Appendable appendSelected(Appendable appendable)
            throws IOException{
        this.caption.appendSelected(appendable);
        this.dialog.appendSelected(appendable);
        return appendable;
    }

    /**
     * {@inheritDoc}
     */
    public void clearSelect(){
        this.caption.clearSelect();
        this.dialog.clearSelect();
        return;
    }

    /**
     * 検索文字列パターンを設定する。
     * @param searchRegex パターン
     * @return ヒット数
     */
    public int setRegex(Pattern searchRegex){
        int total = 0;

        total += this.dialog.setRegex(searchRegex);

        return total;
    }

    /**
     * {@inheritDoc}
     * @param g {@inheritDoc}
     */
    public void paint(Graphics2D g){
        final int xPos = this.bounds.x;
        final int yPos = this.bounds.y;

        g.drawImage(this.faceImage,
                    xPos + this.imageOrigin.x,
                    yPos + this.imageOrigin.y,
                    null );

        this.caption.setPos(xPos + this.captionOrigin.x,
                            yPos + this.captionOrigin.y );
        this.caption.paint(g);

        this.dialog.setPos(xPos + this.dialogOrigin.x,
                           yPos + this.dialogOrigin.y );
        this.dialog.paint(g);

        return;
    }

}
