/*
 * Summarize game information
 *
 * Copyright(c) 2009 olyutorskii
 * $Id: GameSummary.java 888 2009-11-04 06:23:35Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.text.DateFormat;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import jp.sourceforge.jindolf.corelib.Destiny;
import jp.sourceforge.jindolf.corelib.GameRole;
import jp.sourceforge.jindolf.corelib.SysEventType;
import jp.sourceforge.jindolf.corelib.Team;
import jp.sourceforge.jindolf.corelib.VillageState;

/**
 * ゲームのサマリ集計。
 */
public class GameSummary{

    private static Comparator<Player> castingComparator =
            new CastingComparator();

    /**
     * プレイヤーのリストから役職バランス文字列を得る。
     * @param players プレイヤーのリスト
     * @return 役職バランス文字列
     */
    public static CharSequence getRoleBalanceSequence(List<Player> players){
        List<GameRole> roleList = new LinkedList<GameRole>();
        for(Player player : players){
            GameRole role = player.getRole();
            roleList.add(role);
        }
        Collections.sort(roleList, GameRole.getPowerBalanceComparator());

        StringBuilder result = new StringBuilder();
        for(GameRole role : roleList){
            char ch = role.getShortName();
            result.append(ch);
        }

        return result;
    }

    private final Map<Avatar, Player> playerMap =
            new HashMap<Avatar, Player>();
    private final List<Player> playerList =
            new LinkedList<Player>();

    private final Village village;

    private Team winner;

    private int ctScryVillage = 0;
    private int ctScryMadman  = 0;
    private int ctScryWolf    = 0;

    private int ctGuardVillage   = 0;
    private int ctGuardMadman    = 0;
    private int ctGuardWolf      = 0;
    private int ctGuardVillageGJ = 0;
    private int ctGuardMadmanGJ  = 0;
    private int ctGuardFakeGJ    = 0;

    private long talk1stTime = -1;
    private long talkLastTime = -1;

    /**
     * コンストラクタ。
     * @param village 村
     */
    public GameSummary(Village village){
        super();

        if(village == null) throw new NullPointerException();

        VillageState state = village.getState();
        if(   state != VillageState.EPILOGUE
           && state != VillageState.GAMEOVER){
            throw new IllegalStateException();
        }

        this.village = village;

        for(Period period : village.getPeriodList()){
            summarizePeriod(period);
        }

        summarizeJudge();
        summarizeGuard();

        return;
    }

    /**
     * Periodのサマライズ。
     * @param period Period
     */
    private void summarizePeriod(Period period){
        for(Topic topic : period.getTopicList()){
            if(topic instanceof SysEvent){
                SysEvent sysEvent = (SysEvent) topic;
                summarizeSysEvent(period, sysEvent);
            }else if(topic instanceof Talk){
                Talk talk = (Talk) topic;
                summarizeTalk(period, talk);
            }
        }

        return;
    }

    /**
     * システムイベントのサマライズ。
     * @param period Period
     * @param sysEvent システムイベント
     */
    private void summarizeSysEvent(Period period, SysEvent sysEvent){
        List<Avatar>       avatarList  = sysEvent.getAvatarList();
        List<GameRole>     roleList    = sysEvent.getRoleList();
        List<Integer>      integerList = sysEvent.getIntegerList();
        List<CharSequence> textList    = sysEvent.getCharSequenceList();

        SysEventType eventType = sysEvent.getSysEventType();
        switch(eventType){
        case WINVILLAGE:
            this.winner = Team.VILLAGE;
            break;
        case WINWOLF:
            this.winner = Team.WOLF;
            break;
        case WINHAMSTER:
            this.winner = Team.HAMSTER;
            break;
        case PLAYERLIST:
            int avatarNum = avatarList.size();
            for(int idx = 0; idx < avatarNum; idx++){
                Avatar avatar = avatarList.get(idx);
                GameRole role = roleList.get(idx);
                CharSequence urlText = textList.get(idx * 2);
                CharSequence idName    = textList.get(idx * 2 + 1);
                int liveOrDead = integerList.get(idx);

                Player player = registPlayer(avatar);
                player.setRole(role);
                player.setUrlText(urlText.toString());
                player.setIdName(idName.toString());
                if(liveOrDead != 0){
                    player.setObitDay(-1);
                    player.setDestiny(Destiny.ALIVE);
                }

                this.playerList.add(player);
            }
            break;
        case SUDDENDEATH:
            Avatar suddenDeathAvatar = avatarList.get(0);
            Player suddenDeathPlayer = registPlayer(suddenDeathAvatar);
            suddenDeathPlayer.setDestiny(Destiny.SUDDENDEATH);
            suddenDeathPlayer.setObitDay(period.getDay());
            break;
        case COUNTING:
            int avatarTotal = avatarList.size();
            if(avatarTotal % 2 == 0) break;
            Avatar executedAvatar = avatarList.get(avatarTotal - 1);
            Player executedPlayer = registPlayer(executedAvatar);
            executedPlayer.setDestiny(Destiny.EXECUTED);
            executedPlayer.setObitDay(period.getDay());
            break;
        case MURDERED:
            for(Avatar avatar : avatarList){
                Player player = registPlayer(avatar);
                player.setDestiny(Destiny.EATEN);
                player.setObitDay(period.getDay());
            }
            // TODO ハム溶けは後回し
            break;
        case ONSTAGE:
            Avatar onstageAvatar = avatarList.get(0);
            Player onstagePlayer = registPlayer(onstageAvatar);
            onstagePlayer.setEntryNo((int) integerList.get(0));
            break;
        default:
            break;
        }

        return;
    }

    /**
     * 会話のサマライズ。
     * @param period Period
     * @param talk 会話
     */
    private void summarizeTalk(Period period, Talk talk){
        long epoch = talk.getTimeFromID();

        if(talk1stTime < 0) talk1stTime = epoch;
        if(talkLastTime < 0) talkLastTime = epoch;

        if(epoch < talk1stTime) talk1stTime = epoch;
        if(epoch > talkLastTime) talkLastTime = epoch;

        return;
    }

    /**
     * 占い師の活動を集計する。
     */
    private void summarizeJudge(){
        for(Period period : this.village.getPeriodList()){
            for(Topic topic : period.getTopicList()){
                if( ! (topic instanceof SysEvent) ) continue;
                SysEvent sysEvent = (SysEvent) topic;
                if(sysEvent.getSysEventType() != SysEventType.JUDGE) continue;
                List<Avatar> avatarList  = sysEvent.getAvatarList();
                Avatar avatar = avatarList.get(1);
                Player seered = getPlayer(avatar);
                GameRole role = seered.getRole();
                switch(role){
                case WOLF:   this.ctScryWolf++;    break;
                case MADMAN: this.ctScryMadman++;  break;
                default:     this.ctScryVillage++; break;
                }
            }
        }
        return;
    }

    /**
     * 占い師の活動を文字列化する。
     * @return 占い師の活動
     */
    public CharSequence dumpSeerActivity(){
        StringBuilder result = new StringBuilder();

        if(this.ctScryVillage > 0){
            result.append("村陣営を");
            result.append(this.ctScryVillage);
            result.append("回");
        }

        if(this.ctScryMadman > 0){
            if(result.length() > 0) result.append('、');
            result.append("狂人を");
            result.append(this.ctScryMadman);
            result.append("回");
        }

        if(this.ctScryWolf > 0){
            if(result.length() > 0) result.append('、');
            result.append("人狼を");
            result.append(this.ctScryWolf);
            result.append("回");
        }

        if(result.length() <= 0) result.append("誰も占わなかった。");
        else                     result.append("占った。");

        CharSequence seq = WolfBBS.escapeWikiSyntax(result);

        return seq;
    }

    /**
     * 狩人の活動を集計する。
     */
    private void summarizeGuard(){
        for(Period period : village.getPeriodList()){
            if(period.getDay() <= 2) continue;
            boolean hunterAlive = false;
            boolean hasAssault = false;
            int wolfNum = 0;
            GameRole guardedRole = null;
            for(Topic topic : period.getTopicList()){
                if(topic instanceof Talk){
                    Talk talk = (Talk) topic;
                    if(talk.getTalkCount() <= 0){
                        hasAssault = true;
                    }
                    continue;
                }
                if( ! (topic instanceof SysEvent) ) continue;
                SysEvent sysEvent = (SysEvent) topic;
                SysEventType type = sysEvent.getSysEventType();
                if(type == SysEventType.COUNTING){
                    List<Avatar> avatarList  = sysEvent.getAvatarList();
                    int avatarNum = avatarList.size();
                    for(int index = 0; index <= avatarNum - 1; index += 2){
                        Avatar avatar = avatarList.get(index);
                        Player player = getPlayer(avatar);
                        if(player.getRole() == GameRole.HUNTER){
                            if(index >= avatarNum - 1){
                                hunterAlive = false;
                            }else{
                                hunterAlive = true;
                            }
                        }else if(player.getRole() == GameRole.WOLF){
                            if(index >= avatarNum - 1){
                                wolfNum--;
                            }else{
                                wolfNum++;
                            }
                        }
                    }
                }else if(type == SysEventType.GUARD){
                    List<Avatar> avatarList  = sysEvent.getAvatarList();
                    Avatar avatar = avatarList.get(1);
                    Player guarded = getPlayer(avatar);
                    guardedRole = guarded.getRole();
                    switch(guardedRole){
                    case WOLF:   this.ctGuardWolf++;    break;
                    case MADMAN: this.ctGuardMadman++;  break;
                    default:     this.ctGuardVillage++; break;
                    }
                }else if(type == SysEventType.NOMURDER){
                    if( ! hunterAlive ) continue;
                    if( (! hasAssault) && wolfNum > 0){
                        this.ctGuardFakeGJ++;   // 偽装GJ
                    }else if(hasAssault && guardedRole != null){
                        switch(guardedRole){
                        case MADMAN: this.ctGuardMadmanGJ++;  break;
                        default:     this.ctGuardVillageGJ++; break;
                        }
                    }
                }
            }
        }

        return;
    }

    /**
     * 狩人の活動を文字列化する。
     * @return 狩人の活動
     */
    public CharSequence dumpHunterActivity(){
        StringBuilder result = new StringBuilder();

        String atLeast;
        if(this.ctGuardFakeGJ > 0) atLeast = "少なくとも";
        else                       atLeast = "";

        if(this.ctGuardVillage > 0){
            result.append(atLeast);
            result.append("村陣営を");
            result.append(this.ctGuardVillage);
            result.append("回護衛し");
            if(this.ctGuardVillageGJ > 0){
                result.append("GJを");
                result.append(this.ctGuardVillageGJ);
                result.append("回出した。");
            }else{
                result.append("た。");
            }
        }

        if(this.ctGuardMadman > 0){
            result.append(atLeast);
            result.append("狂人を");
            result.append(this.ctGuardMadman);
            result.append("回護衛し");
            if(this.ctGuardMadmanGJ > 0){
                result.append("GJを");
                result.append(this.ctGuardMadmanGJ);
                result.append("回出した。");
            }else{
                result.append("た。");
            }
        }

        if(this.ctGuardWolf > 0){
            result.append(atLeast);
            result.append("人狼を");
            result.append(this.ctGuardWolf);
            result.append("回護衛した。");
        }

        if(this.ctGuardFakeGJ > 0){
            result.append("護衛先は不明ながら偽装GJが");
            result.append(this.ctGuardFakeGJ);
            result.append("回あった。");
        }

        if(result.length() <= 0) result.append("誰も護衛しなかった");

        CharSequence seq = WolfBBS.escapeWikiSyntax(result);

        return seq;
    }

    /**
     * 処刑概観を文字列化する。
     * @return 文字列化した処刑概観
     */
    public CharSequence dumpExecutionInfo(){
        StringBuilder result = new StringBuilder();

        int exeWolf = 0;
        int exeMad = 0;
        int exeVillage = 0;
        for(Player player : this.playerList){
            Destiny destiny = player.getDestiny();
            if(destiny != Destiny.EXECUTED) continue;
            GameRole role = player.getRole();
            switch(role){
            case WOLF:   exeWolf++;    break;
            case MADMAN: exeMad++;     break;
            default:     exeVillage++; break;
            }
        }

        if(exeVillage > 0){
            result.append("▼村陣営×").append(exeVillage).append("回");
        }
        if(exeMad > 0){
            if(result.length() > 0) result.append("、");
            result.append("▼狂×").append(exeMad).append("回");
        }
        if(exeWolf > 0){
            if(result.length() > 0) result.append("、");
            result.append("▼狼×").append(exeWolf).append("回");
        }
        if(result.length() <= 0) result.append("なし");

        CharSequence seq = WolfBBS.escapeWikiSyntax(result);

        return seq;
    }

    /**
     * 襲撃概観を文字列化する。
     * @return 文字列化した襲撃概観
     */
    public CharSequence dumpAssaultInfo(){
        StringBuilder result = new StringBuilder();

        int eatMad = 0;
        int eatVillage = 0;
        for(Player player : this.playerList){
            if(player.getAvatar() == Avatar.AVATAR_GERD){
                result.append("▲ゲルト");
                continue;
            }
            Destiny destiny = player.getDestiny();
            if(destiny != Destiny.EATEN) continue;
            GameRole role = player.getRole();
            switch(role){
            case MADMAN: eatMad++;     break;
            default:     eatVillage++; break;
            }
        }

        if(eatVillage > 0){
            if(result.length() > 0) result.append("、");
            result.append("▲村陣営×").append(eatVillage).append("回");
        }
        if(eatMad > 0){
            if(result.length() > 0) result.append("、");
            result.append("▲狂×").append(eatMad).append("回");
        }

        if(result.length() <= 0) result.append("襲撃なし");

        CharSequence seq = WolfBBS.escapeWikiSyntax(result);

        return seq;
    }

    /**
     * まとめサイト用投票Boxを生成する。
     * @return 投票BoxのWikiテキスト
     */
    public CharSequence dumpVoteBox(){
        StringBuilder wikiText = new StringBuilder();

        for(Player player : getCastingPlayerList()){
            Avatar avatar = player.getAvatar();
            if(avatar == Avatar.AVATAR_GERD) continue;
            GameRole role = player.getRole();
            CharSequence fullName = avatar.getFullName();
            CharSequence roleName = role.getRoleName();
            StringBuilder line = new StringBuilder();
            line.append("[").append(roleName).append("] ").append(fullName);
            if(wikiText.length() > 0) wikiText.append(',');
            wikiText.append(WolfBBS.escapeWikiSyntax(line));
            wikiText.append("[0]");
        }

        wikiText.insert(0, "#vote(").append(")\n");

        return wikiText;
    }

    /**
     * まとめサイト用キャスト表を生成する。
     * @param iconSet 顔アイコンセット
     * @return キャスト表のWikiテキスト
     */
    public CharSequence dumpCastingBoard(FaceIconSet iconSet){
        StringBuilder wikiText = new StringBuilder();

        String vName = this.village.getVillageFullName();
        String generator = Jindolf.TITLE + " Ver." + Jindolf.VERSION;
        String author = iconSet.getAuthor() + "氏"
                       +" [ "+iconSet.getUrlText()+" ]";

        wikiText.append(WolfBBS.COMMENTLINE);
        wikiText.append("// ↓キャスト表開始\n");
        wikiText.append("//        Village : " + vName + "\n");
        wikiText.append("//        Generator : " + generator + "\n");
        wikiText.append("//        アイコン作者 : " + author + '\n');
        wikiText.append("// ※アイコン画像の著作財産権保持者"
                       +"および画像サーバ運営者から\n");
        wikiText.append("// 新しい意向が示された場合、"
                       +"そちらを最優先で尊重してください。\n");
        wikiText.append(WolfBBS.COMMENTLINE);

        wikiText.append("|配役")
                .append("|参加者")
                .append("|役職")
                .append("|運命")
                .append("|その活躍")
                .append("|h")
                .append('\n');
        wikiText.append(WolfBBS.COMMENTLINE);

        for(Player player : getCastingPlayerList()){
            Avatar avatar   = player.getAvatar();
            GameRole role   = player.getRole();
            Destiny destiny = player.getDestiny();
            int obitDay     = player.getObitDay();
            String name     = player.getIdName();
            String urlText  = player.getUrlText();
            if(urlText == null) urlText = "";
            urlText = urlText.replace("~", "%7e");
            urlText = urlText.replace(" ", "%20");
            try{
                URL url = new URL(urlText);
                URI uri = url.toURI();
                urlText = uri.toASCIIString();
            }catch(MalformedURLException e){
                // NOTHING
            }catch(URISyntaxException e){
                // NOTHING
            }
            // PukiWikiではURL内の&のエスケープは不要?

            wikiText.append("// ========== ");
            wikiText.append(name + " acts as [" + avatar.getName() + "]");
            wikiText.append(" ==========\n");

            String teamColor =  "BGCOLOR("
                              + WolfBBS.getTeamWikiColor(role)
                              + "):";

            String avatarIcon = iconSet.getAvatarIconWiki(avatar);

            wikiText.append('|').append(teamColor);
            wikiText.append(avatarIcon).append("&br;");

            wikiText.append("[[").append(avatar.getName()).append("]]");

            wikiText.append('|').append(teamColor);
            wikiText.append("[[").append(WolfBBS.escapeWikiBracket(name));
            if(urlText != null && urlText.length() > 0){
                wikiText.append('>').append(urlText);
            }
            wikiText.append("]]");

            wikiText.append('|').append(teamColor);
            wikiText.append(WolfBBS.getRoleIconWiki(role));
            wikiText.append("&br;");
            wikiText.append("[[");
            wikiText.append(role.getRoleName());
            wikiText.append("]]");

            String destinyColor = WolfBBS.getDestinyColorWiki(destiny);
            wikiText.append('|');
            wikiText.append("BGCOLOR(").append(destinyColor).append("):");
            if(destiny == Destiny.ALIVE){
                wikiText.append("最後まで&br;生存");
            }else{
                wikiText.append(obitDay).append("日目").append("&br;");
                wikiText.append(destiny.getMessage());
            }

            wikiText.append('|');
            wikiText.append(avatar.getJobTitle()).append('。');

            if(avatar == Avatar.AVATAR_GERD){
                wikiText.append("寝てばかりいた。");
            }else if(role == GameRole.HUNTER){
                CharSequence report = dumpHunterActivity();
                wikiText.append(report);
            }else if(role == GameRole.SEER){
                CharSequence report = dumpSeerActivity();
                wikiText.append(report);
            }

            wikiText.append("|\n");

        }

        wikiText.append("|>|>|>|>|");
        wikiText.append("RIGHT:");
        wikiText.append("顔アイコン提供 : [[");
        wikiText.append(WolfBBS.escapeWikiBracket(iconSet.getAuthor()));
        wikiText.append(">" + iconSet.getUrlText());
        wikiText.append("]]氏");
        wikiText.append("|\n");

        wikiText.append(WolfBBS.COMMENTLINE);
        wikiText.append("// ↑キャスト表ここまで\n");
        wikiText.append(WolfBBS.COMMENTLINE);

        return wikiText;
    }

    /**
     * 村詳細情報を出力する。
     * @return 村詳細情報
     */
    public CharSequence dumpVillageWiki(){
        StringBuilder wikiText = new StringBuilder();

        DateFormat dform =
                DateFormat.getDateTimeInstance(DateFormat.FULL,
                                               DateFormat.FULL);

        String vName = this.village.getVillageFullName();
        String generator = Jindolf.TITLE + " Ver." + Jindolf.VERSION;

        wikiText.append(WolfBBS.COMMENTLINE);
        wikiText.append("// ↓村詳細開始\n");
        wikiText.append("//        Village : " + vName + "\n");
        wikiText.append("//        Generator : " + generator + "\n");

        wikiText.append("* 村の詳細\n");

        wikiText.append(WolfBBS.COMMENTLINE);
        wikiText.append("- 勝者\n");
        Team winnerTeam = getWinnerTeam();
        String wonTeam = winnerTeam.getTeamName();
        wikiText.append(wonTeam).append('\n');

        wikiText.append(WolfBBS.COMMENTLINE);
        wikiText.append("- エントリー開始時刻\n");
        Date date = get1stTalkDate();
        String talk1st = dform.format(date);
        wikiText.append(talk1st).append('\n');

        wikiText.append(WolfBBS.COMMENTLINE);
        wikiText.append("- 参加人数\n");
        int avatarNum = countAvatarNum();
        String totalMember = "ゲルト + " + (avatarNum - 1) + "名 = "
                            + avatarNum + "名";
        wikiText.append(WolfBBS.escapeWikiSyntax(totalMember))
                .append('\n');

        wikiText.append(WolfBBS.COMMENTLINE);
        wikiText.append("- 役職内訳\n");
        StringBuilder roleMsg = new StringBuilder();
        for(GameRole role : GameRole.values()){
            List<Player> players = getRoledPlayerList(role);
            String roleName = role.getRoleName();
            if(players.size() <= 0) continue;
            if(roleMsg.length() > 0) roleMsg.append('、');
            roleMsg.append(roleName)
                   .append(" × ")
                   .append(players.size())
                   .append("名");
        }
        wikiText.append(WolfBBS.escapeWikiSyntax(roleMsg)).append('\n');

        wikiText.append(WolfBBS.COMMENTLINE);
        wikiText.append("- 処刑内訳\n");
        wikiText.append(dumpExecutionInfo()).append('\n');

        wikiText.append(WolfBBS.COMMENTLINE);
        wikiText.append("- 襲撃内訳\n");
        wikiText.append(dumpAssaultInfo()).append('\n');

        wikiText.append(WolfBBS.COMMENTLINE);
        wikiText.append("- 突然死\n");
        wikiText.append(countSuddenDeath()).append("名").append('\n');

        wikiText.append(WolfBBS.COMMENTLINE);
        wikiText.append("- 人口推移\n");
        for(int day = 1; day < this.village.getPeriodSize(); day++){
            List<Player> players = getSurvivorList(day);
            CharSequence roleSeq =
                    GameSummary.getRoleBalanceSequence(players);
            String daySeq;
            Period period = this.village.getPeriod(day);
            daySeq = period.getCaption();
            wikiText.append('|')
                    .append(daySeq)
                    .append('|')
                    .append(roleSeq)
                    .append("|\n");
        }

        wikiText.append(WolfBBS.COMMENTLINE);
        wikiText.append("- 占い師の成績\n");
        wikiText.append(dumpSeerActivity()).append('\n');

        wikiText.append(WolfBBS.COMMENTLINE);
        wikiText.append("- 狩人の成績\n");
        wikiText.append(dumpHunterActivity()).append('\n');

        wikiText.append(WolfBBS.COMMENTLINE);
        wikiText.append("// ↑村詳細ここまで\n");
        wikiText.append(WolfBBS.COMMENTLINE);

        return wikiText;
    }

    /**
     * 最初の発言の時刻を得る。
     * @return 時刻
     */
    public Date get1stTalkDate(){
        return new Date(this.talk1stTime);
    }

    /**
     * 最後の発言の時刻を得る。
     * @return 時刻
     */
    public Date getLastTalkDate(){
        return new Date(this.talkLastTime);
    }

    /**
     * 指定した日の生存者一覧を得る。
     * @param day 日
     * @return 生存者一覧
     */
    public List<Player> getSurvivorList(int day){
        if(day < 0 || this.village.getPeriodSize() <= day){
            throw new IndexOutOfBoundsException();
        }

        List<Player> result = new LinkedList<Player>();

        Period period = this.village.getPeriod(day);

        if(   period.isPrologue()
           || (period.isProgress() && day == 1) ){
            result.addAll(this.playerList);
            return result;
        }

        if(period.isEpilogue()){
            for(Player player : this.playerList){
                if(player.getDestiny() == Destiny.ALIVE){
                    result.add(player);
                }
            }
            return result;
        }

        for(Topic topic : period.getTopicList()){
            if( ! (topic instanceof SysEvent) ) continue;
            SysEvent sysEvent = (SysEvent) topic;
            if(sysEvent.getSysEventType() == SysEventType.SURVIVOR){
                List<Avatar> avatarList = sysEvent.getAvatarList();
                for(Avatar avatar : avatarList){
                    Player player = getPlayer(avatar);
                    result.add(player);
                }
            }
        }

        return result;
    }

    /**
     * プレイヤー一覧を得る。
     * 参加エントリー順
     * @return プレイヤーのリスト
     */
    public List<Player> getPlayerList(){
        List<Player> result = Collections.unmodifiableList(this.playerList);
        return result;
    }

    /**
     * キャスティング表用にソートされたプレイヤー一覧を得る。
     * @return プレイヤーのリスト
     */
    public List<Player> getCastingPlayerList(){
        List<Player> sortedPlayers =
                new LinkedList<Player>();
        sortedPlayers.addAll(this.playerList);
        Collections.sort(sortedPlayers, castingComparator);
        return sortedPlayers;
    }

    /**
     * 指定された役職のプレイヤー一覧を得る。
     * @param role 役職
     * @return 役職に合致するプレイヤーのリスト
     */
    public List<Player> getRoledPlayerList(GameRole role){
        List<Player> result = new LinkedList<Player>();

        for(Player player : this.playerList){
            if(player.getRole() == role){
                result.add(player);
            }
        }

        return result;
    }

    /**
     * 勝利陣営を得る。
     * @return 勝利した陣営
     */
    public Team getWinnerTeam(){
        return this.winner;
    }

    /**
     * 突然死者数を得る。
     * @return 突然死者数
     */
    public int countSuddenDeath(){
        int suddenDeath = 0;
        for(Player player : this.playerList){
            if(player.getDestiny() == Destiny.SUDDENDEATH) suddenDeath++;
        }
        return suddenDeath;
    }

    /**
     * 参加プレイヤー総数を得る。
     * @return プレイヤー総数
     */
    public int countAvatarNum(){
        int playerNum = this.playerList.size();
        return playerNum;
    }

    /**
     * AvatarからPlayerを得る。
     * 参加していないAvatarならnullを返す。
     * @param avatar Avatar
     * @return Player
     */
    public Player getPlayer(Avatar avatar){
        Player player = this.playerMap.get(avatar);
        return player;
    }

    /**
     * AvatarからPlayerを得る。
     * 無ければ新規に作る。
     * @param avatar Avatar
     * @return Player
     */
    private Player registPlayer(Avatar avatar){
        Player player = getPlayer(avatar);
        if(player == null){
            player = new Player();
            player.setAvatar(avatar);
            this.playerMap.put(avatar, player);
        }
        return player;
    }

    /**
     * プレイヤーのソート仕様の記述。
     * まとめサイトのキャスト表向け。
     */
    private static class CastingComparator implements Comparator<Player> {

        /**
         * コンストラクタ。
         */
        public CastingComparator(){
            super();
            return;
        }

        /**
         * {@inheritDoc}
         * @param p1 {@inheritDoc}
         * @param p2 {@inheritDoc}
         * @return {@inheritDoc}
         */
        public int compare(Player p1, Player p2){
            if(p1 == p2) return 0;
            if(p2 == null) return +1;
            if(p1 == null) return -1;

            Avatar avatar1 = p1.getAvatar();
            Avatar avatar2 = p2.getAvatar();
            int avatarOrder = avatar1.compareTo(avatar2);
            if(avatarOrder == 0) return 0;

            if(avatar1 == Avatar.AVATAR_GERD) return -1;
            if(avatar2 == Avatar.AVATAR_GERD) return +1;

            if(   p1.getDestiny() != Destiny.ALIVE
               && p2.getDestiny() == Destiny.ALIVE){
                return -1;
            }else if(   p1.getDestiny() == Destiny.ALIVE
                     && p2.getDestiny() != Destiny.ALIVE){
                return +1;
            }

            if(p1.getObitDay() > p2.getObitDay()) return +1;
            if(p1.getObitDay() < p2.getObitDay()) return -1;

            int destinyOrder = p1.getDestiny().compareTo(p2.getDestiny());
            if(destinyOrder != 0) return destinyOrder;

            int entryOrder = p1.getEntryNo() - p2.getEntryNo();

            return entryOrder;
        }
    }

    // TODO ハムスター対応
}
