/* Copyright 2007 Harai Akihiro.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *     http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sourceforge.jlogtest.sample.minesweeper;

import java.util.Set;

public class MineSweeper {
	
	private Cell[] board;
	private final int widthOfBoard;

	public MineSweeper(
			final Size boardSize,
			final int numOfMines,
			final IRandomPermutator permutator) {
		
		if (boardSize.getCellNum() <= numOfMines)
			throw new IllegalArgumentException();

		board = new Cell[boardSize.getCellNum()];
		widthOfBoard = boardSize.getWidth();
		final boolean[] mines = permutator.getRandomizedMineMap(getNumOfCells(), numOfMines);
		for (int i = 0; i < board.length; i++)
			board[i] = mines[i] ? new CellWithMine() : new CellWithoutMine();
	}

	private int getX(final int serial) {
		return serial % widthOfBoard;
	}
	
	private int getY(final int serial) {
		return serial / widthOfBoard;
	}
	
	private int getNumOfCells() {
		return board.length;
	}
	
	private int getHeight() {
		return getNumOfCells() / widthOfBoard;
	}

	private int getSerialNum(final Point cell) {
		return cell.getX() + cell.getY() * widthOfBoard;
	}
	
	private Cell getCell(final Point cell) {
		return board[getSerialNum(cell)];
	}

	public void open(final Point cell) {
		if (!hasOpenedCell() && getCell(cell) instanceof CellWithMine) {
			rotateCellUntilNoMine(getSerialNum(cell));
			assert getCell(cell) instanceof CellWithoutMine;
		}
		
		getCell(cell).open();
		
		if (getCell(cell) instanceof CellWithMine) {
			openAll();
			return ;
		}
		openNeighborsAutomatically(cell);
	}
	
	private void openAll() {
		for (final Cell cell : board)
			cell.open();
	}

	private void openNeighborsAutomatically(final Point cell) {
		assert getCell(cell).isOpen();
		assert getCell(cell) instanceof CellWithoutMine;
		
		if (getNumOfNeighbors(getSerialNum(cell)) != 0)
			return ;
		
		final Set<Point> notOpenedNeighbors = getNotOpenedNeighbors(cell);
		for (final Point p : notOpenedNeighbors)
			getCell(p).open();
		for (final Point p : notOpenedNeighbors)
			openNeighborsAutomatically(p);
	}
	
	private Set<Point> getNotOpenedNeighbors(final Point cell) {
		return Utils.filterAsSet(cell.getAdjacentEight(), new Utils.Pred<Point>() {
			public boolean eval(final Point p) {
				return isInBoard(p) && (!getCell(p).isOpen());
			}
		});
	}

	private void rotateCellUntilNoMine(final int serial) {
		
		final int noMine = Utils.findFirstIndex(board, new Utils.Pred<Cell>() {
			public boolean eval(final Cell obj) {
				return obj instanceof CellWithoutMine;
			}
		});
		assert noMine != -1;
		
		board = Utils.rotate(board, serial - noMine);
	}


	public String getCellOutput(final Point cell) {
		final int numOfNeighbors = getNumOfNeighbors(getSerialNum(cell));
		return getCell(cell).asTextOutput(numOfNeighbors);
	}
	
	private int getNumOfNeighbors(final int serial) {
		final Point p = new Point(getX(serial), getY(serial));
		return Utils.count(p.getAdjacentEight(), new Utils.Pred<Point>() {
			public boolean eval(final Point obj) {
				return isMine(obj);
			}
		});
	}
	
	public boolean isInBoard(final Point cell) {
		return
			0 <= cell.getX() && cell.getX() < widthOfBoard &&
			0 <= cell.getY() && cell.getY() < getHeight();
	}
	
	private boolean isMine(final Point cell) {
		return isInBoard(cell) && getCell(cell) instanceof CellWithMine;
	}

	public boolean gameOver() {
		return hasOpenedMine();
	}
	
	private boolean hasOpenedMine() {
		return Utils.exists(board, new Utils.Pred<Cell>() {
			public boolean eval(final Cell obj) {
				return obj.isOpen() && obj instanceof CellWithMine;
			}
		});
	}

	public void toggleFlag(final Point cell) {
		getCell(cell).toggleFlag();
	}

	public boolean isFlagged(final Point cell) {
		return getCell(cell).isFlagged();
	}

	public int getNumOfRemainingFlags() {
		return getNumOfMines() - getNumOfFlaggedCells();
	}
	
	private int getNumOfFlaggedCells() {
		return Utils.count(board, new Utils.Pred<Cell>() {
			public boolean eval(final Cell cell) {
				return cell.isFlagged();
			}
		});
	}
	
	private int getNumOfMines() {
		return Utils.count(board, new Utils.Pred<Cell>() {
			public boolean eval(final Cell cell) {
				return cell instanceof CellWithMine;
			}
		});
	}

	private int getNumOfNotOpenedCells() {
		return Utils.count(board, new Utils.Pred<Cell>() {
			public boolean eval(final Cell cell) {
				return !cell.isOpen();
			}
		});
	}

	private boolean hasOpenedCell() {
		return Utils.exists(board, new Utils.Pred<Cell>() {
			public boolean eval(final Cell cell) {
				return cell.isOpen();
			}
		});
	}

	public boolean isCleared() {
		return
			!hasOpenedMine() &&
			getNumOfMines() == getNumOfNotOpenedCells();
	}

	public String getBoardAsString() {
		final StringBuilder out = new StringBuilder(100);
		
		out.append("XX|");
		for (int x = 0; x < widthOfBoard; x++)
			out.append(String.format("%2d", x));
		out.append('\n');
		
		out.append("--+");
		for (int x = 0; x < widthOfBoard; x++)
			out.append("--");
		out.append('\n');
		
		for (int y = 0; y < getHeight(); y++) {
			out.append(String.format("%2d|", y));
			
			for (int x = 0; x < widthOfBoard; x++)
				out.append(getCellOutput(new Point(x, y)));
			out.append('\n');
		}
		return out.toString();
	}

	public boolean isOpen(final Point cell) {
		return getCell(cell).isOpen();
	}

	private abstract class Cell {
		
		protected ICellState state;
		
		public Cell() {
			state = new NotFlaggedState();
		}

		public abstract String asTextOutput(int numOfNeighbors);

		public void open() {
			if (state instanceof NotOpenedState)
				state = new OpenedState();
		}
		
		public void toggleFlag() {
			if (state instanceof NotFlaggedState)
				state = new FlaggedState();
			else if (state instanceof FlaggedState)
				state = new NotFlaggedState();
		}
		
		public boolean isFlagged() {
			return state instanceof FlaggedState;
		}
		
		public boolean isOpen() {
			return state instanceof OpenedState;
		}
	}
	
	private class CellWithMine extends Cell {

		@Override
		public String asTextOutput(final int numOfNeighbors) {
			if (state instanceof NotOpenedState)
				return ((NotOpenedState)state).getTextOutput();
			return "＊";
		}
	}
	
	private class CellWithoutMine extends Cell {

		@Override
		public String asTextOutput(final int numOfNeighbors) {
			if (state instanceof NotOpenedState)
				return ((NotOpenedState)state).getTextOutput();
			return String.format("%2d", numOfNeighbors);
		}
	}
}
