/* Copyright 2007 Harai Akihiro.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *     http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sourceforge.jlogtest.sample.minesweeper;

import static java.lang.String.valueOf;
import static org.junit.Assert.assertTrue;

import java.io.IOException;

import jp.sourceforge.jlogtest.Logger;
import jp.sourceforge.jlogtest.LoggerFactory;
import jp.sourceforge.jlogtest.annotations.Log;
import jp.sourceforge.jlogtest.annotations.Sequence;
import jp.sourceforge.jlogtest.annotations.TargetLogs;
import jp.sourceforge.jlogtest.junit.JLogTestRunner;

import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;

@TargetLogs({
	@Sequence(name = "", log = { @Log("output"), @Log("input") }),
	@Sequence(name = "board", log = @Log("board"))
})
@RunWith(JLogTestRunner.class)
public class CommandLineInterfaceTest {
	
	private static final Logger output = LoggerFactory.getLogger("output");
	private static final Logger input = LoggerFactory.getLogger("input");
	private static final Logger board = LoggerFactory.getLogger("board");
	
	@Before
	public void setUp() throws Exception {
		JLogTestRunner.setLogDirectory("src/test/resources/jlogtest");
	}
	
	@Test
	public void 模範的なゲーム() throws Exception {
		final Point[] mineCells = {
				new Point(0, 0),
				new Point(1, 0),
				new Point(1, 1),
				new Point(1, 5),
				new Point(2, 4),
				new Point(2, 5),
				new Point(3, 0),
				new Point(3, 1),
				new Point(3, 6),
		};
		final Size boardSize = new Size(6, 7);
		// 標準入力からの一行ごとの入力では、最後の改行文字は含まれないらしい。
		// よって、こちらでも各入力の最後に改行文字を含める必要はない
		final String[] inputs = {
				valueOf(boardSize.getWidth()),
				valueOf(boardSize.getHeight()),
				valueOf(mineCells.length),
				"5 3 o",
				"3 0 f",
				"3 1 f",
				"3 6 f",
				"2 2 o",
				"2 1 o",
				"2 3 o",
				"2 4 f",
				"2 5 f",
				"1 3 o",
				"1 4 o",
				"0 3 o",
				"1 1 f",
				"0 1 o",
				"0 0 f",
				"1 0 f",
				"2 0 o",
				"0 1 o",
				"2 6 o",
				"1 5 f",
				"0 5 o",
				"0 6 o",
				"1 6 o",
		};
		execTest(mineCells, boardSize, inputs);
	}
	
	@Test
	public void 地雷を踏んでゲームオーバー() throws Exception {
		final Point[] mineCells = {
				new Point(0, 0),
				new Point(1, 0),
				new Point(1, 1),
				new Point(1, 5),
				new Point(2, 4),
				new Point(2, 5),
				new Point(3, 0),
				new Point(3, 1),
				new Point(3, 6),
		};
		final Size boardSize = new Size(6, 7);
		// 標準入力からの一行ごとの入力では、最後の改行文字は含まれないらしい。
		// よって、こちらでも各入力の最後に改行文字を含める必要はない
		final String[] inputs = {
				valueOf(boardSize.getWidth()),
				valueOf(boardSize.getHeight()),
				valueOf(mineCells.length),
				"0 1 o",
				"0 0 o",
		};
		execTest(mineCells, boardSize, inputs);
	}
	
	@Test
	public void クリック一回でクリア() throws Exception {
		final Point[] mineCells = {
				new Point(0, 0),
				new Point(0, 1),
				new Point(1, 0),
		};
		final Size boardSize = new Size(6, 7);
		// 標準入力からの一行ごとの入力では、最後の改行文字は含まれないらしい。
		// よって、こちらでも各入力の最後に改行文字を含める必要はない
		final String[] inputs = {
				valueOf(boardSize.getWidth()),
				valueOf(boardSize.getHeight()),
				valueOf(mineCells.length),
				"2 2 o",
		};
		execTest(mineCells, boardSize, inputs);
	}
	
	@Test
	public void 旗を立て間違えてゲームオーバー() throws Exception {
		final Point[] mineCells = {
				new Point(0, 0),
				new Point(0, 2),
		};
		final Size boardSize = new Size(6, 7);
		// 標準入力からの一行ごとの入力では、最後の改行文字は含まれないらしい。
		// よって、こちらでも各入力の最後に改行文字を含める必要はない
		final String[] inputs = {
				valueOf(boardSize.getWidth()),
				valueOf(boardSize.getHeight()),
				valueOf(mineCells.length),
				"1 1 o",
				"0 1 f",
				"0 0 o",
		};
		execTest(mineCells, boardSize, inputs);
	}
	
	private void execTest(
			final Point[] mineCells, final Size boardSize, final String[] inputs) {

		final MockInput input = new MockInput(inputs);
		final MockPermutator permutator =
				MockPermutator.createPermutator(boardSize, mineCells);
		board.out(permutator.getBoard());
		final CommandLineInterface ms =
				new CommandLineInterface(input, new MockOutput(), permutator);
		ms.mainLoop();
		assertTrue(input.verify());
	}
	
	private class MockInput implements IInput {
		private final String[] inputStr;
		private int index;
		
		private MockInput(final String[] inputs) {
			inputStr = inputs;
			index = 0;
		}
		
		public String getLine() throws IOException {
			input.out(inputStr[index]);
			return inputStr[index++];
		}
		
		private boolean verify() {
			return inputStr.length == index;
		}
	}
	
	private class MockOutput implements IOutput {

		public void out(final String out) {
			output.out(out);
		}

		public void outln(final String out) {
			output.out(out + '\n');
		}
	}
}
