/* Copyright 2007 Harai Akihiro.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *     http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sourceforge.jlogtest.sample.minesweeper;

import java.util.Arrays;

import jp.sourceforge.jlogtest.sample.minesweeper.IRandomPermutator;
import jp.sourceforge.jlogtest.sample.minesweeper.Point;
import jp.sourceforge.jlogtest.sample.minesweeper.Size;
import jp.sourceforge.jlogtest.sample.minesweeper.Utils;

public class MockPermutator implements IRandomPermutator {
	
	private final boolean[] mines;
	private final Size size;

	public MockPermutator(final Size size, final boolean[] mines) {
		assert size.getCellNum() == mines.length;
		
		this.size = size;
		this.mines = mines;
	}
	
	public static MockPermutator createMineBasedPermutator(
			final Size size, final Point... excluded) {

		final boolean[] mines = new boolean[size.getCellNum()];
		
		// デフォルト値
		Arrays.fill(mines, true);
		
		for (final Point p : excluded)
			mines[size.getSerializedPos(p)] = false;

		return new MockPermutator(size, mines);
	}

	public static MockPermutator createPermutator(
			final Size size, final Point... minePos) {

		final boolean[] mines = new boolean[size.getCellNum()];
		
		// デフォルト値
		Arrays.fill(mines, false);
		
		for (final Point p : minePos)
			mines[size.getSerializedPos(p)] = true;

		return new MockPermutator(size, mines);
	}

	public boolean[] getRandomizedMineMap(final int numOfCells, final int numOfMines) {
		assert numOfCells == mines.length;
		assert getNumOfMines() == numOfMines;
		
		return mines; 
	}
	
	private int getNumOfMines() {
		return Utils.count(mines, new Utils.Pred<Boolean>() {
			public boolean eval(final Boolean obj) {
				return obj;
			}
		});
	}
	
	public String getBoard() {
		final StringBuilder output = new StringBuilder(100); 
		for (int i = 0; i < mines.length; i++) {
			output.append(mines[i] ? '＊' : '□');
			
			if ((i + 1) % size.getWidth() == 0)
				output.append('\n');
		}
		return output.toString();
	}
}
