/*
** Job Arranger for ZABBIX
** Copyright (C) 2012 FitechForce, Inc. All Rights Reserved.
** Copyright (C) 2013 Daiwa Institute of Research Business Innovation Ltd. All Rights Reserved.
** Copyright (C) 2021 Daiwa Institute of Research Ltd. All Rights Reserved.
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
**/


#include "common.h"
#include "log.h"

#if defined(ZABBIX_SERVICE)
#include "service.h"
#elif defined(ZABBIX_DAEMON)
#include "daemon.h"
#endif
#include "jacommon.h"
#include "jaagent.h"
#include "jadbbackup.h"
#include "jafile.h"
#include "jajobfile.h"
#include "jajournal.h"
#ifdef _WINDOWS
#include <dirent.h>
#else
#include <time.h>
#endif


/******************************************************************************
 *                                                                            *
 * Function:                                                                  *
 *                                                                            *
 * Purpose:                                                                   *
 *                                                                            *
 * Parameters:                                                                *
 *                                                                            *
 * Return value:                                                              *
 *                                                                            *
 * Comments:                                                                  *
 *                                                                            *
 ******************************************************************************/
ZBX_THREAD_ENTRY(jadbbackup_thread, args)
{
    int local_request_failed = 0;
    struct tm	*tm;
    time_t		now;
	struct tm* log_tm;
	time_t		log_now;
    int backup_flag=0;
	int write_log = 1;
	int thread_number = 0;
	int end_sleep = 10;
	int remainder = 0;


    assert(args);

    zabbix_log(LOG_LEVEL_INFORMATION,
               "jobarg_agentd #%d started [jabackup]",
               ((zbx_thread_args_t *) args)->thread_num);
	thread_number = ((zbx_thread_args_t*)args)->thread_num;
    zbx_free(args);
    while (ZBX_IS_RUNNING()) {
        zbx_setproctitle("process jabackup_thread");
		time(&log_now);
		log_tm = localtime(&log_now);
		if (0 == log_tm->tm_hour || 4 == log_tm->tm_hour || 8 == log_tm->tm_hour || 12 == log_tm->tm_hour || 16 == log_tm->tm_hour || 20 == log_tm->tm_hour) {
			if (write_log == 0) {
				zabbix_log(LOG_LEVEL_INFORMATION, "jobarg_agentd #%d running [jabackup]", thread_number);
				write_log = 1;
			}
		}
		else {
			write_log = 0;
		}
        if(CONFIG_BACKUP_RUN_TIME == 24){
			remainder = time(NULL) % ((size_t)CONFIG_BACKUP_TIME * 3600);
        	if (remainder <= end_sleep) {
        		zabbix_log(LOG_LEVEL_INFORMATION, "jabackup CONFIG_BACKUP_TIME[%d] time[%d]",CONFIG_BACKUP_TIME, remainder);
        		zabbix_log(LOG_LEVEL_INFORMATION, "jabackup  CONFIG_BACKUP_TIME START BACKUP");
        		ja_datafile_remove_oldjob(CONFIG_JOB_HISTORY);
        		zabbix_log(LOG_LEVEL_INFORMATION, "jabackup  CONFIG_BACKUP_TIME END BACKUP");
        	}
        }else{
        	time(&now);
        	tm = localtime(&now);
        	if(tm->tm_min == 0  && CONFIG_BACKUP_RUN_TIME == tm->tm_hour && backup_flag == 0 ){
        		zabbix_log(LOG_LEVEL_INFORMATION, "jabackup CONFIG_BACKUP_RUN_TIME [%d] current H[%d]  backup_flag[%d]",CONFIG_BACKUP_RUN_TIME,tm->tm_hour,backup_flag);
        		backup_flag = 1;
        		zabbix_log(LOG_LEVEL_INFORMATION, "jabackup CONFIG_BACKUP_RUN_TIME START BACKUP");
        		ja_datafile_remove_oldjob(CONFIG_JOB_HISTORY);
        		zabbix_log(LOG_LEVEL_INFORMATION, "jabackup CONFIG_BACKUP_RUN_TIME END BACKUP");
        	}else if(backup_flag == 1 && CONFIG_BACKUP_RUN_TIME != tm->tm_hour){
        		backup_flag = 0;
        	}
        }
        zbx_sleep(end_sleep);
    }

  error:
    zabbix_log(LOG_LEVEL_INFORMATION, "jobarg_agentd jabackup stopped");
    ZBX_DO_EXIT();
    zbx_thread_exit(0);
}
int ja_datafile_remove_oldjob(int day)
{	
	const int total_data_file_count = 7;
    int i, j, ii, jj, deleteFolder;
	struct stat      stat_buf;
	time_t history;
	char close_file[JA_FILE_PATH_LEN];
	char close_filepath[JA_MAX_STRING_LEN];
	char archive_folder[JA_FILE_PATH_LEN];
	char archive_folder_name[JA_FILE_PATH_LEN];
	char archive_file[JA_FILE_PATH_LEN];
	char del_file[JA_FILE_PATH_LEN];
	char data_file[JA_FILE_PATH_LEN];
	DIR *close_dir;
	struct dirent *entry;
	int wait_count;
	int wait_max = 3;

    const char *__function_name = "ja_jobmain_remove_oldjob";



    zabbix_log(LOG_LEVEL_DEBUG, "In %s() day: %d", __function_name, day);
    history = time(NULL) - ((time_t)day * 24 * 60 * 60);
	zabbix_log(LOG_LEVEL_DEBUG, "In %s  history = %ld ", __function_name, history);
	zbx_snprintf(close_filepath, sizeof(close_filepath), "%s%cclose", CONFIG_TMPDIR, JA_DLM);
	close_dir = opendir(close_filepath);
	if (NULL == close_dir)
	{
		zabbix_log(LOG_LEVEL_ERR, "In %s(), Folder cannot be read.", __function_name);
		return FAIL;
	}
	wait_count = 1;
	while ((entry = readdir(close_dir)) != NULL)
	{
		
		if (NULL != strstr(entry->d_name, ".job")) {
			zbx_snprintf(close_file, sizeof(close_file), "%s%c%s", close_filepath, JA_DLM, entry->d_name);
			if (stat(close_file, & stat_buf) == 0) {
				zabbix_log(LOG_LEVEL_DEBUG, "In %s file[%s] modify time[%ld]", __function_name, close_file, stat_buf.st_mtime);
				//check history time.
				if ((history - stat_buf.st_mtime) > 0 ) {

					zbx_snprintf(archive_file, sizeof(archive_file), "%s%c%s", close_filepath, JA_DLM, entry->d_name);
					zbx_snprintf(archive_folder_name, strlen(entry->d_name) - 3, "%s", entry->d_name);
					zbx_snprintf(archive_folder, sizeof(archive_folder), "%s%c%s", close_filepath, JA_DLM, archive_folder_name);

					//check if sub folder exists in close folder;
					if (get_file_count(archive_folder) < total_data_file_count) {
						zabbix_log(LOG_LEVEL_DEBUG, "In %s()  some files are missing or sub folder is not under close folder.Moving to error folder.", __function_name);
						//move files from data and close/sub-folder to error folder.
						if (job_to_error(close_filepath, archive_folder_name) == FAIL) {
							zabbix_log(LOG_LEVEL_ERR, "In %s() [%s] data folder file delete failed. (%s)", __function_name, entry->d_name, strerror(errno));
						}
						continue;
					}

					//execute each files using extensions.
					i = 0;
					while (FILE_EXT[i] != NULL) {
						jj = 0;
						if ( 0 == i) {
							for (ii = 0; ii < strlen(archive_folder_name); ii++) {
								data_file[ii] = archive_folder_name[ii];
								if (archive_folder_name[ii] == '-') {
									jj++;
									if (jj > 1) {
										break;
									}
								}
							}
							data_file[ii] = '\0';
						}
						else {
							zbx_snprintf(data_file, sizeof(data_file), "%s", archive_folder_name);
						}
						//delete files under /data folder,/close/sub-folder folder with same .job name in the close folder.	
						zbx_snprintf(del_file, sizeof(del_file), "%s%c%s.%s", archive_folder, JA_DLM, data_file, FILE_EXT[i]);
						zabbix_log(LOG_LEVEL_DEBUG, "In %s()  deletefile is : [%s]", __function_name, del_file);
						if (0 == remove(del_file)) {
							zabbix_log(LOG_LEVEL_DEBUG, "In %s() [%s] close file deleted.", __function_name, del_file);
						}
						else {
							zabbix_log(LOG_LEVEL_ERR, "In %s() [%s] file close delete failed. (%s)", __function_name, del_file, strerror(errno));
							//delete data file under /data folder if close/sub-folder data files cannot be removed.
							zbx_snprintf(del_file, sizeof(del_file), "%s%cdata%c%s.%s", CONFIG_TMPDIR, JA_DLM, JA_DLM, data_file, FILE_EXT[i]);
							if (0 == remove(del_file)) {
								zabbix_log(LOG_LEVEL_WARNING, "In %s() [%s] data folder file deleted.", __function_name, del_file);
							}
							else {
								//file does not exist or other error under /data folder.
								zabbix_log(LOG_LEVEL_ERR, "In %s() [%s] data folder file delete failed. (%s)", __function_name, del_file, strerror(errno));
								continue;
							}
						}
						i++;
					}
					if (wait_count >= wait_max) {					
						zbx_sleep(1);
						wait_count = 0;
					}
					else {
						wait_count++;
					}
					j = 0;
					deleteFolder = rmdir(archive_folder);
					if (0 == deleteFolder) {
						zabbix_log(LOG_LEVEL_INFORMATION, "In %s() [%s] deleted folder.", __function_name, archive_folder_name);
						j = 1;
					}
					else {
						if (stat(archive_folder, &stat_buf) == 0) {
							zabbix_log(LOG_LEVEL_ERR, "In %s() [%s] folder, delete failed. (%s)", __function_name, archive_folder, strerror(errno));
							continue;
						}
						else {
							zabbix_log(LOG_LEVEL_WARNING, "In %s() [%s] This folder does not exist. (%s)", __function_name, archive_folder, strerror(errno));
							j = 1;
						}
					}
					if (1 == j && 0 == remove(close_file)) {
						zabbix_log(LOG_LEVEL_DEBUG, "In %s() deleted [%s] job file.", __function_name, close_file);
					}
					else {
						zabbix_log(LOG_LEVEL_ERR, "In %s() [%s] job file, delete failed. (%s)", __function_name, close_file, strerror(errno));
					}
				}
			}
			else {
				zabbix_log(LOG_LEVEL_ERR, "In %s(), file cannot be read.[%s]", __function_name, close_file);
			}
		}
	}
	closedir(close_dir);

    return SUCCEED;
}