/*
** Job Arranger for ZABBIX
** Copyright (C) 2012 FitechForce, Inc. All Rights Reserved.
** Copyright (C) 2013 Daiwa Institute of Research Business Innovation Ltd. All Rights Reserved.
** Copyright (C) 2021 Daiwa Institute of Research Ltd. All Rights Reserved.
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
**/


#include "common.h"
#include "comms.h"
#include "log.h"

#if defined(ZABBIX_SERVICE)
#include "service.h"
#elif defined(ZABBIX_DAEMON)
#include "daemon.h"
#endif

#include "jacommon.h"
#include "jajobobject.h"
#include "jatcp.h"
#include "jatelegram.h"
#include "jaagent.h"
#include "jafcopy.h"
#include "listener.h"
#include "jafile.h"
#include "jajobfile.h"
#include "errno.h"	//for error number fwrite/fread
#include "time.h" 	//get current local time
#include "jastr.h"

/******************************************************************************
 *                                                                            *
 * Function:                                                                  *
 *                                                                            *
 * Purpose:                                                                   *
 *                                                                            *
 * Parameters:                                                                *
 *                                                                            *
 * Return value:                                                              *
 *                                                                            *
 * Comments:                                                                  *
 *                                                                            *
 ******************************************************************************/
static int process_listener(zbx_sock_t * s,ja_job_object * job, ja_job_object * temp_job)
{
    int ret;
    char socket_ip[JA_SERVERID_LEN];
    char data_filename[JA_FILE_PATH_LEN];
    char full_datafile_path[JA_FILE_PATH_LEN];
    //Added by ThihaOo@DAT 11/01/2021
    char current_time[20];
    char data_file[JA_FILE_PATH_LEN];
    char jobid_datetime[JA_FILE_PATH_LEN];
    char filename[JA_FILE_PATH_LEN];
    int proc_run_flag = 0;
    struct tm *tm;
    const char *__function_name = "process_listener";

    zabbix_log(LOG_LEVEL_DEBUG, "In %s()", __function_name);
	#ifdef _WINDOWS
		struct _timeb		currentTime;
		_ftime(&currentTime);
		tm = localtime(&currentTime.time);
	#else
		struct timeval		currentTime;
		gettimeofday(&currentTime,NULL);
		tm = localtime(&currentTime.tv_sec);
	#endif
		zbx_snprintf(current_time, 20, "%.4d%.2d%.2d%.2d%.2d%.2d",tm->tm_year + 1900,tm->tm_mon + 1,tm->tm_mday,tm->tm_hour,tm->tm_min,tm->tm_sec);
	zabbix_log(LOG_LEVEL_DEBUG, "In %s() current Datetime is %s.", __function_name,current_time);
	//End
    socket_ip[0]='\0';
    if (ja_tcp_check_security(s, CONFIG_HOSTS_ALLOWED, 0, socket_ip) == FAIL) {
        zbx_tcp_unaccept(s);
        zabbix_log(LOG_LEVEL_ERR, "In %s() ja_tcp_check_security() check failed.", __function_name);
        return FAIL;
    }
    zabbix_log(LOG_LEVEL_DEBUG, "In %s() socket_ip=[%s]", __function_name ,socket_ip);
    if (FAIL == ja_job_object_init(job)) {
        zabbix_log(LOG_LEVEL_ERR, "In %s() job object cannot be initialized. job is Null.", __function_name);
        return FAIL;
    }
    else {
        ret = ja_tcp_recv_to(s, job, CONFIG_TIMEOUT);
        zabbix_log(LOG_LEVEL_DEBUG, "In %s() job object accepted.", __function_name);
        job->result = JA_RESPONSE_SUCCEED;
    }

    zbx_snprintf(job->serverip, sizeof(job->serverip), "%s",socket_ip);

    if (strcmp(job->kind, JA_PROTO_VALUE_JOBRUN) == 0) {
        zbx_snprintf(job->kind, sizeof(job->kind), "%s",
                     JA_PROTO_VALUE_JOBRUN_RES);
        if (ret == SUCCEED) {
        	//Added by ThihaOo@DAT 11/01/2021
        	if(job->method !=JA_AGENT_METHOD_KILL){
                //check for job ID and server ID.
                zbx_snprintf(jobid_datetime, sizeof(jobid_datetime), ZBX_FS_UI64"-%s", job->jobid, current_time);
                proc_run_flag = ja_jobfile_check_processexist(filename,jobid_datetime);// 0: No running process; 1: Running process
                //if the same jobid from same server is running, 
                if (proc_run_flag == 1) {
                    //check for server
                    if (FAIL == ja_job_object_init(temp_job)) {
                        zabbix_log(LOG_LEVEL_ERR, "In %s() job object cannot be initialized. job is Null.", __function_name);
                        return FAIL;
                    }
                    int datafile_read_flg = SUCCEED;
                    get_jobid_datetime(filename, data_filename);
                    zbx_snprintf(full_datafile_path, sizeof(full_datafile_path), "%s%cdata%c%s.json", CONFIG_TMPDIR, JA_DLM,JA_DLM,data_filename);
                    datafile_read_flg = read_datafile(temp_job, full_datafile_path);
                    if (datafile_read_flg != FAIL && temp_job != NULL && strncmp(job->serverip,temp_job->serverip,sizeof(job->serverip)) != 0) {
                        zbx_snprintf(job->message, sizeof(job->message), "Job with same job id is already running.");
                        if (job->result == JA_RESPONSE_SUCCEED)
                            job->result = JA_RESPONSE_FAIL;
                        goto contd;
                    }
                    else if (datafile_read_flg == FAIL || temp_job == NULL) {
                        zabbix_log(LOG_LEVEL_DEBUG, "In %s(), cannot read data file to compare server ip.",__function_name);
                    }
                    else {
                        zabbix_log(LOG_LEVEL_DEBUG, "In %s(), server ip does not match.",__function_name);
                    }
                }
            	if(ja_write_file_data(job,&current_time)==FAIL){
            		zabbix_log(LOG_LEVEL_ERR, "In %s() Data file write failed.", __function_name );
                    if (job->result == JA_RESPONSE_SUCCEED)
                        job->result = JA_RESPONSE_FAIL;
                    goto contd;
            	}
        	}
        	//End
            zbx_snprintf(data_file, sizeof(data_file), "%s%cdata%c" ZBX_FS_UI64 "-%s.json", CONFIG_TMPDIR, JA_DLM, JA_DLM, job->jobid, current_time);
            zabbix_log(LOG_LEVEL_DEBUG, "In %s() job type is : %s and data file is : %s", __function_name,job->type,data_file);
            if (ja_agent_begin(job,&current_time, & data_file) != SUCCEED) {
                zabbix_log(LOG_LEVEL_ERR, "In %s() job id :"ZBX_FS_UI64" job agent begin failed.", __function_name, job->jobid);
                    job->result = JA_RESPONSE_FAIL;
            }
        } else {
            zabbix_log(LOG_LEVEL_ERR, "In %s() tcp revcieve to Failed.", __function_name);
                job->result = JA_RESPONSE_FAIL;
        }
    } else if (strcmp(job->kind, JA_PROTO_VALUE_FCOPY) == 0) {
        zbx_snprintf(job->kind, sizeof(job->kind), "%s",
                     JA_PROTO_VALUE_FCOPY_RES);
        if (ret == SUCCEED) {
            ja_fcopy_begin(job, s);
            return SUCCEED;
        } else {
            if (job->result == JA_RESPONSE_SUCCEED)
                job->result = JA_RESPONSE_FAIL;
        }
    } else {
        zbx_snprintf(job->kind, sizeof(job->kind), "%s",
                     JA_PROTO_VALUE_JOBRUN_RES);
        job->version = JA_PROTO_TELE_VERSION;
        job->result = JA_RESPONSE_FAIL;
    }
    ret = SUCCEED;
contd:
    ja_tcp_send_to(s, job, CONFIG_TIMEOUT);
    zbx_tcp_unaccept(s);
    return ret;
}

/******************************************************************************
 *                                                                            *
 * Function:                                                                  *
 *                                                                            *
 * Purpose:                                                                   *
 *                                                                            *
 * Parameters:                                                                *
 *                                                                            *
 * Return value:                                                              *
 *                                                                            *
 * Comments:                                                                  *
 *                                                                            *
 ******************************************************************************/
ZBX_THREAD_ENTRY(listener_thread, args)
{
    int ret, local_request_failed = 0, write_log = 1, thread_number = 0;
    zbx_sock_t sock;
    struct tm* tm;
    time_t		now;
    ja_job_object* job;
    ja_job_object* temp_job;

    assert(args);
    assert(((zbx_thread_args_t *) args)->args);

    zabbix_log(LOG_LEVEL_INFORMATION,
               "jobarg_agentd #%d started [listener]",
               ((zbx_thread_args_t *) args)->thread_num);
    thread_number = ((zbx_thread_args_t*)args)->thread_num;
    memcpy(&sock, (zbx_sock_t *) ((zbx_thread_args_t *) args)->args,
           sizeof(zbx_sock_t));
    zbx_free(args);
    job = (ja_job_object*)zbx_malloc(NULL, sizeof(ja_job_object));
    temp_job = (ja_job_object*)zbx_malloc(NULL, sizeof(ja_job_object));
    while (ZBX_IS_RUNNING()) {
        zbx_setproctitle("listener [waiting for connection]");
        time(&now);
        tm = localtime(&now);
        if (0 == tm->tm_hour || 4 == tm->tm_hour || 8 == tm->tm_hour || 12 == tm->tm_hour || 16 == tm->tm_hour || 20 == tm->tm_hour) {
            if (write_log == 0) {
                zabbix_log(LOG_LEVEL_INFORMATION, "jobarg_agentd #%d running [listener]", thread_number);
                write_log = 1;
            }
        }
        else {
            write_log = 0;
        }
        if (SUCCEED == (ret = ja_tcp_accept(&sock))
            && sock.socket != ZBX_SOCK_ERROR) {
            local_request_failed = 0;
            zbx_setproctitle("listener [processing request]");
            ret = process_listener(&sock,job,temp_job);

        }
        if (SUCCEED == ret)
            continue;

        zabbix_log(LOG_LEVEL_WARNING, "Listener error: %s",
                   zbx_tcp_strerror());

        if (local_request_failed++ > 1000) {
            zabbix_log(LOG_LEVEL_WARNING,
                       "Too many consecutive errors on accept() call.");
            local_request_failed = 0;
        }
        if (ZBX_IS_RUNNING())
            zbx_sleep(1);
    }

    zabbix_log(LOG_LEVEL_INFORMATION, "jobarg_agentd listener stopped");
    zbx_free(job);
    zbx_free(temp_job);
    ZBX_DO_EXIT();
    zbx_thread_exit(0);
}
