package org.asyrinx.joey.gen.task;

/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *    "Apache Turbine" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache",
 *    "Apache Turbine", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.DirectoryScanner;
import org.apache.tools.ant.types.FileSet;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.context.Context;
import org.apache.velocity.texen.ant.TexenTask;
import org.asyrinx.brownie.core.lang.ClassUtils;
import org.asyrinx.brownie.core.lang.StringUtils;
import org.asyrinx.joey.gen.command.rdb2java.standard.Rdb2JavaBuilder;
import org.asyrinx.joey.gen.hibernate.HibernateUtils;
import org.asyrinx.joey.gen.model.java.AppDomain;
import org.asyrinx.joey.gen.model.rdb.Databases;
import org.asyrinx.joey.gen.model.rdb.xml.XmlToRdb;
import org.xml.sax.SAXException;

/**
 * A base torque task that uses either a single XML schema representing a data
 * model, or a &lt;fileset&gt; of XML schemas. We are making the assumption that
 * an XML schema representing a data model contains tables for a <strong>single
 * </strong> database.
 * 
 * @author <a href="mailto:jvanzyl@zenplex.com">Jason van Zyl </a>
 * @author <a href="mailto:dlr@finemaltcoding.com">Daniel Rall </a>
 */
public class DataModelTask extends TexenTask {
    /**
     * XML that describes the database model, this is transformed into the
     * application model object.
     */
    protected String xmlFile;

    /**
     * Fileset of XML schemas which represent our data models.
     */
    protected List filesets = new ArrayList();

    /**
     * Velocity context which exposes our objects in the templates.
     */
    protected Context context;

    /**
     * Get the xml schema describing the application model.
     * 
     * @return String xml schema file.
     */
    public String getXmlFile() {
        return xmlFile;
    }

    /**
     * Set the xml schema describing the application model.
     * 
     * @param xmlFile
     *            The new XmlFile value
     */
    public void setXmlFile(String xmlFile) {
        this.xmlFile = project.resolveFile(xmlFile).toString();
    }

    /**
     * Adds a set of xml schema files (nested fileset attribute).
     * 
     * @param set
     *            a Set of xml schema files
     */
    public void addFileset(FileSet set) {
        filesets.add(set);
    }

    /**
     * Set up the initial context for generating the SQL from the XML schema.
     * 
     * @return the context
     * @throws Exception
     */
    public Context initControlContext() throws Exception {
        return loadModels();
    }

    /**
     * @return
     * @throws SAXException
     * @throws IOException
     * @throws InstantiationException
     */
    private Context loadModels() throws IOException, SAXException, InstantiationException {
        if (xmlFile == null && filesets.isEmpty())
            throw new BuildException("You must specify an XML schema or " + "fileset of XML schemas!");
        final Databases databases = new Databases();
        if (xmlFile != null) {
            // Transform the XML database schema into
            // data model object.
            databases.getDatabases().add(loadModelXmlFile(xmlFile));
        } else {
            // Deal with the filesets.
            for (int i = 0; i < filesets.size(); i++) {
                FileSet fs = (FileSet) filesets.get(i);
                DirectoryScanner ds = fs.getDirectoryScanner(project);
                File srcDir = fs.getDir(project);
                String[] dataModelFiles = ds.getIncludedFiles();
                // Make a transaction for each file
                for (int j = 0; j < dataModelFiles.length; j++) {
                    final File f = new File(srcDir, dataModelFiles[j]);
                    final Databases loaded = loadModelXmlFile(f.toString());
                    databases.appendDatabases(loaded);
                }
            }
        }
        //Java֌W̃IuWFNg𐶐
        final Rdb2JavaBuilder builder = newBuilder();
        final AppDomain domain = builder.execute(databases);
        //
        context = new VelocityContext();
        // Place our set of data models into the context along
        // with the names of the databases as a convenience for now.
        context.put("databases", databases);
        context.put("domain", domain);
        context.put("builder", builder);
        context.put("helper", new VelocityHelper());
        context.put("stringUtils", new org.asyrinx.brownie.core.lang.StringUtils());
        context.put("hibernateUtils", new HibernateUtils());

        return context;
    }

    protected Databases loadModelXmlFile(String filename) throws IOException, SAXException {
        final XmlToRdb xmlToRdb = new XmlToRdb();
        xmlToRdb.setDebug("true".equals(getContextProperties().getString("schemaDebug")));
        return xmlToRdb.load(filename);
    }

    /**
     * Override Texen's context properties to map the torque.xxx properties
     * (including defaults set by the org/apache/torque/defaults.properties) to
     * just xxx.
     * 
     * <p>
     * Also, move xxx.yyy properties to xxxYyy as Velocity doesn't like the
     * xxx.yyy syntax.
     * </p>
     * 
     * @param file
     *            the file to read the properties from
     */
    public void setContextProperties(String file) {
        super.setContextProperties(file);
        // Map the torque.xxx elements from the env to the contextProperties
        Hashtable env = super.getProject().getProperties();
        for (Iterator i = env.keySet().iterator(); i.hasNext();) {
            String key = (String) i.next();
            if (key.startsWith("joey-gen.")) {
                String newKey = toVelocityKey(key.substring("joey-gen.".length()));
                contextProperties.setProperty(newKey, env.get(key));
                System.out.println("joey-gen property available: " + newKey + ":" + env.get(key));
            }
        }
        for (Iterator i = env.keySet().iterator(); i.hasNext();) {
            String key = (String) i.next();
            if (key.startsWith("proj.")) {
                String newKey = toVelocityKey(key);
                contextProperties.setProperty(newKey, env.get(key));
                System.out.println("project property available: " + newKey + ":" + env.get(key));
            }
        }
    }

    /**
     * @param newKey
     * @return
     */
    private String toVelocityKey(String newKey) {
        int j = newKey.indexOf(".");
        while (j != -1) {
            newKey = newKey.substring(0, j) + StringUtils.capitalize(newKey.substring(j + 1));
            j = newKey.indexOf(".");
        }
        return newKey;
    }

    private String builderClassName = "org.asyrinx.joey.gen.command.rdb2java.standard.BasicBuilder"; //BasicBuilder.class.getName();

    /**
     * @return Returns the builderClassName.
     */
    public String getBuilderClassName() {
        return builderClassName;
    }

    /**
     * @param builderClassName
     *            The builderClassName to set.
     */
    public void setBuilderClassName(String builderClassName) {
        this.builderClassName = builderClassName;
    }

    protected Rdb2JavaBuilder newBuilder() throws InstantiationException {
        return (Rdb2JavaBuilder) ClassUtils.newObject(getBuilderClassName(), Rdb2JavaBuilder.class);
    }

}