/*
 * Created on 2004/05/12
 */
package org.asyrinx.brownie.swing;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;

import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.plaf.BorderUIResource;

import org.asyrinx.brownie.core.util.DateUtils;

public final class DatePicker extends JPanel {
	/** Small font. */
	private static final Font smallFont = new Font("Dialog", Font.PLAIN, 10);

	/** Large font. */
	private static final Font largeFont = new Font("Dialog", Font.PLAIN, 12);

	/** Highlighted color. */
	private static final Color highlight = new Color(255, 255, 204);

	/** Enabled color. */
	private static final Color white = new Color(255, 255, 255);

	/** Disabled color. */
	private static final Color gray = new Color(204, 204, 204);

	/** Most recently selected day component. */
	private Component selectedDay = null;

	/** Currently selected date. */
	private GregorianCalendar selectedDate = null;

	/** Tracks the original date set when the component was created. */
	private GregorianCalendar originalDate = null;

	/**
	 * When true, the panel will be hidden as soon as a day is selected by
	 * clicking the day or clicking the Today button.
	 */
	private boolean hideOnSelect = true;

	/**
	 * When clicked, displays the previous year.
	 */
	private final JButton yearBackButton = new JButton();

	/**
	 * When clicked, displays the previous month.
	 */
	private final JButton monthBackButton = new JButton();

	/**
	 * Displays the currently selected month and year.
	 */
	private final JLabel monthAndYear = new JLabel();

	/**
	 * When clicked, displays the next month.
	 */
	private final JButton monthForwardButton = new JButton();

	/**
	 * When clicked, displays the next year.
	 */
	private final JButton yearForwardButton = new JButton();

	/**
	 * currently selected time field.
	 */
	private final JTextField selectedTimeField = new JTextField();

	/**
	 * Column headings for the days of the week.
	 */
	private final JTextField[] dayHeadings = new JTextField[] {
			new JTextField("S"), new JTextField("M"), new JTextField("T"),
			new JTextField("W"), new JTextField("T"), new JTextField("F"),
			new JTextField("S") };

	/**
	 * 2-dimensional array for 6 weeks of 7 days each.
	 */
	private final JTextField[][] daysInMonth = initDayTextFields();

	private JTextField[][] initDayTextFields() {
		final JTextField[][] result = new JTextField[6][7];
		for (int w = 0; w < result.length; w++) {
			for (int d = 0; d < result[w].length; d++) {
				result[w][d] = new JTextField();
			}
		}
		return result;
	}

	/**
	 * When clicked, sets the selected day to the current date.
	 */
	private final JButton todayButton = new JButton();

	/**
	 * When clicked, hides the calendar and sets the selected date to "empty".
	 */
	private final JButton cancelButton = new JButton();

	private DateFormat titleDateFormat = new SimpleDateFormat("yyyy/MM");

	private DateFormat timeFormat = new SimpleDateFormat("HH:mm:ss");

	/**
	 * Default constructor that sets the currently selected date to the current
	 * date.
	 */
	public DatePicker() {
		super();
		selectedDate = getToday();
		initialize();
	}

	/**
	 * Alternate constructor that sets the currently selected date to the
	 * specified date if non-null.
	 * 
	 * @param initialDate
	 */
	public DatePicker(final Date initialDate) {
		super();
		if (null == initialDate)
			selectedDate = getToday();
		else
			(selectedDate = new GregorianCalendar()).setTime(initialDate);
		originalDate = new GregorianCalendar(selectedDate.get(Calendar.YEAR),
				selectedDate.get(Calendar.MONTH), selectedDate
						.get(Calendar.DATE));
		initialize();
	}

	/**
	 * This method initializes this
	 * 
	 * @return void
	 */
	private void initialize() {
		final Insets insets = new Insets(2, 2, 2, 2);
		//
		setLayout(new AbsoluteLayout());
		this.setMinimumSize(new Dimension(210, 240));
		this.setMaximumSize(getMinimumSize());
		this.setPreferredSize(getMinimumSize());
		this.setBorder(new BorderUIResource.EtchedBorderUIResource());
		//
		yearBackButton.setFont(smallFont);
		yearBackButton.setText("<<");
		yearBackButton.setMargin(insets);
		yearBackButton.setDefaultCapable(false);
		yearBackButton.addActionListener(new ActionListener() {
			public void actionPerformed(final ActionEvent evt) {
				onYearBackClicked(evt);
			}
		});
		add(yearBackButton, new AbsoluteConstraints(10, 10, 30, 20));
		//
		monthBackButton.setFont(smallFont);
		monthBackButton.setText("<");
		monthBackButton.setMargin(insets);
		monthBackButton.setDefaultCapable(false);
		monthBackButton.addActionListener(new ActionListener() {
			public void actionPerformed(final ActionEvent evt) {
				onMonthBackClicked(evt);
			}
		});
		add(monthBackButton, new AbsoluteConstraints(40, 10, 20, 20));
		//
		monthAndYear.setFont(largeFont);
		monthAndYear.setHorizontalAlignment(JTextField.CENTER);
		monthAndYear.setText(titleDateFormat.format(selectedDate.getTime()));
		add(monthAndYear, new AbsoluteConstraints(50, 10, 100, 20));
		//
		monthForwardButton.setFont(smallFont);
		monthForwardButton.setText(">");
		monthForwardButton.setMargin(insets);
		monthForwardButton.setDefaultCapable(false);
		monthForwardButton.addActionListener(new ActionListener() {
			public void actionPerformed(final ActionEvent evt) {
				onMonthForwardClicked(evt);
			}
		});
		add(monthForwardButton, new AbsoluteConstraints(150, 10, 20, 20));
		//
		yearForwardButton.setFont(smallFont);
		yearForwardButton.setText(">>");
		yearForwardButton.setMargin(insets);
		yearForwardButton.setDefaultCapable(false);
		yearForwardButton.addActionListener(new ActionListener() {
			public void actionPerformed(final ActionEvent evt) {
				onYearForwardClicked(evt);
			}
		});
		add(yearForwardButton, new AbsoluteConstraints(170, 10, 30, 20));
		// layout the column headings for the days of the week
		final int startX = 35;
		final int startY = 60;
		int x = startX;
		for (int ii = 0; ii < dayHeadings.length; ii++) {
			dayHeadings[ii].setBackground(gray);
			dayHeadings[ii].setEditable(false);
			dayHeadings[ii].setFont(smallFont);
			dayHeadings[ii].setHorizontalAlignment(JTextField.CENTER);
			dayHeadings[ii].setFocusable(false);
			add(dayHeadings[ii], new AbsoluteConstraints(x, 40, 21, 21));
			x += 20;
		}
		// layout the days of the month
		x = startX;
		int y = startY;
		for (int ii = 0; ii < daysInMonth.length; ii++) {
			for (int jj = 0; jj < daysInMonth[ii].length; jj++) {
				daysInMonth[ii][jj].setBackground(gray);
				daysInMonth[ii][jj].setEditable(false);
				daysInMonth[ii][jj].setFont(smallFont);
				daysInMonth[ii][jj].setHorizontalAlignment(JTextField.RIGHT);
				daysInMonth[ii][jj].setText("");
				daysInMonth[ii][jj].setFocusable(false);
				daysInMonth[ii][jj].addMouseListener(new MouseAdapter() {
					public void mouseClicked(final MouseEvent e) {
						onDayClicked(e);
					}
				});
				add(daysInMonth[ii][jj], new AbsoluteConstraints(x, y, 21, 21));
				x += 20;
			}
			x = startX;
			y += 20;
		}
		//
		//monthAndYear.setFont(largeFont);
		selectedTimeField.setHorizontalAlignment(JTextField.CENTER);
		selectedTimeField.setText(timeFormat.format(selectedDate.getTime()));
		add(selectedTimeField, new AbsoluteConstraints(60, 186, 100, 20));

		initButtons(true);
		calculateCalendar();
		this.setSize(300, 229);
	}

	/**
	 * Initializes Today and Cancel buttons dependent on whether hideOnSelect is
	 * set; if the panel will stay open, the Cancel button is invisible.
	 * 
	 * @param firstTime
	 */
	private void initButtons(final boolean firstTime) {
		final int posY = 210;
		if (firstTime) {
			final Insets insets = new Insets(2, 2, 2, 2);
			//
			final Dimension buttonSize = new Dimension(68, 24);
			todayButton.setFont(largeFont);
			todayButton.setText("Today");
			todayButton.setMargin(insets);
			todayButton.setMaximumSize(buttonSize);
			todayButton.setMinimumSize(buttonSize);
			todayButton.setPreferredSize(buttonSize);
			todayButton.setDefaultCapable(true);
			todayButton.setSelected(true);
			todayButton.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent evt) {
					onToday(evt);
				}
			});
			cancelButton.setFont(largeFont);
			cancelButton.setText("Cancel");
			cancelButton.setMargin(insets);
			cancelButton.setMaximumSize(buttonSize);
			cancelButton.setMinimumSize(buttonSize);
			cancelButton.setPreferredSize(buttonSize);
			cancelButton.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent evt) {
					onCancel(evt);
				}
			});
		} else {
			this.remove(todayButton);
			this.remove(cancelButton);
		}
		if (hideOnSelect) {
			add(todayButton, new AbsoluteConstraints(50, posY, 52, -1));
			add(cancelButton, new AbsoluteConstraints(112, posY, 52, -1));
		} else {
			add(todayButton, new AbsoluteConstraints(70, posY, 52, -1));
		}
	}

	/**
	 * Returns true if the panel will be made invisible after a day is selected.
	 * 
	 * @return true or false
	 */
	public boolean isHideOnSelect() {
		return hideOnSelect;
	}

	/**
	 * Controls whether the panel will be made invisible after a day is
	 * selected.
	 * 
	 * @param hideOnSelect
	 */
	public void setHideOnSelect(final boolean hideOnSelect) {
		if (this.hideOnSelect != hideOnSelect) {
			this.hideOnSelect = hideOnSelect;
			initButtons(false);
		}
	}

	/**
	 * Returns the currently selected date.
	 * 
	 * @return date
	 */
	public Date getDate() {
		if (null != selectedDate)
			return selectedDate.getTime();
		return null;
	}

	/**
	 * Event handler for the Today button that sets the currently selected date
	 * to the current date.
	 * 
	 * @param evt
	 */
	protected void onToday(final java.awt.event.ActionEvent evt) {
		selectedDate = getToday();
		setVisible(!hideOnSelect);
		if (isVisible()) { // don't bother with calculation if not visible
			monthAndYear
					.setText(titleDateFormat.format(selectedDate.getTime()));
			calculateCalendar();
		}
	}

	/**
	 * Event handler for the Cancel button that unsets the currently selected
	 * date.
	 * 
	 * @param evt
	 */
	protected void onCancel(final ActionEvent evt) {
		selectedDate = originalDate;
		setVisible(!hideOnSelect);
	}

	/**
	 * @param diff
	 */
	private void moveMonth(final int diff) {
		final int day = selectedDate.get(Calendar.DATE);
		selectedDate.set(Calendar.DATE, 1);
		selectedDate.add(Calendar.MONTH, diff);
		selectedDate.set(Calendar.DATE, Math.min(day,
				calculateDaysInMonth(selectedDate)));
		monthAndYear.setText(titleDateFormat.format(selectedDate.getTime()));
		calculateCalendar();
	}

	/**
	 * Event handler for the forward button that increments the currently
	 * selected month.
	 * 
	 * @param evt
	 */
	protected void onMonthForwardClicked(final java.awt.event.ActionEvent evt) {
		moveMonth(1);
	}

	/**
	 * Event handler for the back button that decrements the currently selected
	 * month.
	 * 
	 * @param evt
	 */
	protected void onMonthBackClicked(final java.awt.event.ActionEvent evt) {
		moveMonth(-1);
	}

	/**
	 * Event handler for the forward button that increments the currently
	 * selected month.
	 * 
	 * @param evt
	 */
	protected void onYearForwardClicked(final java.awt.event.ActionEvent evt) {
		moveMonth(12);
	}

	/**
	 * Event handler for the back button that decrements the currently selected
	 * month.
	 * 
	 * @param evt
	 */
	protected void onYearBackClicked(final java.awt.event.ActionEvent evt) {
		moveMonth(-12);
	}

	/**
	 * Event handler that sets the currently selected date to the clicked day.
	 * 
	 * @param evt
	 */
	protected void onDayClicked(final java.awt.event.MouseEvent evt) {
		final javax.swing.JTextField fld = (javax.swing.JTextField) evt
				.getSource();
		if (!"".equals(fld.getText())) {
			if (null != selectedDay) {
				selectedDay.setBackground(white);
			}
			fld.setBackground(highlight);
			selectedDay = fld;
			selectedDate.set(Calendar.DATE, Integer.parseInt(fld.getText()));
			setVisible(!hideOnSelect);
		}
	}

	/**
	 * Returns the current date.
	 * 
	 * @return date
	 */
	private static GregorianCalendar getToday() {
		final GregorianCalendar gc = new GregorianCalendar();
		gc.set(Calendar.HOUR_OF_DAY, 0);
		gc.set(Calendar.MINUTE, 0);
		gc.set(Calendar.SECOND, 0);
		gc.set(Calendar.MILLISECOND, 0);
		return gc;
	}

	/**
	 * Calculates the days of the month.
	 */
	private void calculateCalendar() {
		// clear the selected date
		if (null != selectedDay) {
			selectedDay.setBackground(white);
			selectedDay = null;
		}
		// get the first day of the selected year and month
		final GregorianCalendar c = new GregorianCalendar(selectedDate
				.get(Calendar.YEAR), selectedDate.get(Calendar.MONTH), 1);
		// figure out the maximum number of days in the month
		final int maxDay = calculateDaysInMonth(c);
		// figure out the day that should be selected in this month
		// based on the previously selected day and the maximum number
		// of days in the month
		final int selectedDayValue = Math.min(maxDay, selectedDate
				.get(Calendar.DATE));
		// clear the days up to the first day of the month
		int dow = c.get(Calendar.DAY_OF_WEEK);
		for (int dd = 0; dd < dow; dd++) {
			daysInMonth[0][dd].setText("");
			daysInMonth[0][dd].setBackground(gray);
		}
		// construct the days in the selected month
		int week;
		do {
			week = c.get(Calendar.WEEK_OF_MONTH);
			dow = c.get(Calendar.DAY_OF_WEEK);
			final JTextField fld = this.daysInMonth[week - 1][dow - 1];
			fld.setText(Integer.toString(c.get(Calendar.DATE)));
			if (selectedDayValue == c.get(Calendar.DATE)) {
				fld.setBackground(highlight);
				this.selectedDay = fld;
			} else
				fld.setBackground(white);
			if (c.get(Calendar.DATE) >= maxDay)
				break;
			c.add(Calendar.DATE, 1);
		} while (c.get(Calendar.DATE) <= maxDay);
		// clear all the days after the last day of the month
		week--;
		for (int ww = week; ww < daysInMonth.length; ww++) {
			for (int dd = dow; dd < daysInMonth[ww].length; dd++) {
				daysInMonth[ww][dd].setText("");
				daysInMonth[ww][dd].setBackground(gray);
			}
			dow = 0;
		}
		// set the currently selected date
		c.set(Calendar.DATE, selectedDayValue);
		selectedDate = c;
	}

	/**
	 * Calculates the number of days in the specified month.
	 * 
	 * @param c
	 * @return number of days in the month
	 */
	private static int calculateDaysInMonth(final Calendar c) {
		return DateUtils.getLastDayOfMonth(c.get(Calendar.YEAR), c
				.get(Calendar.MONTH) + 1);
	}

	/**
	 * @return Returns the titleDateFormat.
	 */
	public DateFormat getTitleDateFormat() {
		return titleDateFormat;
	}

	/**
	 * @param titleDateFormat
	 *               The titleDateFormat to set.
	 */
	public void setTitleDateFormat(DateFormat titleDateFormat) {
		this.titleDateFormat = titleDateFormat;
	}
} //  @jve:visual-info decl-index=0 visual-constraint="10,10"
