/*
 * Joey and its relative products are published under the terms
 * of the Apache Software License.
 */
package org.asyrinx.brownie.core.sql;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.beanutils.PropertyUtils;
import org.asyrinx.brownie.core.lang.StringUtils;

/**
 * @author Akima
 */
public class NamedParamSQL {

    /**
     * Constructor for NamedParamSQL.
     */
    public NamedParamSQL(String original) {
        super();
        this.original = original;
        initQuotedParamClasses();
    }

    protected final String original;

    protected final Set quotedParamClasses = new HashSet();

    protected List paramNames = null;

    private char quote = '\'';

    static public final char PARAM_DELIMETER = '$';

    protected void initQuotedParamClasses() {
        quotedParamClasses.add(String.class);
        quotedParamClasses.add(Character.class);
        quotedParamClasses.add(Character[].class);
    }

    protected Iterator getParamNameIterator() {
        if (paramNames != null)
            return paramNames.iterator();
        paramNames = new ArrayList();
        StringUtils.extractStrings(original, paramNames, PARAM_DELIMETER);
        return paramNames.iterator();
    }

    private Map getPreparingProps() {
        Map result = new HashMap();
        Iterator iterator = getParamNameIterator();
        while (iterator.hasNext()) {
            String paramName = (String) iterator.next();
            result.put(paramName, "?");
        }
        return result;
    }

    /**
     * Method toPreparable.
     * 
     * @return String
     */
    public String toPreparable() {
        return toExecutable(getPreparingProps());
    }

    /**
     * Method toExecutable.
     * 
     * @param value
     * @return String
     */
    public String toExecutable(Object value) {
        if (value instanceof Map) {
            return toExecutableByMap((Map) value);
        } else {
            return toExecutableByBean(value);
        }
    }

    /**
     * Method toExecutable.
     * 
     * @param props
     * @return String
     */
    protected String toExecutableByBean(Object bean) {
        StringBuffer result = new StringBuffer(this.original);
        Iterator iterator = getParamNameIterator();
        while (iterator.hasNext()) {
            String paramName = (String) iterator.next();
            Object param = null;
            try {
                param = PropertyUtils.getNestedProperty(bean, paramName);
            } catch (IllegalAccessException e) {
                //ignore intensionally
            } catch (InvocationTargetException e) {
                //ignore intensionally
            } catch (NoSuchMethodException e) {
                //ignore intensionally
            }
            if (param != null)
                StringUtils.replace(result, PARAM_DELIMETER + paramName + PARAM_DELIMETER, getPropParamValue(param));
        }
        return result.toString();
    }

    /**
     * Method toExecutable.
     * 
     * @param props
     * @return String
     */
    protected String toExecutableByMap(Map props) {
        StringBuffer result = new StringBuffer(this.original);
        Iterator iterator = getParamNameIterator();
        while (iterator.hasNext()) {
            String param = (String) iterator.next();
            StringUtils.replace(result, PARAM_DELIMETER + param + PARAM_DELIMETER, getPropParamValue(props.get(param)));
        }
        return result.toString();
    }

    private String getPropParamValue(Object value) {
        if (value == null)
            return StringUtils.EMPTY;
        if (quotedParamClasses.contains(value.getClass())) {
            return quote + String.valueOf(value) + quote;
        } else {
            return String.valueOf(value);
        }
    }

    /**
     * Returns the quotedParamClasses.
     * 
     * @return Set
     */
    public Set getQuotedParamClasses() {
        return quotedParamClasses;
    }

    /**
     * Returns the original.
     * 
     * @return String
     */
    public String getOriginal() {
        return original;
    }

    /**
     * Returns the quote.
     * 
     * @return char
     */
    public char getQuote() {
        return quote;
    }

    /**
     * Sets the quote.
     * 
     * @param quote
     *               The quote to set
     */
    public void setQuote(char quote) {
        this.quote = quote;
    }

}