/*
 * Joey and its relative products are published under the terms
 * of the Apache Software License.
 */
package org.asyrinx.brownie.core.xml.digester;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.util.Iterator;
import java.util.Stack;

import javax.xml.parsers.SAXParser;

import org.apache.commons.digester.Digester;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.xml.sax.Attributes;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;

/**
 * @author Akima
 */
public class AsyrinxDigester extends Digester {

    static public final String ATTRIBUTE_CLASS_DEFAULT = "class";

    /**
     * Constructor for SettingDigester.
     */
    public AsyrinxDigester() {
        super();
    }

    /**
     * Constructor for SettingDigester.
     * 
     * @param parser
     */
    public AsyrinxDigester(SAXParser parser) {
        super(parser);
    }

    /**
     * Constructor for SettingDigester.
     * 
     * @param reader
     */
    public AsyrinxDigester(XMLReader reader) {
        super(reader);
    }

    protected final Log log = LogFactory.getLog(this.getClass());

    private Stack elementNames = new Stack();

    /**
     * @return
     */
    private String getLastElementName() {
        final StringBuffer result = new StringBuffer();
        for (Iterator i = elementNames.iterator(); i.hasNext();) {
            final Tag tag = (Tag) i.next();
            result.append(tag.qName);
            if (i.hasNext())
                result.append("/");
            else {
                result.append(" ");
                result.append(toString(tag.list));
            }
        }
        return result.toString();
    }

    /**
     * @param result
     * @param tag
     */
    private String toString(Attributes attributes) {
        final StringBuffer result = new StringBuffer();
        for (int j = 0; j < attributes.getLength(); j++) {
            result.append(attributes.getQName(j) + "=" + attributes.getValue(j));
            if (j < attributes.getLength() - 1)
                result.append(" ");
        }
        return result.toString();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.apache.commons.digester.Digester#createSAXException(java.lang.Exception)
     */
    public SAXException createSAXException(Exception e) {
        if (e instanceof DigesterException)
            return (DigesterException) e;
        else
            return new DigesterException(getLastElementName(), e);
    }

    public SAXException createSAXException(Throwable e) {
        return new DigesterException(getLastElementName(), e);
    }

    class Tag {
        public Tag(String qName, Attributes list) {
            this.qName = qName;
            this.list = list;
        }

        public final String qName;

        public final Attributes list;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.apache.commons.digester.Digester#startElement(java.lang.String,
     *      java.lang.String, java.lang.String, org.xml.sax.Attributes)
     */
    public void startElement(String namespaceURI, String localName, String qName, Attributes list)
            throws SAXException {
        elementNames.push(new Tag(qName, list));
        super.startElement(namespaceURI, localName, qName, list);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.apache.commons.digester.Digester#endElement(java.lang.String,
     *      java.lang.String, java.lang.String)
     */
    public void endElement(String namespaceURI, String localName, String qName) throws SAXException {
        elementNames.pop();
        super.endElement(namespaceURI, localName, qName);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.apache.commons.digester.Digester#parse(java.io.File)
     */
    public Object parse(File file) throws IOException, SAXException {
        try {
            return super.parse(file);
        } catch (Throwable e) {
            throw createSAXException(e);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.apache.commons.digester.Digester#parse(org.xml.sax.InputSource)
     */
    public Object parse(InputSource input) throws IOException, SAXException {
        try {
            return super.parse(input);
        } catch (Throwable e) {
            throw createSAXException(e);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.apache.commons.digester.Digester#parse(java.io.InputStream)
     */
    public Object parse(InputStream input) throws IOException, SAXException {
        try {
            return super.parse(input);
        } catch (Throwable e) {
            throw createSAXException(e);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.apache.commons.digester.Digester#parse(java.io.Reader)
     */
    public Object parse(Reader reader) throws IOException, SAXException {
        try {
            return super.parse(reader);
        } catch (Throwable e) {
            throw createSAXException(e);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.apache.commons.digester.Digester#parse(java.lang.String)
     */
    public Object parse(String uri) throws IOException, SAXException {
        try {
            return super.parse(uri);
        } catch (Throwable e) {
            throw createSAXException(e);
        }
    }

    /**
     * Method addSettingRoot.
     * 
     * @param string
     * @param class
     */
    public void addRoot(String pattern, Class clazz) {
        addRoot(pattern, ATTRIBUTE_CLASS_DEFAULT, clazz);
    }

    /**
     * Method addSettingRoot.
     * 
     * @param string
     * @param class
     */
    public void addRoot(String pattern, String attributeName, Class clazz) {
        addObjectCreate(pattern, attributeName, clazz);
    }

    /**
     * Method addSettingRoot.
     * 
     * @param string
     * @param class
     */
    public void addRoot(String pattern, String className) {
        addRoot(pattern, className, ATTRIBUTE_CLASS_DEFAULT);
    }

    /**
     * Method addSettingRoot.
     * 
     * @param string
     * @param class
     */
    public void addRoot(String pattern, String className, String attributeName) {
        addObjectCreate(pattern, className, attributeName);
    }

    public void addAsProp(String pattern, String propertyName) {
        addRule(pattern, new StackSetPropertyRule(propertyName));
    }

    public void addProp(String pattern, Class clazz, String propertyName, boolean setProperties) {
        addObjectCreate(pattern, ATTRIBUTE_CLASS_DEFAULT, clazz);
        addRule(pattern, new StackSetPropertyRule(propertyName));
        if (setProperties)
            addSetProperties(pattern);
    }

    public void addProp(String pattern, Class clazz, String propertyName) {
        addProp(pattern, clazz, propertyName, true);
    }

    /**
     * Method addNewProperties.
     * 
     * @param string
     * @param class
     * @param string1
     */
    public void addProps(String pattern, Class clazz, String addingMethodName) {
        addObjectCreate(pattern, ATTRIBUTE_CLASS_DEFAULT, clazz);
        addSetProperties(pattern);
        addSetNext(pattern, addingMethodName);
        //addSetProperty(pattern, "property", "value");
        addSetProperties(pattern);
    }

    /**
     * Method addNewProperties.
     * 
     * @param pattern
     * @param clazz
     */
    public void addProps(String pattern, Class clazz) {
        addProps(pattern, clazz, "add");
    }

    /**
     * 
     * @param pattern
     * @param methodName
     * @param paramTypes
     * @param paramName
     * @see org.apache.commons.digester.Digester#addCallMethod(java.lang.String,
     *      java.lang.String, int, java.lang.Class[])
     */
    public void addCallMethod(String pattern, String methodName, Class[] paramTypes,
            String[] paramName) {
        addCallMethod(pattern, methodName, paramTypes.length, paramTypes);
        for (int i = 0; i < paramName.length; i++) {
            addCallParam(pattern, i, paramName[i]);
        }
    }

    /**
     * 
     * @param patten
     * @param propertyName
     */
    public void addPushProperty(String patten, String propertyName) {
        addRule(patten, new PushPropertyRule(propertyName));
    }

    /**
     * 
     * @param patten
     */
    public void addPop(String patten) {
        addRule(patten, new PopRule());
    }

    /**
     * @param string
     * @param class1
     * @param classes
     * @param strings
     */
    public void addConstructor(String patten, Class clazz, Class[] paramTypes, String[] paramNames) {
        addRule(patten, new ConstructorRule(clazz, paramTypes, paramNames));
    }

    /**
     * @param string
     * @param class1
     * @param classes
     * @param strings
     * @param propertyName
     * @param setProperties
     */
    public void addProp(String pattern, Class clazz, Class[] paramTypes, String[] paramNames,
            String propertyName) {
        addProp(pattern, clazz, paramTypes, paramNames, propertyName, true);
    }

    /**
     * @param string
     * @param class1
     * @param classes
     * @param strings
     * @param propertyName
     * @param setProperties
     */
    public void addProp(String pattern, Class clazz, Class[] paramTypes, String[] paramNames,
            String propertyName, boolean setProperties) {
        addRule(pattern, new ConstructorRule(clazz, paramTypes, paramNames));
        addRule(pattern, new StackSetPropertyRule(propertyName));
        if (setProperties)
            addSetProperties(pattern);
    }

    /**
     * @param string
     */
    public void addSetToMap(String pattern) {
        addRule(pattern, new SetToMapRule());
    }

    /**
     * @param string
     * @param string2
     */
    public void addBodyToProp(String pattern, String propertyName) {
        addRule(pattern, new SetBodyToPropertyRule(propertyName));
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.apache.commons.digester.Digester#addSetProperties(java.lang.String)
     */
    public void addSetProperties(String pattern) {
        addRule(pattern, new SetPropertiesWithOptionsRule());
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.apache.commons.digester.Digester#addSetProperties(java.lang.String,
     *      java.lang.String, java.lang.String)
     */
    public void addSetProperties(String pattern, String attributeName, String propertyName) {
        addRule(pattern, new SetPropertiesWithOptionsRule(attributeName, propertyName));
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.apache.commons.digester.Digester#addSetProperties(java.lang.String,
     *      java.lang.String[], java.lang.String[])
     */
    public void addSetProperties(String pattern, String[] attributeNames, String[] propertyNames) {
        addRule(pattern, new SetPropertiesWithOptionsRule(attributeNames, propertyNames));
    }

    public void addSetProperties(String pattern, String optionsPropertyName) {
        addRule(pattern, new SetPropertiesWithOptionsRule(optionsPropertyName));
    }

    public void addSetProperties(String pattern, String attributeName, String propertyName,
            String optionsPropertyName) {
        addRule(pattern, new SetPropertiesWithOptionsRule(attributeName, propertyName,
                optionsPropertyName));
    }

    public void addSetProperties(String pattern, String[] attributeNames, String[] propertyNames,
            String optionsPropertyName) {
        addRule(pattern, new SetPropertiesWithOptionsRule(attributeNames, propertyNames,
                optionsPropertyName));
    }

}