/*
 * Joey and its relative products are published under the terms
 * of the Apache Software License.
 */
package org.asyrinx.brownie.jdbc.measure;

import java.io.InputStream;
import java.io.Reader;
import java.math.BigDecimal;
import java.net.URL;
import java.sql.Array;
import java.sql.Blob;
import java.sql.Clob;
import java.sql.Date;
import java.sql.ParameterMetaData;
import java.sql.PreparedStatement;
import java.sql.Ref;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Time;
import java.sql.Timestamp;
import java.util.Calendar;

import org.asyrinx.brownie.core.log.MeasureLog;

/**
 * @author Akima
 */
public class PreparedStatementMeasure
	extends StatementMeasure
	implements PreparedStatement {

	/**
	 * Constructor for PreparedStatementMeasure.
	 * @param source
	 * @param log
	 */
	public PreparedStatementMeasure(
		PreparedStatement source,
		Object key,
		MeasureLog measureLog) {
		super(source, measureLog);
		this.source = source;
		this.keyName = String.valueOf(key);
	}

	/**
	 * Constructor for PreparedStatementMeasure.
	 * @param source
	 */
	public PreparedStatementMeasure(PreparedStatement source, Object key) {
		this(source, key, null);
	}

	private final PreparedStatement source;
	private final String keyName;

	/**
	 * @see java.sql.PreparedStatement#addBatch()
	 */
	public void addBatch() throws SQLException {
		source.addBatch();
	}

	/**
	 * @see java.sql.PreparedStatement#clearParameters()
	 */
	public void clearParameters() throws SQLException {
		source.clearParameters();
	}

	/**
	 * @see java.sql.PreparedStatement#execute()
	 */
	public boolean execute() throws SQLException {
		measureLog.reset();
		boolean result = source.execute();
		measureLog.done(keyName);
		return result;
	}

	/**
	 * @see java.sql.PreparedStatement#executeUpdate()
	 */
	public int executeUpdate() throws SQLException {
		measureLog.reset();
		int result = source.executeUpdate();
		measureLog.done(keyName);
		return result;
	}

	/**
	 * @see java.sql.PreparedStatement#executeQuery()
	 */
	public ResultSet executeQuery() throws SQLException {
		measureLog.reset();
		ResultSet result = source.executeQuery();
		measureLog.done(keyName);
		return result;
	}

	/**
	 * @see java.sql.PreparedStatement#setBigDecimal(int, BigDecimal)
	 */
	public void setBigDecimal(int parameterIndex, BigDecimal x)
		throws SQLException {
		source.setBigDecimal(parameterIndex, x);
	}

	/**
	 * @see java.sql.PreparedStatement#setBoolean(int, boolean)
	 */
	public void setBoolean(int parameterIndex, boolean x) throws SQLException {
		source.setBoolean(parameterIndex, x);
	}

	/**
	 * @see java.sql.PreparedStatement#setBytes(int, byte[])
	 */
	public void setBytes(int parameterIndex, byte[] x) throws SQLException {
		source.setBytes(parameterIndex, x);
	}

	/**
	 * @see java.sql.PreparedStatement#setDate(int, Date)
	 */
	public void setDate(int parameterIndex, Date x) throws SQLException {
		source.setDate(parameterIndex, x);
	}

	/**
	 * @see java.sql.PreparedStatement#setDate(int, Date, Calendar)
	 */
	public void setDate(int parameterIndex, Date x, Calendar cal)
		throws SQLException {
		source.setDate(parameterIndex, x, cal);
	}

	/**
	 * @see java.sql.PreparedStatement#setDouble(int, double)
	 */
	public void setDouble(int parameterIndex, double x) throws SQLException {
		source.setDouble(parameterIndex, x);
	}

	/**
	 * @see java.sql.PreparedStatement#setFloat(int, float)
	 */
	public void setFloat(int parameterIndex, float x) throws SQLException {
		source.setFloat(parameterIndex, x);
	}

	/**
	 * @see java.sql.PreparedStatement#setInt(int, int)
	 */
	public void setInt(int parameterIndex, int x) throws SQLException {
		source.setInt(parameterIndex, x);
	}

	/**
	 * @see java.sql.PreparedStatement#setLong(int, long)
	 */
	public void setLong(int parameterIndex, long x) throws SQLException {
		source.setLong(parameterIndex, x);
	}

	/**
	 * @see java.sql.PreparedStatement#setNull(int, int)
	 */
	public void setNull(int parameterIndex, int sqlType) throws SQLException {
		source.setNull(parameterIndex, sqlType);
	}

	/**
	 * @see java.sql.PreparedStatement#setNull(int, int, String)
	 */
	public void setNull(int paramIndex, int sqlType, String typeName)
		throws SQLException {
		source.setNull(paramIndex, sqlType, typeName);
	}

	/**
	 * @see java.sql.PreparedStatement#setObject(int, Object)
	 */
	public void setObject(int parameterIndex, Object x) throws SQLException {
		source.setObject(parameterIndex, x);
	}

	/**
	 * @see java.sql.PreparedStatement#setObject(int, Object, int)
	 */
	public void setObject(int parameterIndex, Object x, int targetSqlType)
		throws SQLException {
		source.setObject(parameterIndex, x, targetSqlType);
	}

	/**
	 * @see java.sql.PreparedStatement#setObject(int, Object, int, int)
	 */
	public void setObject(
		int parameterIndex,
		Object x,
		int targetSqlType,
		int scale)
		throws SQLException {
		source.setObject(parameterIndex, x, targetSqlType, scale);
	}

	/**
	 * @see java.sql.PreparedStatement#setShort(int, short)
	 */
	public void setShort(int parameterIndex, short x) throws SQLException {
		source.setShort(parameterIndex, x);
	}

	/**
	 * @see java.sql.PreparedStatement#setString(int, String)
	 */
	public void setString(int parameterIndex, String x) throws SQLException {
		source.setString(parameterIndex, x);
	}

	/**
	 * @see java.sql.PreparedStatement#setTime(int, Time)
	 */
	public void setTime(int parameterIndex, Time x) throws SQLException {
		source.setTime(parameterIndex, x);
	}

	/**
	 * @see java.sql.PreparedStatement#setTime(int, Time, Calendar)
	 */
	public void setTime(int parameterIndex, Time x, Calendar cal)
		throws SQLException {
		source.setTime(parameterIndex, x, cal);
	}

	/**
	 * @see java.sql.PreparedStatement#setTimestamp(int, Timestamp)
	 */
	public void setTimestamp(int parameterIndex, Timestamp x)
		throws SQLException {
		source.setTimestamp(parameterIndex, x);
	}

	/**
	 * @see java.sql.PreparedStatement#setTimestamp(int, Timestamp, Calendar)
	 */
	public void setTimestamp(int parameterIndex, Timestamp x, Calendar cal)
		throws SQLException {
		source.setTimestamp(parameterIndex, x, cal);
	}

	/**
	 * @see java.sql.PreparedStatement#setArray(int, Array)
	 */
	public void setArray(int i, Array x) throws SQLException {
		source.setArray(i, x);
	}

	/**
	 * @see java.sql.PreparedStatement#setAsciiStream(int, InputStream, int)
	 */
	public void setAsciiStream(int parameterIndex, InputStream x, int length)
		throws SQLException {
		source.setAsciiStream(parameterIndex, x, length);
	}

	/**
	 * @see java.sql.PreparedStatement#setBinaryStream(int, InputStream, int)
	 */
	public void setBinaryStream(int parameterIndex, InputStream x, int length)
		throws SQLException {
		source.setBinaryStream(parameterIndex, x, length);
	}

	/**
	 * @see java.sql.PreparedStatement#setBlob(int, Blob)
	 */
	public void setBlob(int i, Blob x) throws SQLException {
		source.setBlob(i, x);
	}

	/**
	 * @see java.sql.PreparedStatement#setByte(int, byte)
	 */
	public void setByte(int parameterIndex, byte x) throws SQLException {
		source.setByte(parameterIndex, x);
	}

	/**
	 * @see java.sql.PreparedStatement#setCharacterStream(int, Reader, int)
	 */
	public void setCharacterStream(
		int parameterIndex,
		Reader reader,
		int length)
		throws SQLException {
		source.setCharacterStream(parameterIndex, reader, length);
	}

	/**
	 * @see java.sql.PreparedStatement#setClob(int, Clob)
	 */
	public void setClob(int i, Clob x) throws SQLException {
		source.setClob(i, x);
	}

	/**
	 * @see java.sql.PreparedStatement#setRef(int, Ref)
	 */
	public void setRef(int i, Ref x) throws SQLException {
		source.setRef(i, x);
	}

	/**
	 * @see java.sql.PreparedStatement#setUnicodeStream(int, InputStream, int)
	 * @deprecated
	 */
	public void setUnicodeStream(int parameterIndex, InputStream x, int length)
		throws SQLException {
		//source.setUnicodeStream(parameterIndex, x, length);
		//params.add(parameterIndex, x);
	}

	/**
	 * @see java.sql.PreparedStatement#getMetaData()
	 */
	public ResultSetMetaData getMetaData() throws SQLException {
		return source.getMetaData();
	}
	/**
	 * @return
	 * @throws java.sql.SQLException
	 */
	public ParameterMetaData getParameterMetaData() throws SQLException {
		return source.getParameterMetaData();
	}

	/**
	 * @param parameterIndex
	 * @param x
	 * @throws java.sql.SQLException
	 */
	public void setURL(int parameterIndex, URL x) throws SQLException {
		source.setURL(parameterIndex, x);
	}

}
