/*
 * Joey and its relative products are published under the terms
 * of the Apache Software License.
 */
/*
 * Created on 2004/01/27
 */
package org.asyrinx.joey.gen.model;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.apache.commons.collections.IteratorUtils;
import org.apache.commons.lang.ObjectUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.torque.engine.EngineException;
import org.apache.torque.engine.database.model.Column;
import org.apache.torque.engine.database.model.ForeignKey;
import org.apache.torque.engine.database.model.Index;
import org.apache.torque.engine.database.model.NameFactory;
import org.apache.torque.engine.database.model.Table;
import org.asyrinx.joey.gen.model.visitor.MockBaseVisitor;
import org.asyrinx.joey.gen.model.visitor.TopDownVisitor;

/**
 * @author akima
 */
public class WmTable extends Table implements Cloneable, IElement {

	/**
	 * 
	 */
	public WmTable() {
		super();
	}

	/**
	 * @param name
	 */
	public WmTable(String name) {
		super(name);
	}

	public void accept(IVisitor visitor) {
		visitor.visit(this);
	}

	final Log log = LogFactory.getLog(this.getClass());

	/**
	 * @see org.asyrinx.brownie.windmill.model.IElement#finishLoad()
	 */
	public void finishLoad() {
		if (StringUtils.isEmpty(getJavaNamingMethod()))
			setJavaNamingMethod(getDatabase().getDefaultJavaNamingMethod());

		if ("null".equals(getIdMethod())) {
			setIdMethod(getDatabase().getDefaultIdMethod());
		}
		if ("autoincrement".equals(getIdMethod())
			|| "sequence".equals(getIdMethod())) {
			log.warn(
				"The value '"
					+ getIdMethod()
					+ "' for Torque's "
					+ "table.idMethod attribute has been deprecated in favor "
					+ "of '"
					+ NATIVE
					+ "'.  Please adjust your "
					+ "Torque XML schema accordingly.");
			setIdMethod(NATIVE);
		}
		if (!isHeavyIndexing() && getDatabase().isHeavyIndexing())
			setHeavyIndexing(true);
	}

	/**
	 * <p>A hook for the SAX XML parser to call when this table has
	 * been fully loaded from the XML, and all nested elements have
	 * been processed.</p>
	 *
	 * <p>Performs heavy indexing and naming of elements which weren't
	 * provided with a name.</p>
	 */
	public void doFinalInitialization() {
		// Heavy indexing must wait until after all columns composing
		// a table's primary key have been parsed.
		if (isHeavyIndexing()) {
			doHeavyIndexing();
		}

		// Name any indices which are missing a name using the
		// appropriate algorithm.
		doNaming();
	}

	/**
	 * <p>Adds extra indices for multi-part primary key columns.</p>
	 *
	 * <p>For databases like MySQL, values in a where clause must
	 * match key part order from the left to right.  So, in the key
	 * definition <code>PRIMARY KEY (FOO_ID, BAR_ID)</code>,
	 * <code>FOO_ID</code> <i>must</i> be the first element used in
	 * the <code>where</code> clause of the SQL query used against
	 * this table for the primary key index to be used.  This feature
	 * could cause problems under MySQL with heavily indexed tables,
	 * as MySQL currently only supports 16 indices per table (i.e. it
	 * might cause too many indices to be created).</p>
	 *
	 * <p>See <a href="http://www.mysql.com/doc/E/X/EXPLAIN.html">the
	 * manual</a> for a better description of why heavy indexing is
	 * useful for quickly searchable database tables.</p>
	 */
	protected final void doHeavyIndexing() {
		if (log.isDebugEnabled()) {
			log.debug("doHeavyIndex() called on table " + getName());
		}

		List pk = getPrimaryKey();
		int size = pk.size();

		try {
			// We start at an offset of 1 because the entire column
			// list is generally implicitly indexed by the fact that
			// it's a primary key.
			for (int i = 1; i < size; i++) {
				addIndex(new WmIndex(this, pk.subList(i, size)));
			}
		} catch (EngineException e) {
			log.error(e, e);
		}
	}

	/**
	 * Names composing objects which haven't yet been named.  This
	 * currently consists of foreign-key and index entities.
	 */
	protected final void doNaming() {

		// Assure names are unique across all databases.
		try {
			ForeignKey[] foreignKeies = getForeignKeys();
			for (int i = 0, size = foreignKeies.length; i < size; i++) {
				ForeignKey fk = foreignKeies[i];
				String name = fk.getName();
				if (StringUtils.isEmpty(name)) {
					name = acquireConstraintName("FK", i + 1);
					fk.setName(name);
				}
			}
			Index[] indexs = getIndices();
			for (int i = 0, size = indexs.length; i < size; i++) {
				Index index = indexs[i];
				String name = index.getName();
				if (StringUtils.isEmpty(name)) {
					name = acquireConstraintName("I", i + 1);
					index.setName(name);
				}
			}

			// NOTE: Most RDBMSes can apparently name unique column
			// constraints/indices themselves (using MySQL and Oracle
			// as test cases), so we'll assume that we needn't add an
			// entry to the system name list for these.
		} catch (EngineException nameAlreadyInUse) {
			log.error(nameAlreadyInUse, nameAlreadyInUse);
		}
	}

	/**
	 * Macro to a constraint name.
	 *
	 * @param nameType constraint type
	 * @param nbr unique number for this constraint type
	 * @return unique name for constraint
	 * @throws EngineException
	 */
	protected final String acquireConstraintName(String nameType, int nbr)
		throws EngineException {
		List inputs = new ArrayList(4);
		inputs.add(getDatabase());
		inputs.add(getName());
		inputs.add(nameType);
		inputs.add(new Integer(nbr));
		return NameFactory.generateName(
			NameFactory.CONSTRAINT_GENERATOR,
			inputs);
	}

	/**
	 * @see org.apache.torque.engine.database.model.Table#toString()
	 */
	public String toString() {
		return ObjectUtils.identityToString(this);
	}
	/**
	 * @see java.lang.Object#clone()
	 */
	public Object clone() throws CloneNotSupportedException {
		return super.clone();
	}

	private String javaNamingMethod = null;
	private boolean heavyIndexing = false;

	private String displayPropertyName = null;
	private String loginNamePropertyName = null;
	private String label = null;
	private boolean isLoginUser = false;

	/**
	 * @return
	 */
	public String getLabel() {
		if (StringUtils.isEmpty(this.label))
			return getName();
		else
			return label;
	}

	/**
	 * @param string
	 */
	public void setLabel(String string) {
		label = string;
	}

	/**
	 * @see org.apache.torque.engine.database.model.Table#addColumn(org.apache.torque.engine.database.model.Column)
	 */
	public void addColumn(Column col) {
		super.addColumn(col);
	}

	private final List aspectRefs = new ArrayList();

	public void addAspectRef(WmAspectRef aspectRef) {
		aspectRef.setTable(this);
		aspectRefs.add(aspectRef);
	}

	public List getAspectRefs() {
		return aspectRefs;
	}

	public List getAspects() {
		final List result = new ArrayList();
		final Iterator iterator = aspectRefs.iterator();
		while (iterator.hasNext()) {
			final WmAspectRef ref = (WmAspectRef) iterator.next();
			final WmAspect aspect =
				((WmDatabase) this.getDatabase()).findAspect(ref.getName());
			if (aspect != null)
				result.add(aspect);
		}
		return result;
	}

	public List getProperties() {
		final List result = new ArrayList();
		final Iterator colIter = IteratorUtils.arrayIterator(getColumns());
		while (colIter.hasNext()) {
			final WmColumn column = (WmColumn) colIter.next();
			if (column.getAdditional() != null)
				result.add(column.getAdditional());
		}
		return result;
	}

	public List getAllProperties() {
		final List result = new ArrayList();
		final Iterator colIter = IteratorUtils.arrayIterator(getColumns());
		while (colIter.hasNext()) {
			final WmColumn column = (WmColumn) colIter.next();
			result.add(column);
			if (column.getAdditional() != null)
				result.add(column.getAdditional());
		}
		return result;
	}

	public IProperty findProperty(String name) {
		if (StringUtils.isEmpty(name))
			return null;
		final Iterator iterator = getProperties().iterator();
		while (iterator.hasNext()) {
			WmProperty property = (WmProperty) iterator.next();
			if (name.equals(property.getName()))
				return property;
		}
		final Column column = getColumn(name);
		if (column instanceof IProperty)
			return (IProperty) column;
		else
			return null;
	}

	public IProperty getDisplayProperty() {
		return findProperty(this.displayPropertyName);
	}

	public IProperty getLoginNameProperty() {
		return findProperty(this.loginNamePropertyName);
	}

	private final List joins = new ArrayList();

	public void addJoin(WmJoin join) {
		join.setParentTable(this);
		joins.add(join);
	}

	public List getJoins() {
		return joins;
	}

	public List getPrimaryKeyNames() {
		final List pkeys = getPrimaryKey();
		final List result = new ArrayList(pkeys.size());
		final Iterator iterator = pkeys.iterator();
		while (iterator.hasNext()) {
			final Column col = (Column) iterator.next();
			result.add(col.getName());
		}
		return result;
	}

	public WmForeignKey findForeignKey(String name) {
		if (StringUtils.isEmpty(name))
			return null;
		final Iterator iterator = IteratorUtils.arrayIterator(getForeignKeys());
		while (iterator.hasNext()) {
			WmForeignKey foreignKey = (WmForeignKey) iterator.next();
			if (name.equals(foreignKey.getName()))
				return foreignKey;
		}
		return null;
	}

	public Set getJoinTableSet() {
		final Set result = new HashSet();
		final IVisitor visitor = new TopDownVisitor(new MockBaseVisitor() {
			/**
			 * @see org.asyrinx.brownie.windmill.model.visitor.MockBaseVisitor#visit(org.asyrinx.brownie.windmill.model.WmJoin)
			 */
			public void visit(WmJoin join) {
				result.add(join.getTargetTable());
			}
		});
		this.accept(visitor);
		result.remove(this);
		return result;
	}

	/**
	 * ̃e[ũvpeBgĂEnumׂĎ擾
	 * @return
	 */
	public Set getEnums() {
		final Set result = new HashSet();
		for (Iterator iter = getProperties().iterator(); iter.hasNext();) {
			final WmProperty prop = (WmProperty) iter.next();
			if (prop.getEnum() != null)
				result.add(prop.getEnum());
		}
		return result;
	}

	/**
	 * @return
	 */
	public boolean isLoginUser() {
		return isLoginUser;
	}

	/**
	 * @param b
	 */
	public void setLoginUser(boolean b) {
		isLoginUser = b;
	}

	/**
	 * @return
	 */
	public String getDisplayPropertyName() {
		return displayPropertyName;
	}

	/**
	 * @param string
	 */
	public void setDisplayPropertyName(String string) {
		displayPropertyName = string;
	}

	/**
	 * @return
	 */
	public String getLoginNamePropertyName() {
		return loginNamePropertyName;
	}

	/**
	 * @param string
	 */
	public void setLoginNamePropertyName(String string) {
		loginNamePropertyName = string;
	}

	/**
	 * @return
	 */
	public String getJavaNamingMethod() {
		return javaNamingMethod;
	}

	/**
	 * @param string
	 */
	public void setJavaNamingMethod(String string) {
		javaNamingMethod = string;
	}

	/**
	 * @return
	 */
	public boolean isHeavyIndexing() {
		return heavyIndexing;
	}

	/**
	 * @param b
	 */
	public void setHeavyIndexing(boolean b) {
		heavyIndexing = b;
	}

}
