#
#  The pkgbuild build engine
#
#  Copyright (c) 2010, Oracle and/or its affiliates. All rights reserved.
#
#  pkgbuild is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License 
#  version 2 published by the Free Software Foundation.
#
#  pkgbuild is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#  General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#  As a special exception to the GNU General Public License, if you
#  distribute this file as part of a program that contains a
#  configuration script generated by Autoconf, you may include it under
#  the same distribution terms that you use for the rest of that program.
#
#  Authors:  Laszlo Peter  <laca@sun.com>
#

use strict;
use warnings;
use Socket;
use IO::Socket::INET;

package ips_utils;

my $my_local_ip;

sub new ($;$) {
    my $class = shift;
    my $altroot = shift;
    my $self = {};

    $self->{_altroot} = $altroot;
    $altroot = "" if not defined ($altroot);

    if (! -d "${altroot}/var/pkg") {
	return undef;
    }
    $self->{_publishers} = {};
    $self->{_properties} = {};
    $self->{_filter} = {};
    $self->{_variant} = {};
    $self->{_policy} = {};
    $self->{_unknown} = {};
    $self->{_pkginfo} = {};
    $self->{_installed} = {};
    bless ($self, $class);
    $self->read_cfg ();
    return $self;
}

sub read_cfg ($) {
    my $self = shift;

    my $altroot = $self->{_altroot};
    $altroot = "" if not defined ($altroot);

    # FIXME: does not consider an altroot scenario
    open IPS_AUTH, "/usr/bin/pkg publisher -H |" or
	die "Cannot read IPS configuration";
    my $pkgbuild_ips_host;
    my $pkgbuild_ips_port;
    my $pkgbuild_ips_server = $ENV{PKGBUILD_IPS_SERVER};
    if (defined ($pkgbuild_ips_server)) {
	if ($pkgbuild_ips_server =~ /^http:\/\/([^:]+):([0-9]+)/) {
	    $pkgbuild_ips_host = $1;
	    $pkgbuild_ips_port = $2;
	}
    }
    while (my $line = <IPS_AUTH>) {
	chomp ($line);
	if ($line =~ /^(\S+)\s+.*\s*((file|https?):\/\/\S+).*$/) {
	    my $publisher = $1;
	    my $origin = $2;
	    $self->{_publishers}->{$publisher} = {};
	    $self->{_publishers}->{$publisher}->{origin} = $origin;
	    if ($origin =~ /^https?:\/\/(.+):(.+)\/$/) {
		my $host = $1;
		my $port = $2;
		my $local_port = $self->get_local_ips_port ();
		if ($port == $local_port) {
		    my $packed_ip = gethostbyname ($host);
		    if (not defined($my_local_ip)) {
			my $sock = IO::Socket::INET->new(
			    PeerAddr=> "${host}",
			    PeerPort=> ${port},
			    Proto   => "tcp");
			if (defined ($sock)) {
			    $my_local_ip = $sock->sockhost;
			    shutdown ($sock, 2);
			} else {
			    print "WARNING: failed to connect to IPS server $host\n";
			}
		    }
		    if (defined $packed_ip and
			defined $my_local_ip) {
			my $ip_address = Socket::inet_ntoa($packed_ip);
			if (($ip_address eq $my_local_ip) or
			    ($ip_address eq "127.0.0.1")) {
			    $self->{_local_publisher} = $publisher;
			}
		    }
		    if (defined ($pkgbuild_ips_server) and
			($port == $pkgbuild_ips_port)) {
			my $packed_ip = gethostbyname ($host);
			my $pkgbuild_packed_ip = 
			    gethostbyname ($pkgbuild_ips_host);
			if (defined $packed_ip and
			    defined $pkgbuild_packed_ip) {
			    my $ip_address = Socket::inet_ntoa($packed_ip);
			    my $pkgbuild_ip = Socket::inet_ntoa ($pkgbuild_packed_ip);
			    if (($ip_address eq $pkgbuild_ip) or
				($ip_address eq "127.0.0.1")) {
				$self->{_pkgbuild_publisher} = $publisher;
			    }
			}
		    }
		}
	    } elsif ($origin =~ /^file:\/\//) {
		if (not defined ($self->{_local_publisher})) {
		    $self->{_local_publisher} = $publisher;
		}
		if (defined ($pkgbuild_ips_server) and
		    $pkgbuild_ips_server eq $origin) {
		    $self->{_pkgbuild_publisher} = $publisher;
		}
	    }
	} else {
	    print "ips_utils: failed to parse line: $line\n";
	}
    }
    close IPS_AUTH;
    if (not defined ($pkgbuild_ips_server) and defined($self->{_local_publisher})) {
	$self->{_pkgbuild_publisher} = $self->{_local_publisher};
    }
}

sub is_installed($$) {
    my $self = shift;
    my $pkgname = shift;

    if (defined ($self->{_installed}->{$pkgname})) {
	return $self->{_installed}->{$pkgname};
    }
    my $pkg_out = `pkg info -l $pkgname 2>&1`;
    my $result = (! $?);
    $self->{_pkginfo}->{$pkgname} = $pkg_out;
    $self->{_installed}->{$pkgname} = $result;
    return $result;
}

sub verify_server ($) {
    my $ips_server = shift;

    return undef if not defined ($ips_server);

    if ($ips_server =~ /([^\/])$/) {
	return "server URI must end with a \"/\"";
    }

    return undef;
}

sub get_publisher_setting ($$$) {
    my $self = shift;
    my $auth_name = shift;
    my $setting = shift;

    return $self->{_publisher}->{$auth_name}->{$setting};
}

sub get_local_ips_port ($) {
    my $self = shift;

    if (not defined ($self->{_local_ips_port})) {
	my $port = `/usr/bin/svcprop -p pkg/port svc:/application/pkg/server:default`;
	chomp ($port);
	$self->{_local_ips_port} = $port;
    }
    return $self->{_local_ips_port};
}

sub get_local_ips_server ($) {
    my $self = shift;

    if (not defined ($self->{_local_ips_server})) {
	my $port = $self->get_local_ips_port ();
	$self->{_local_ips_server} = "http://localhost:$port/";
    }
    return $self->{_local_ips_server};
}

sub get_local_publisher ($) {
    my $self = shift;

    return $self->{_local_publisher};
}

sub get_pkgbuild_publisher ($) {
    my $self = shift;

    return $self->{_pkgbuild_publisher};
}

sub is_depotd_enabled ($) {
    my $self = shift;

    if (not defined ($self->{_depotd_enabled})) {
	my $server_state = `svcs -H -o STATE svc:/application/pkg/server:default`;
	my $read_only = `svcprop -c -p pkg/readonly svc:/application/pkg/server:default`;
	chomp($server_state);
	chomp($read_only);
	if ($server_state eq "online" and $read_only eq "true") {
	    print "WARNING: the local IPS server is running in read-only mode\n";
	    print "HINT: use svccfg -s svc:/application/pkg/server setprop pkg/readonly = false\n";
	    print "HINT: followed by svcadm refresh svc:/application/pkg/server\n";
	    print "HINT: to switch to read-write mode, or set PKGBUILD_IPS_SERVER\n";
	    print "HINT: to a different publisher URL\n";
	}
	if ($server_state eq "online" and $read_only eq "false") {
	    $self->{_depotd_enabled} = 1;
	} else {
	    $self->{_depotd_enabled} = 0;
	}
    }
    return $self->{_depotd_enabled};
}

1;
