#!/usr/bin/perl
#
#  The pkgbuild build engine
#
#  Copyright (c) 2010, Oracle and/or its affiliates. All rights reserved.
#
#  pkgbuild is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License 
#  version 2 published by the Free Software Foundation.
#
#  pkgbuild is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#  General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#  As a special exception to the GNU General Public License, if you
#  distribute this file as part of a program that contains a
#  configuration script generated by Autoconf, you may include it under
#  the same distribution terms that you use for the rest of that program.
#
#  Authors:  Laszlo Peter  <laca@sun.com>
#

use strict;
use warnings;
use Getopt::Long qw(:config gnu_compat no_auto_abbrev bundling pass_through);
use rpm_spec;
use config;
use File::Copy;
use File::Basename;
use ips_utils;
use ips_package;

my $ips_utils = new ips_utils ();

my $ips_server = $ENV{PKGBUILD_IPS_SERVER};
my $server_error = ips_utils::verify_server ($ips_server);
if (defined $server_error) {
    print STDERR "ERROR: invalid PKGBUILD_IPS_SERVER setting: $server_error\n";
    exit (1);
}

if (not defined($ips_server)) {
    if (defined ($ips_utils)) {
	$ips_server = $ips_utils->get_local_ips_server();
    }
}

my $myname = "pkgtool";
my $myversion = rpm_spec::get_version();

# --------- global vars ----------------------------------------------------
# config settings
my $the_good_build_dir;
my $the_good_rpms_copy_dir;
my $live_summary = 1;

my $build_command;

my $exit_val = 0;
my $full_path = 0;

# counter used as an id for the spec files
my $spec_counter = 0;

# array of spec objects
my @specs_to_build = ();
my @build_status;
my @status_details;
my @pkg_list = ();

my %all_specs;
my %provider;
my %warned_about;
my %specs_copied;

my $logname = $ENV{USER} || $ENV{LOGNAME} || `logname`;
chomp ($logname);
my $_homedir = $ENV{HOME};

# defined if the pkgformat is datastream
my $ds;

# --------- defaults -------------------------------------------------------
my $defaults;
my @predefs = ();
my $build_engine_name = "pkgbuild";
my $pkgbuild_path = "pkgbuild";
my $build_engine = "pkgbuild";
my $topdir = "$_homedir/packages";
my $can_notify = 0;
my $download_test = 0;

# Which package mechanism are we going to install by default?
my $ips;
my $svr4;

sub process_defaults () {
    my $default_spec_dir = "$topdir/SPECS";
    $topdir = rpm_spec::get_topdir ($build_engine, \@predefs);

    $defaults = config->new ();
    $defaults->define ('topdir', $topdir);
    $defaults->add ('target', 's', 
		    'the value of the --target option passed on to rpm');
    $defaults->add ('logdir', 's',
		    'the directory for saving log files',
		    '/tmp');
    $defaults->add ('logdir_url', 's',
		    'a URL pointing to the log directory (used in the HTML build report)',
		    'file:///tmp');
    $defaults->add ('tarballdirs', 's',
		    'colon (:) separated list of directories where source tarballs are searched for',
		    "%{topdir}/SOURCES");
    $defaults->add ('sourcedirs', 's',
		    'colon (:) separated list of directories where extra sources (not tarballs) are searched for',
		    "%{topdir}/SOURCES");
    $defaults->add ('specdirs', 's',
		    'colon (:) separated list of directories where spec files are searched for',
		    "%{topdir}/SPECS");
    $defaults->add ('patchdirs', 's',
		    'colon (:) separated list of directories where source patches are searched for',
		    "%{topdir}/SOURCES");
    $defaults->add ('nightly', '!',
		    'suffix the Release rpm tag with the date (specified by date_format)',
		    0);
    $defaults->add ('verbose', 'n', 
		    'level of verbosity; 0 means quiet operation',
		    1);
    $defaults->add ('debug', 'n',
		    'debug level',
		    0);
    $defaults->add ('prodname', 's',
		    'name of the product to appear in the error mail subject',
		    'unnamed');
    $defaults->add ('summary_log', 's',
		    'file name for the HTML summary build report');
    $defaults->add ('summary_title', 's',
		    'title of the HTML summary build report',
		    'Build Report');
    $defaults->add ('rpm_url', 's',
		    'a URL pointing to the directory where the resulting rpms will be save (used in the HTML build report)');
    $defaults->add ('srpm_url', 's',
		    'a URL pointing to the directory where the resulting source srpms will be save (used in the HTML build report)');
    $defaults->add ('autodeps', '!',
		    'attempt to find spec files for missing dependencies',
		    0);
    $defaults->add ('deps', '!',
		    'whether to check dependencies; use nodeps to ignore dependencies',
		    1);
    $defaults->add ('update_if_newer', '!',
		    'rebuild and update packages if the version in the spec file is newer than the version installed',
		    0);
    $defaults->add ('update_always', '!',
		    'rebuild and update packages, even if the same version is already installed',
		    0);
    $defaults->add ('notify', '!',
		    'whether to send a desktop notification when the build passes/fails, use --nonotify to disable',
		    $can_notify);
    $defaults->add ('halt_on_errors', '!',
		    'whether to abort the build if an error occurs', 
		    0);
    $defaults->add ('interactive', '!',
		    '[EXPERIMENTAL] display the build output and enter interactive mode if an error occurs', 
		    0);
    $defaults->add ('maintainers', 's',
		    'file containing the list of maintainers for each spec file');
    $defaults->add ('mail_errors_to', 's',
		    'email address to send build error reports to');
    $defaults->add ('mail_errors_cc', 's',
		    'email address to Cc build error reports');
    $defaults->add ('date_format', 's',
		    'string passed on the command line to the date(1) command for calculating the suffix for the Release tag in nightly builds',
		    "%y%m%d");
    $defaults->add ('pkgformat', 's',
		    'Format of Solaris packages: filesystem or datastream',
		    'filesystem');
    $defaults->add ('build_engine', 's',
		    'The build engine to use.',
		    $build_engine);
    $defaults->add ('download', '!',
		    'download missing sources as needed.  requires wget',
		    0);
    $defaults->add ('download_to', 's',
		    'save downloaded files in the given directory.',
		    "%{topdir}/SOURCES");
    $defaults->add ('source_mirrors', 's',
		    'comma-separated list of mirror sites for source downloads');
    $defaults->add ('rmlog', '!',
                   'whether to remove the log file with each build',
                   0);
}

# --------- utility functions ----------------------------------------------
my $arch;
my $os;
my $os_rel;
my $os_build;

sub find_in_path ($) {
    my $executable = shift;
    my $PATH = $ENV{PATH};
    $PATH = "/bin:/usr/bin" unless defined $PATH;
    my @path = split /[:]/, $PATH;
    foreach my $dir (@path) {
	if ( -x "$dir/$executable" ) {
	    return "$dir/$executable";
	}
    }
    return undef;
}

# return 1 if the current user has the Software Installation profile
# return 0 otherwise
my $cache_can_install;
sub can_install () {
    return $cache_can_install if defined ($cache_can_install);
    my $prof_sw_inst=`/bin/profiles | nl -s: | grep 'Software Installation' | cut -f1 -d:`;
    my $prof_pri_adm=`/bin/profiles | nl -s: | grep 'Primary Administrator' | cut -f1 -d:`;
    my $prof_basic_usr=`/bin/profiles | nl -s: | grep 'Basic Solaris User' | cut -f1 -d:`;
    chomp ($prof_sw_inst);
    chomp ($prof_pri_adm);
    chomp ($prof_basic_usr);
    $prof_sw_inst = 0 if $prof_sw_inst eq "";
    $prof_pri_adm = 0 if $prof_pri_adm eq "";
    $prof_basic_usr = 0 if $prof_basic_usr eq "";
    if ($prof_sw_inst or $prof_pri_adm) {
	if ($prof_basic_usr) {
	    if ($prof_basic_usr < $prof_pri_adm) {
		msg_warning (0, "The \"Primary Administrator\" profile should appear");
		msg_warning (0, "before \"Basic Solaris User\"");
		$cache_can_install=0;
	    } elsif ($prof_basic_usr < $prof_sw_inst) {
		msg_warning (0, "The \"Software Installation\" profile should appear");
		msg_warning (0, "before \"Basic Solaris User\"");
		$cache_can_install=0;
	    } else {
		# Basic Solaris User is after Primary Adminstrator / Software Installation
		$cache_can_install=1
	    }
	} else {
	    # No Basic Solaris User profile found but
	    # either Primary Adminstrator or Software Installation was found
	    $cache_can_install=1;
	}
    } else {
	# No Primary Adminstrator or Software Installation
	$cache_can_install=0;
    }
    return $cache_can_install;
}

sub cannot_install_error ($$;$) {
    my $mode = shift;
    my $doing_what = shift;
    my $other_mode = shift;

    print "You need the Software Installation or the Primary Administrator profile\n";
    print "in order to install or remove packages.  See the profiles(1) and user_attr(4)\n";
    print "man pages for more information\n\n";
    print "The \"$mode\" command involves $doing_what packages.\n";
    print "Cannot continue.\n\n";
    if (defined ($other_mode)) {
	print "You may want to try the \"$other_mode\" command instead.\n";
	print "Use \"pkgtool --help\" for more information about the \"$other_mode\" command\n";
    }
    exit (1);
}

my $wget;
sub wget_in_path () {
    msg_info (2, "Looking for wget");
    if (not defined ($wget)) {
	$wget = find_in_path ("wget");
	if (not defined ($wget)) {
	    msg_warning (0, "wget is not found in the PATH, automatic downloads are not possible");
	    $wget = "-";
	    return 0;
	}
	msg_info (2, "Found $wget");
	
	# find out which option to use for the disabling checking ssl certs
	my $check_cert_option = `$wget --help 2>&1 | grep 'check.*cert'`;
	if ($check_cert_option =~ /--no-check-certificate/s) {
	    $wget = "$wget --no-check-certificate";
	} elsif ($check_cert_option =~ /--sslcheckcert/s) {
	    $wget = "$wget --sslcheckcert=0";
	}
	return 1;
    } elsif ($wget ne "-") {
	return 1;
    } else {
	return 0;
    }
}

# init: set up some global variables.
# call this before any other function.
sub init () {
    $topdir = rpm_spec::get_topdir ($build_engine, \@predefs);
    $arch = `uname -p`;
    chomp ($arch);
    if ($arch eq "unknown") {
	$arch = `uname -m`;
	chomp ($arch);
	if ($arch eq 'i586') {
	    $arch = 'i386';
	} elsif ($arch eq 'i686') {
	    $arch = 'i386';
	}
    }
    
    $os = `uname -s`;
    chomp ($os);
    $os = lc($os);
    $os_rel = `uname -r`;
    chomp ($os_rel);
    if ($os eq 'sunos') {
	if ($os_rel =~ /^5\./) {
	    $os = 'solaris';
	}
	$os_build = `uname -v`;
	chomp ($os_build);
	$os_build =~ s/.*_([0-9]+).*/$1/;
    }

    my $uid;
    if (-x "/usr/xpg4/bin/id") {
	$uid = `/usr/xpg4/bin/id -u`;
	chomp ($uid);
    } else {
	$uid = `LC_ALL=C /usr/bin/id`;
	chomp ($uid);
	$uid =~ s/^[^=]+=([0-9]+)\(.*$/$1/;
    }
    
    if ($uid eq (getpwnam($logname))[2]) {
	$_homedir = (getpwnam($logname))[7];
    } else {
	# logname is incorrect, look up the uid
	$logname = (getpwuid($uid))[0];
	$_homedir = (getpwuid($uid))[7];
    }

    if ($os eq "solaris") {
	$build_engine_name = "pkgbuild";
	$build_engine = $pkgbuild_path;
    } elsif (defined (find_in_path ("rpmbuild"))) {
	$build_engine = "rpmbuild";
	$build_engine_name = "rpmbuild";
    } elsif (defined (find_in_path ("rpm"))) {
	$build_engine = "rpm";
	$build_engine_name = "rpm";
    } else {
	$build_engine_name = "pkgbuild";
	$build_engine = $pkgbuild_path;
    }
    if (defined ($ENV{PKGBUILD_IPS_SERVER}) or
	(defined ($ips_utils) and $ips_utils->is_depotd_enabled())) {
	$ips = 1;
	$svr4 = undef;
    } else {
	$ips = undef;
	$svr4 = 1;
    }
}

# return the name of the log file given the id of the spec file
sub get_log_name ($) {
    my $spec_id = shift;
    my $spec = $specs_to_build[$spec_id];
    my $spec_file = $spec->get_file_name ();
    my $base_name = $spec->get_base_file_name ();
    my $log_name = "$base_name";
    $log_name =~ s/(\.spec|)$/.log/;
    
    return $log_name;
}

my $current_log;

sub open_log ($) {
    my $log_filename = shift;
    
    if (not defined ($log_filename)) {
	return;
    }
    
    if (defined ($current_log)) {
	if ($current_log ne $log_filename) {
	    close LOG_FILE;
	}
    }
    
    $current_log = undef;
    
    if (! open LOG_FILE, ">>$log_filename") {
	msg_warning (0, "Failed to open log file $log_filename for writing");
	return;
    }
    
    $current_log = $log_filename;
    
    msg_log ("--- date stamp --- " . `date`);
}

sub log_cmdout_starts () {
    msg_log ("--- command output follows --- started at " . `date`);
    close LOG_FILE;
    $current_log = undef;
}

sub log_cmdout_ends ($) {
    my $log_filename = shift;
    
    if (not defined ($log_filename)) {
	return;
    }
    
    if (defined ($current_log)) {
	if ($current_log ne $log_filename) {
	    close LOG_FILE;
	}
    }
    
    $current_log = undef;
    
    if (! open LOG_FILE, ">>$log_filename") {
	msg_warning (0, "Failed to open log file $log_filename for writing");
	return;
    }
    
    $current_log = $log_filename;
    
    msg_log ("--- command output ends --- finished at " . `date`);
}

sub close_log () {
    msg_log ("--- log ends --- " . `date`);
    close LOG_FILE;
    $current_log = undef;
}

sub print_message ($$) {
    my $min_verbose = shift;
    my $message = shift;
    
    chomp $message;
    
    my $verbose = $defaults->get ('verbose');
    if ($verbose > $min_verbose) {
	print "$message\n";
    }
    if (defined ($current_log)) {
	print LOG_FILE "$message\n";
    }
}

sub msg_error ($) {
    my $message = shift;
    
    print_message (0, "ERROR: $message");
    my $halt_on_errors = $defaults->get ('halt_on_errors');
    if ($halt_on_errors > 0) {
	print "ERROR: Exiting...\n";
	exit (255);
    } else {
	my $interactive_mode = $defaults->get ('interactive');
	if ($interactive_mode) {
	    print "Would you like to continue? (yes/no) [yes]";
	    my $ans = <STDIN>;
	    chomp ($ans);
	    $ans = lc($ans);
	    if ($ans ne "y" and $ans ne "yes" and $ans ne "") {
		exit(254);
	    }
	    return 1;
	}
    }
    return 0;
}

sub msg_warning ($$) {
    my $min_verbose = shift;
    my $message = shift;
    
    print_message ($min_verbose, "WARNING: $message");
}

sub msg_info ($$) {
    my $min_verbose = shift;
    my $message = shift;
    
    print_message ($min_verbose, "INFO: $message");
}

sub msg_debug ($$) {
    my $min_debug = shift;
    my $message = shift;
    
    my $debug_level = $defaults->get ('debug');
    if ($debug_level > $min_debug) {
	print "DEBUG: $message\n";
    }
}

sub msg_log ($) {
    my $message = shift;
    
    if (defined ($current_log)) {
	print LOG_FILE "$message\n";
    }
}

sub fatal ($) {
    my $message = shift;
    
    print STDERR "$message\n";
    exit (1);
}

sub find_file ($) {
    my $glob = shift;
    
    my @files = `find $glob 2>&1`;
    if ($? == 0) {
	my $file = $files[$#files];
	chomp $file;
	return ($file);
    }
    
    return (undef);
}

sub mail_log ($) {
    my $spec_id = shift;
    my $address;
    my $cc;
    my $log_name = get_log_name ($spec_id);
    my $spec_name = $specs_to_build[$spec_id]->get_name ();
    my $base_name = $log_name;
    $log_name =~ s/\.log$//;
    
    my $mail_errors_file = $defaults->get ('maintainers');
    if (defined ($mail_errors_file)) {
	$address = get_address_from_file ($mail_errors_file, $base_name);
    }
    
    my $mail_errors_to = $defaults->get ('mail_errors_to');
    if (not defined ($address)) {
	$address = $mail_errors_to;
    }

    my $mail_errors_cc = $defaults->get ('mail_errors_cc');
    if (not defined ($address)) {
	$address = $mail_errors_cc;
    } else {
	$cc = $mail_errors_cc;
    }

    if (not defined ($address)) {
	return;
    }
    
    my $log_file = get_log_name ($spec_id);
    msg_info (1, "emailing the build log to $address");

    my $subject;
    
    my $prodname = $defaults->get ('prodname');
    if (defined ($prodname)) {
	$subject = "BUILD FAILED ($prodname): $spec_name";
    } else {
	$subject = "BUILD FAILED: $spec_name";	
    }
 
    my $the_log_dir = $defaults->get ('logdir');
    my $the_logdir_url = $defaults->get ('logdir_url');
    my $log_pointer;
    if (defined ($the_logdir_url)) {
	$log_pointer = "$the_logdir_url/${log_name}.log";
    } else {
	$log_pointer = "$the_log_dir/${log_file}";
    }

    if (defined ($cc)) {
	`( echo "Full log: $log_pointer" ; echo; echo "--- tail of the log follows ---"; echo; tail -100 $the_log_dir/$log_file ) | mailx -s "$subject" -c "$cc" $address`;
    } else {
	`( echo "Full log: $log_pointer" ; echo; echo "--- tail of the log follows ---"; echo; tail -100 $the_log_dir/$log_file ) | mailx -s "$subject" $address`;
    }
}

sub get_address_from_file ($$) {
    my $fname = shift;
    my $pkg = shift;
    
    $pkg =~ s/\.spec$//;
    
    if (! open ADDR_FILE, "<$fname") {
        msg_warning (0, "Could not open file $fname");
	return undef;
    }
    my @lines = <ADDR_FILE>;
    my @addresses = grep /^$pkg:/, @lines;
    my $address = $addresses[0];
    if (not defined ($address)) {
	return undef;
    }
    $address =~ s/^$pkg://;
    $address =~ s/\s//g;
    close ADDR_FILE;
    return $address;
}

# I couldn't find a standard perl implementation of mkdir -p so
# this is simple a recursive implementation
sub mkdir_p ($);

sub mkdir_p ($) {
    my $dir = shift;

    if (-d $dir) {
	return 1;
    }

    my $pdir = dirname ($dir);
    if (! -d $pdir) {
	mkdir_p ($pdir) or return 0;
    }

    mkdir ($dir) or return 0;
    return 1;
}

# --------- functions to process the command line args ---------------------
my @specs_to_read = ();

sub add_spec ($) {
    my $spec_name = shift;

    @specs_to_read = (@specs_to_read, $spec_name);
}

sub read_spec ($) {
    my $spec_name = shift;
    
    my $spec;
    
    my $rpm_target = $defaults->get ('target');
    if (defined $rpm_target) {
	@predefs = (@predefs, "_target $rpm_target");
    }

    if (-f $spec_name) {
	$spec = rpm_spec->new ($spec_name, \@predefs);
	# ignore duplicate specs
	return if defined ($all_specs{$spec->get_file_name()});
    } elsif (-f "${spec_name}.spec") {
	$spec = rpm_spec->new ("${spec_name}.spec", \@predefs);
	# ignore duplicate specs
	return if defined ($all_specs{$spec->get_file_name()});	
    } else {
	if (not $spec_name =~ /^\//) {
	    my @the_spec_dirlist = split /[:]/, $defaults->get ('specdirs');
	    foreach my $specdir (@the_spec_dirlist) {
		next if not defined $specdir;
		$spec = rpm_spec->new ("$specdir/$spec_name", \@predefs);
		last if defined $spec;
	    }
	    if (not defined $spec) {
		foreach my $specdir (@the_spec_dirlist) {
		    next if not defined $specdir;
		    $spec = rpm_spec->new ("$specdir/${spec_name}.spec", \@predefs);
		    last if defined $spec;
		}
	    }
	}
    }
    
    if (not defined ($spec)) {
	msg_error ("$spec_name not found\n");
    } else {
	my $this_spec_id = $spec_counter ++;
	$specs_to_build[$this_spec_id] = $spec;

	$build_status[$this_spec_id] = 'NOT_BUILT';
	$status_details[$this_spec_id] = '';
	$all_specs{$spec->get_file_name ()} = $this_spec_id;
    }
}

sub process_args {
    my $arg = shift;
    
    if ($arg =~ /^--with-(.*)/) {
	process_with ("with", $1);
	return;
    } elsif ($arg =~ /^--without-(.*)/) {
	process_with ("without", $1);
	return;
    } elsif ($arg =~ /^-/) {
	fatal ("Unknown option: $arg\n");
    }

    if (not defined ($build_command)) {
	if (($arg ne "build") and ($arg ne "build-order") and
	    ($arg ne "build-only") and ($arg ne "build-install") and
	    ($arg ne "install-pkgs") and ($arg ne "prep") and
	    ($arg ne "uninstall-pkgs") and ($arg ne "spkg") and
	    ($arg ne "publish-pkgs") and
	    ($arg ne "install-order") and ($arg ne "download")) {
	    msg_error ("unknown command: $arg");
	    usage (1);
	}
	$build_command = $arg;
    } else {
	add_spec ($arg);
    }
}

my $read_rc = 1;

sub process_pkgformat ($$) {
    shift;
    my $pkgformat = shift;

    if ($pkgformat =~ /(ds|datastream|fs|filesystem)/) {
	$defaults->set ('pkgformat', $pkgformat);
    } else {
	msg_error ("invalid value for --pkgformat");
	exit (1);
    }
}

sub process_with ($$) {
    my $with = shift;
    my $opt = shift;
    
    if ($with ne "with" and $with ne "without") {
	die ("Internal error in sub process_with()");
    }
    my $optname = $opt;
    $optname =~ tr /\-/_/;
    push (@predefs, "_${with}_${optname} --${with}-${opt}");
}

my $version_printed = 0;
sub print_version () {
    $version_printed = 1;
    print "$myname version $myversion\n";
}

sub set_ips($) {
	msg_info (0,"IPS packages will be installed by default from $ips_server");
	if (not defined ($ips_utils)) {
	    msg_error ("Cannot find IPS on this system, reverting to SVr4 packaging");
	    $svr4 = shift;
	    $ips = undef;
	    return;
	}
	$ips = shift;
	$svr4 = undef;
}

sub set_svr4($) {
	msg_info (0,"SVr4 packages will be installed by default");
	$svr4 = shift;
	$ips = undef;
}

sub process_options {
    
    my $default_topdir = $topdir;

    Getopt::Long::Configure ("bundling");
      
    our $opt_good_build_dir;
    our $opt_good_rpms_copy_dir;
    our $opt_live_summary = 1;
    our $verbose = $defaults->get ('verbose');

    $defaults->readrc ("$_homedir/.pkgtoolrc");
    $defaults->readrc ('./.pkgtoolrc');

    GetOptions ('v|verbose+' => \$verbose,
		'debug=n' => sub { shift; $defaults->set ('debug', shift); },
		'version' => \&print_version,
		'q|quiet' => sub { $verbose = 0; },
		'specdirs|specdir|spec|specs|S=s' => sub { shift; $defaults->set ('specdirs', shift); },
		'halt-on-errors!' => sub { shift; $defaults->set ('halt_on_errors', shift); },
		'mail-errors-to=s' => sub { shift; $defaults->set ('mail_errors_to', shift); },
		'mail-errors-cc=s' => sub { shift; $defaults->set ('mail_errors_cc', shift); },
		'mail-errors-file|maintainers=s' => sub { shift; $defaults->set ('maintainers', shift); },
		'prodname=s' => sub { shift; $defaults->set ('prodname', shift); },
		'sourcedirs|sourcedir|src|srcdirs|srcdir|sources|source|s=s'  => sub { shift; $defaults->set ('sourcedirs', shift); },
		'logdir|log|l=s' => sub { shift; $defaults->set ('logdir', shift); },
		'summary-log|report=s' => sub { shift; $defaults->set ('summary_log', shift); },
		'live-summary|live!',
		'summary-title|report-title=s' => sub { shift; $defaults->set ('summary_title', shift); },
		'rcfile=s' => sub { shift; my $dummy = shift; $read_rc=0; $defaults->readrc ($dummy) or msg_error ("Config file not found: $dummy"); },
		'logdir-url=s' => sub { shift; $defaults->set ('logdir_url', shift); },
		'rpm-url=s' => sub { shift; $defaults->set ('rpm_url', shift); },
		'srpm-url=s' => sub { shift; $defaults->set ('srpm_url', shift); },
		'target=s' => sub { shift; $defaults->set ('target', shift); },
		'deps!' => sub { shift; $defaults->set ('deps', shift); },
		'notify!' => sub { shift; $defaults->set ('notify', shift); },
		'autodeps!' => sub { shift; $defaults->set ('autodeps', shift); },
		'rc!' => sub { shift; $read_rc = shift; if (not $read_rc) { $defaults->norc(); } },
		'interactive!' => sub { shift; $defaults->set ('interactive', shift); },
		'tarballdirs|tarballdir|tar|tarballs|tardirs|t=s' => sub { shift; $defaults->set ('tarballdirs', shift); },
		'good-build-dir|substitutes=s',
		'good-rpms-copy-dir=s',
		'nightly!' => sub { shift; $defaults->set ('nightly', shift); },
		'define=s' => sub { 
		    shift; 
		    my $def = shift;
		    @predefs = ( @predefs, $def );
		    $topdir = rpm_spec::get_topdir ($build_engine, \@predefs);
		},
		'update!' => sub { shift; $defaults->set ('update_always', shift); },
		'update-if-newer!' => sub { shift; $defaults->set ('update_if_newer', shift); },
		'with=s' => \&process_with,
		'without=s' => \&process_with,
		'pkgformat=s' => \&process_pkgformat,
		'date|date-format|nightly-date-format=s' => sub { shift; $defaults->set ('date_format', shift); },
		'patchdirs|patchdir|patch|patches|p=s' => sub { shift; $defaults->set ('patchdirs', shift); },
		'rpmdir|rpm|topdir|r=s' => sub { 
		    shift; 
		    $topdir = shift;
		    @predefs = ( @predefs, "_topdir $topdir" );
		},
		'full-path' => \$full_path,
		'help' => \&usage,
		'dumprc' => sub { $defaults->dumprc (); exit (0); }, 
		'pkgbuild' => sub { 
		    $defaults->set ('build_engine', 'pkgbuild');
		    $build_engine_name = 'pkgbuild';
		    $build_engine = $pkgbuild_path;
		    $topdir = rpm_spec::get_topdir ($build_engine, \@predefs);
		},
		'rpmbuild' => sub {
		    if (defined (find_in_path ('rpmbuild'))) {
			$build_engine = 'rpmbuild';
			$build_engine_name = 'rpmbuild';
		    } elsif (defined (find_in_path ('rpm'))) {
			$build_engine = 'rpm';
			$build_engine_name = 'rpm';
		    } else {
			fatal ('rpm/rpmbuild not found in the PATH');
		    }
		    $defaults->set ('build_engine', $build_engine_name);
		    $topdir = rpm_spec::get_topdir ($build_engine, \@predefs);
		},
		'download' => sub { shift; $defaults->set ('download', shift); },
		'download-to=s' => sub {
		    shift;
		    $defaults->set ('download_to', shift);
		    $defaults->set ('download', 1);
		},
		'dry-run' => sub { $download_test = 1; },
		'source_mirrors=s' => sub {
		    shift;
		    $defaults->set ('source_mirrors', shift);
		},
		'ips' => sub { set_ips(1); },
		'svr4' => sub { set_svr4(1); },
                'rmlog' => sub { shift; $defaults->set ('rmlog', shift); },
		'<>' => \&process_args);

    if ($topdir ne $default_topdir) {
	@predefs = ( @predefs, "_topdir $topdir" );
    }
    for my $spec_name (@specs_to_read) {
	read_spec ($spec_name) unless not defined ($spec_name);
    }
    $the_good_build_dir = $opt_good_build_dir;
    $the_good_rpms_copy_dir = $opt_good_rpms_copy_dir;
    $live_summary = $opt_live_summary;
      
    $defaults->set ('verbose', $verbose);
}

sub usage (;$) {
    my $retval = shift;
    if (not defined ($retval)) {
	$retval = 0;
    } elsif ($retval eq "help" or $retval eq "h") {
	$retval = 0;
    }
    
    print << "EOF";
pkgtool [options] [command] specs...
	
Options:

  General:
	
    -v|--verbose:    
	          Increase verbosity: the more -v's the more diag messages.

    -q|--quiet:
                  Silent operation.

    --halt-on-errors:
                  Halt on the first build error, do not attempt to continue.

    --rcfile=file
                  Read default configuration from file.
                  Default: ./.pkgtoolrc, ~/.pkgtoolrc

    --norc
                  Ignore the default rc files.

    --dumprc
                  Print the current configuration in a format suitable
                  for an rc file, then exit.

    --download

                  Automatically download sources if not found in the local
                  search paths (requires wget)  Specify your proxy servers
                  using the http_proxy and ftp_proxy environment variables.

    --download-to=dir

                  Save downloaded files in dir.  By default, files are
                  downloaded to $topdir/SOURCES.  Implies --download.

    --interactive  [EXPERIMENTAL]

	          Interactive mode: pkgbuild/rpm output is displayed on
                  the standard output; pkgbuild is executed in interactive
                  mode which makes it start a subshell if the build fails

    --ips

		  Install IPS packages by default to local repository
		  http://localhost:80/

    --svr4

		  Install SVr4 packages by default.

    --rmlog

                  Automatically remove the log file with each build

    --dry-run

                  (Download mode) test if the Source urls in the spec
                  file(s) point to existing files, but do not download them.

  Directories and search paths:

    --specdirs=path, --spec=path:
                  Specify a colon separated list of directories to search
                  for spec files in

    --tarballdirs=path, --tarballs=path, --tar=path:
                  Specify a colon separated list of directories to search
                  for tarballs in

    --sourcedirs=path, --src=path:
                  Specify a colon separated list of directories to search
                  for additional source files in

    --patchdirs=path, --patches=path, --patch=path
                  Specify a colon separated list of directories to search
                  for patches (source diffs) in
 
    --topdir=dir
                  Use dir as the rpm base directory (aka %topdir, where the
                  SPECS, SOURCES, RPMS, SRPMS, BUILD directories are found).
                  Default: $topdir

    --logdir=dir, --log=dir:
                  Write build logs to dir.

  Options controlling the build:
                  
    --nodeps, --deps:
                  Ignore/verify dependencies before building a component.
                  Default: --deps

    --autodeps:
                  Attempt to find spec files for missing dependencies
                  in spec file search path (see --specdirs) and add them
                  to the build as needed.

    --with foo, --without foo, --with-foo, --without-foo
                  This option is passed on to rpm/pkgbuild as is.  Use it
                  for enabling/disabling conditional build options.

    --target=arch
                  This option is passed on to rpm/pkgbuild as is.

    --pkgformat={filesystem|fs|datastream|ds}
                  Create Solaris packages in filesystem or datastream format.
                  This option is ignored when running on Linux.
                  Default: filesystem

    --nightly, --nonightly:
                  Suffix/Don't suffix the rpm Release with the current date.
		  Default: --nonightly;  See also: --date-format
    
    --date-format=format, --date=format:
                  Use "date +format" to generate the date suffix for
	          the nightly builds.  Default: %y%m%d

  Reporting:

    --notify, --nonotify

                  Send desktop notifications when the build of a spec
		  file passes or fails.  Default: --notify

    --mail-errors-to=address

                  Send the last few lines of the build log to address
		  if the build fails

    --report=file

                  Write a build report to file (in HTML format)

    --prodname=string

                  The name of the product as appears in the build report
    
    --full-path:
                  Print the full path to the package when running install-order

Commands:
	
    build-install  Build and install the specs listed on the command line.
                   The build order is determined by the dependencies
		   defined in the spec files.

    build          Same as build-install

    build-only     Build the specs listed on the command line, don't install
                   them.

    prep           run the %prep section of the spec files listed on the
                   command line

    spkg           create source package(s) only (no build done)

    publish-pkgs   publish the previously build packages to an IPS repository

    build-order    Print the build order of the specs listed on the
                   command line.

    install-order  Print the rpms in the order they should be installed
	
    install-pkgs   install the packages defined by the spec files listed
                   on the command line from the PKGS directory.  No build
                   is done.  Useful to install packages previously built
                   using the build-only command, or built manually using
                   pkgbuild.
	
    uninstall-pkgs Uninstall all packages defined in the spec files listed
	           on the command line. (runs rpm --erase --nodeps on
                   Linux, pkgrm on Solaris)

    download       Download the source files from the URLs specified in
                   the spec files listed on the command line.  Source
                   files found in the local search paths will not be
                   downloaded.  (See --tarballdirs, --sourcedirs,
                   --download-to)
	
specs...
	
    List of spec files to build. Either full path names or names of spec
    files in the spec directory search path.

Environment variables:

    PKGBUILD_IPS_SERVER - specify the URL for the IPS server to publish 
                   packages to.  Example: http://myserver:9000/

EOF
#' <-- (keep emacs syntax highlighting happy

    exit $retval;
}

# --------- print reports --------------------------------------------------
# plain text report on stdout and
# html report if summary log file name
# was specified on the command line
sub print_status {
    print_status_text ();
    print_status_html ();
}

sub print_live_status {
    if ($live_summary) {
	print_status_html ();
    }
}

sub print_status_text {
    my $spec_name;
    print "\nSummary:\n\n";
    printf "%32s | %11s | %s\n", "package", "status", "details";
    print "---------------------------------+-------------+-------------------------------\n";
    for (my $i = 0; $i <= $#specs_to_build; $i++) {
	$spec_name=$specs_to_build[$i]->get_name ();
	if (not defined ($spec_name)) {
	    $spec_name = $specs_to_build[$i]->get_base_file_name ();
	}
	printf "%32s | %11s | %s\n", $spec_name, $build_status[$i], 
		$status_details[$i];
    }
}

sub print_status_html {
    my $spec_name;

    my $the_summary_log = $defaults->get ('summary_log');
    if (not defined ($the_summary_log)) {
	return;
    }

    if (! open SUM_LOG, ">$the_summary_log") {
	msg_warning (0, "Failed to open file $the_summary_log for writing");
	return;
    }

    my $the_summary_title = $defaults->get ('summary_title');
    print SUM_LOG '<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Strict//EN"', "\n";
    print SUM_LOG '    "http://www.w3.org/tr/xhtml1/DTD/xhtml1-strict.dtd">', "\n";
    print SUM_LOG '<html xmlns="http://www.w3.org/1999/xhtml" xml:lang="en" lang="en">', "\n";
    print SUM_LOG "<head>\n";
    if (defined ($the_summary_title)) {
        print SUM_LOG "<title>$the_summary_title - Build report</title>\n";
    }
    else {
        print SUM_LOG "<title>Build report</title>\n";
    }
    print SUM_LOG '<meta http-equiv="Content-Type" content="text/html;charset=utf-8" />', "\n";
    print SUM_LOG '<style type="text/css">', "\n";
    print SUM_LOG 'body { background-color: #ffffff; }', "\n";
    print SUM_LOG 'table { border: 1px solid #999999; border-collapse: collapse; }', "\n";
    print SUM_LOG 'td { border: 1px solid #999999; padding: 5px; border-spacing: 0px; }', "\n";
    print SUM_LOG 'td.header { background-color: #666699; }', "\n";
    print SUM_LOG '.passed { color: #11aa11; }', "\n";
    print SUM_LOG '.failed { color: #cc1111; }', "\n";
    print SUM_LOG '.building { color: #9c23f9; }', "\n";
    print SUM_LOG '.dependency { color: #9c23f9; }', "\n";
    print SUM_LOG '.dep_failed { color: #ffa500; }', "\n";
    print SUM_LOG '.skipped { color: #00a5ff; }', "\n";
    print SUM_LOG '.other { color: #cc1111; }', "\n";
    print SUM_LOG '</style>', "\n";
    print SUM_LOG "</head>\n<body>\n";
    if (defined ($the_summary_title)) {
	print SUM_LOG "<h3>$the_summary_title</h3>\n";
    }
    print SUM_LOG "<table>\n";
    print SUM_LOG " <tr>\n";
    print SUM_LOG '  <td class="header"><b>package</b></td>', "\n";
    print SUM_LOG '  <td class="header"><b>version</b></td>', "\n";
    print SUM_LOG '  <td class="header"><b>release</b></td>', "\n";
    print SUM_LOG '  <td class="header"><b>status</b></td>', "\n";
    print SUM_LOG '  <td class="header"><b>details</b></td>', "\n";
    print SUM_LOG " </tr>\n";
    for (my $i = 0; $i <= $#specs_to_build; $i++) {
	my $log_name = get_log_name ($i);
	$spec_name=$specs_to_build[$i]->get_name ();
	my $version = $specs_to_build[$i]->get_value_of ("version");
	$version = "unknown" if not defined $version;
	my $release = $specs_to_build[$i]->get_value_of ("release");
	$release = "" if not defined $release;
	print SUM_LOG " <tr>\n";
	my $the_srpm_url = $defaults->get ('srpm_url');
	if (defined ($the_srpm_url)) {
	    if (($build_status[$i] eq "PASSED") and defined ($the_srpm_url)) {
		print SUM_LOG "  <td><a href=\"$the_srpm_url/$spec_name-$version-$release.src.rpm\">$spec_name</a></td>\n";
	    } else {
		print SUM_LOG "  <td>$spec_name</td>\n";
	    }
	} else {
	    print SUM_LOG "  <td>$spec_name</td>\n";
	}
	print SUM_LOG "  <td>", $version, "</td>\n";
	print SUM_LOG "  <td>", $release, "</td>\n";
	my $color_start = "";
	my $color_end = "";
	if ($build_status[$i] eq "PASSED") {
	    $color_start = '<span class="passed">';
	    $color_end = '</span>';
	} elsif ($build_status[$i] eq "BEING_BUILT") {
	    $color_start = '<span class="building">';
	    $color_end = '</span>';
	} elsif ($build_status[$i] eq "DEP") {
	    $color_start = '<span class="dependency">';
	    $color_end = '</span>';
	} elsif ($build_status[$i] eq "NOT_BUILT") {
	    $color_start = "";
	    $color_end = "";
	} elsif ($build_status[$i] eq "SKIPPED") {
	    $color_start = '<span class="skipped">';
	    $color_end = '</span>';
	} elsif ($build_status[$i] eq "DEP_FAILED") {
	    $color_start = '<span class="dep_failed">';
	    $color_end = '</span>';
	} elsif ($build_status[$i] eq "FAILED") {
	    $color_start = '<span class="failed">';
	    $color_end = '</span>';
	} else {
	    $color_start = '<span class="other">';
	    $color_end = '</span>';
	}
	my $the_logdir_url = $defaults->get ('logdir_url');
	if (defined ($the_logdir_url) and ($build_status[$i] ne "NOT_BUILT")) {
	    print SUM_LOG "  <td><a href=\"$the_logdir_url/$log_name\">",
	    $color_start, $build_status[$i], $color_end, "</a></td>\n";
	} else {
	    print SUM_LOG "  <td>", 
	    $color_start, $build_status[$i], $color_end,
	    "</td>\n";
	}
	if ($build_status[$i] eq "PASSED") {
	    my $the_rpm_url = $defaults->get ('rpm_url');
	    if (defined ($the_rpm_url)) {
		print SUM_LOG "  <td>package: \n";
		my @pkgs = $specs_to_build[$i]-> get_package_names ($ds);
		my $ctr = 1;
		for my $pkg (@pkgs) {
		    if ($os eq "solaris") {
			if (not $ds) {
			    $pkg = "$pkg.tar.gz";
			}
		    }
		    print SUM_LOG "    <a href=\"$the_rpm_url/$pkg\">[$ctr]</a> \n";
		    $ctr++;
		}
		print SUM_LOG "  </td>\n";
	    } else {
		print SUM_LOG "  <td>&nbsp;</td>\n";
	    }
	} else {
	    print SUM_LOG "  <td><pre>", $status_details[$i], "</pre></td>\n";
	}
	print SUM_LOG " </tr>\n";
    }
    print SUM_LOG "</table>\n";
    print SUM_LOG "<p><small>", scalar localtime(), "</small></p>\n";
    print SUM_LOG "</body>\n</html>\n";
    close SUM_LOG;
}

# --------- rpm utility functions ------------------------------------------
my %pkginfo; # cache the pkginfo results, as it's very slow
my %pkginfo_version;
sub is_provided ($) {
    my $capability = shift;

    # FIXME: this deletes the version requirements, but we really should
    # implement that
    $capability =~ s/\s.*//;

    if ($capability =~ /^\//) {
	# Requires: /path/to/file or BuildRequires: /path/to/file
	if (-e $capability) {
	    return 1;
	} else {
	    return 0;
	}
    }
    if ($os eq "solaris") {
	if (defined $pkginfo{$capability}) {
	    return $pkginfo{$capability};
	}
	my $pkg_instance = `pkginfo $capability'.*' 2>/dev/null | awk '{print \$2;}'`;
	chomp ($pkg_instance);
	my $result = ($pkg_instance ne "");
	$pkginfo{$capability} = $result;
	if ($result) {
	    my $version = `pkgparam $pkg_instance VERSION`;
	    chomp $version;
	    $version =~ s/([^,]+)(,|$).*/$1/;
            $pkginfo_version{$capability} = $version;
	}
	# if no svr4 package was found, look for an IPS package
	if (not $result and defined ($ips)) {
	    # no svr4 package (or IPS package with a legacy action) found
	    # let's look for an IPS package
	    my $pkg_out = `pkg info -l $capability 2>&1`;
	    $result = (! $?);
	    $pkginfo{$capability} = $result;
	    my $version = $pkg_out;
	    $version =~ s/^.*\n\s*Branch:\s([0-9.]+|None)\s*\n.*$/$1/s;
	    if ($version eq "None") {
		# Branch is None, look for Version instead
		$version = $pkg_out;
		$version =~ s/^.*\n\s*Version:\s([0-9.]+)\s*\n.*$/$1/s;
	    }
	    $pkginfo_version{$capability} = $version;
	}
	return $result;
    } else {
	`sh -c "rpm -q --whatprovides $capability" >/dev/null 2>&1`;
	my $result = (! $?);
	`sh -c "rpm -q $capability" >/dev/null 2>&1`;
	$result = ($result or (! $?));
    
	return ($result);
    }
}

sub is_installed ($) {
    my $pkg = shift;

    if ($os eq "solaris") {
	if (defined $pkginfo{$pkg}) {
	    return $pkginfo{$pkg};
	}
	my $pkg_instance = `pkginfo $pkg'.*' 2>/dev/null | awk '{print \$2;}'`;
	chomp ($pkg_instance);
	my $result = ($pkg_instance ne "");
	$pkginfo{$pkg} = $result;
	if ($result) {	
	    my $version = `pkgparam $pkg_instance VERSION`;
	    chomp $version;
	    $version =~ s/([^,]+)(,|$).*/$1/;
            $pkginfo_version{$pkg} = $version;
        }
	return $result;
    } else {
	`sh -c "rpm -q $pkg" >/dev/null 2>&1`;
	my $result = (! $?);
	return ($result);
    }
}

sub what_provides ($) {
    my $capability = shift;

    $capability =~ s/ .*//;

    if ($os eq "solaris") {
	return $capability . "-" . $pkginfo_version{$capability};
    } else {
	my $rpm=`sh -c "rpm -q --whatprovides $capability" 2>&1 | head -1`;
	if ($?) {
	    $rpm=`sh -c "rpm -q $capability" 2>&1 | head -1`;
	}
	chomp $rpm;
	$rpm =~ s/-[^-]+$//;
	return ($rpm);
    }
}

sub install_good_pkgs ($) {
    my $spec_id = shift;
    my $spec = $specs_to_build[$spec_id];

#FIXME!
    if ($os eq "solaris") {
	return 0;
    }

    my @rpms =  $spec->get_rpms ();
    map $_ =~ s/[0-9]*-[0-9]+\..*\.rpm/[0-9]*-[0-9]*.*.rpm/, @rpms;
    map $_ =~ s/^/$the_good_build_dir\//, @rpms;


    for (my $i = 0; $i <= $#rpms; $i++) {
	my $current_rpm = $rpms[$i];
	msg_info (1, "Looking for last known good rpm as $current_rpm");
	$rpms[$i] = find_file ($current_rpm);
	if (not defined ($rpms[$i])) {
	    msg_error ("No file matches $current_rpm");
	    return 0;
	} else {
	    msg_info (1, "Found $rpms[$i]");
	}
    }

    my $command = "rpm --upgrade -v @rpms";

    my $verbose = $defaults->get ('verbose');
    if ($verbose > 0) {
	map msg_info (0, "Installing last known good rpm: $_\n"), @rpms;
    }

    my $msg=`$command 2>&1`;

    if ($? > 0) {
	msg_error "failed to install last known good rpm: $msg";
	$status_details[$spec_id] = $status_details[$spec_id] . 
	    "; Failed to install last known good rpm: " . $msg;
	return 0;
    }

    if (defined ($the_good_rpms_copy_dir)) {
	if (mkdir_p ($the_good_rpms_copy_dir)) {
	    foreach my $rpm (@rpms) {
		copy ($rpm, $the_good_rpms_copy_dir);
	    }
	}
    }

    return 1;
}

sub do_publish_pkgs () {
    my $retval = 0;
    for (my $i = 0; $i <= $#specs_to_build; $i++) {
	$retval += publish_ips_pkg ($i);
    }
    push_incorporations($specs_to_build[0]->eval ("%_pkgmapdir"));
    return $retval;
}

sub publish_ips_pkg ($) {
    my $spec_id = shift;
    my $spec = $specs_to_build[$spec_id];
    my $retval = 0;
    my @pkgs = $spec->get_packages ();
    foreach my $pkg (@pkgs) {
	next if $pkg->is_subpkg ();
	my $script = $spec->eval ("%_pkgmapdir") . "/scripts/${pkg}_ips.sh";
	if (! -e $script) {
	    msg_error ("Script not found for publishing $pkg: $script");
	    msg_info (0, "Hint: this script is generated by pkgbuild when building IPS packages");
	    $retval++;
	    next;
	}
	msg_info (1, "Calling script $script");
	`chmod +x $script`;
	my $msg = `$script | sed -e 's/^/INFO: /'`;
	$retval += ($? != 0);
	if ($? == 0) {
	    update_incorporations ($spec_id);
	}
	print $msg;
    }
    return $retval;
}

sub install_rpms ($) {
    my $spec_id = shift;
    my $spec = $specs_to_build[$spec_id];

    my @rpms =  $spec->get_rpm_paths ();
# FIXME: is this OK?
#    map $_ =~ s/^/$topdir\//, @rpms;
    my $command = "rpm --upgrade -v @rpms";

    my $verbose = $defaults->get ('verbose');
    if ($verbose > 0) {
	map msg_info (0, "Installing $_\n"), @rpms;
    }

    my $msg=`$command 2>&1`;
    if ($? > 0) {
	msg_error "failed to install rpm: $msg";
	$build_status[$spec_id] = 'FAILED';
	$status_details[$spec_id] = $msg;
	return 0;
    }

    return 1;
}

sub install_pkgs_svr4 ($) {
    my $spec_id = shift;
    my $spec = $specs_to_build[$spec_id];
    
    my @pkgs = $spec->get_package_names ($ds);
    my $verbose = $defaults->get ('verbose');
    if ($verbose > 0) {
	map msg_info (0, "Installing $_\n"), @pkgs;
    }
    
    my $pkgsdir = $spec->get_value_of ("_topdir") . "/PKGS";
    
    my $adminfile = "/tmp/pkg.admin.$$";
    make_admin_file ($adminfile);

# FIXME: should install in dependency order
    foreach my $pkg (@pkgs) {
	my $msg;
	
	# Only install SVr4 package if --svr4 is defined
	if ( defined $svr4) {
	    if (defined ($ds)) {
		$msg=`pfexec /usr/sbin/pkgadd -a $adminfile -n -d $pkgsdir/$pkg all 2>&1`;
	    } else {
		$msg=`pfexec /usr/sbin/pkgadd -a $adminfile -n -d $pkgsdir $pkg 2>&1`;
	    }
	    
	    if ($? > 0) {
		unlink ($adminfile);
		msg_error "failed to install package: $msg";
		$build_status[$spec_id] = 'FAILED';
		$status_details[$spec_id] = $msg;
		return 0;
	    }
	}
    }
    
    unlink ($adminfile);
    return 1;
}

my %all_incorporations;
my %incorporated;

sub update_incorporations ($) {
    my $spec_id = shift;
    my $spec = $specs_to_build[$spec_id];

    my @ps = $spec->get_packages ();
    foreach my $p (@ps) {
	# subpackages are merged in the main package
	next if ($p->is_subpkg());
	my $pn = $p->get_ips_name();
	my $ips_vendor_version = $p->eval ("%{?ips_vendor_version}%{!?ips_vendor_version:%version}");
	my $version = $p->eval("%ips_component_version,%ips_build_version-$ips_vendor_version");
	# get all the incorporations the packages are included in
	my @pkg_inc = 
	    `pkg search -o pkg.shortfmri -Hl :depend:incorporate:$pn 2>/dev/null`;
	# always try entire, for older IPS clients that do not support the
	# pkg search command above
	if (not @pkg_inc) {
	    @pkg_inc = ("entire");
	}
	foreach my $incorp (@pkg_inc) {
	    chomp ($incorp);
	    if (not defined($all_incorporations{$incorp})) {
		msg_info (1, "Loading incorporation $incorp");
		$all_incorporations{$incorp} = 
		    ips_package->new_from_fmri ($incorp) or return 1;
	    }
	    msg_info (1, "Updating incorporation $incorp");
	    $all_incorporations{$incorp}->update_depend ($pn, $version);
	    if ($all_incorporations{$incorp}->changed()) {
		$incorporated{$pn} = $incorp;
	    }
	}
    }
    return 0;
}

sub push_incorporations ($) {
    my $pkgmapsdir = shift;
    foreach my $incorp (keys %all_incorporations) {
	if ($all_incorporations{$incorp}->changed()) {
	    msg_info (0, "Publishing updated incorporation $incorp");
	    $all_incorporations{$incorp}->publish($pkgmapsdir) or return 0;
	}
    }
    return 1;
}

sub install_pkgs_ips ($) {
    my $spec_id = shift;
    my $spec = $specs_to_build[$spec_id];
    
    my $auth = $ips_utils->get_pkgbuild_publisher ();
    if (not defined $auth) {
	msg_error ("Unable to identify the publisher for $ips_server");
	msg_info (0, "Hint: use \"pfexec pkg set-publisher -O $ips_server mypkgs\"");
	msg_info (0, "to define a repository called \"mypkgs\" or set the");
	msg_info (0, "PKGBUILD_IPS_SERVER environment variable to one of the URIs");
	msg_info (0, "listed by the \"pkg publisher\" command.");
	return 0;
    }

    push_incorporations($spec->eval ("%_pkgmapdir")) or return 0;

    my @pkgs = $spec->get_packages ();
    my $verbose = $defaults->get ('verbose');
    msg_info (0, "Installing package(s) from publisher $auth\n");
    
    my $all_pkgs = "";
    my $ips_vendor_version = $spec->eval ("%{?ips_vendor_version}%{!?ips_vendor_version:%version}");
    my $version = $spec->eval("%ips_component_version,%ips_build_version-$ips_vendor_version");
    my $prefix = $ips_utils->get_publisher_setting ($auth, 'prefix');
    # FIXME: hack?
    $prefix = $auth unless defined ($prefix);
    my %incorps;
    foreach my $pkg (@pkgs) {
	# subpackages are merged in the main package
	next if ($pkg->is_subpkg());
	my $pn = $pkg->get_ips_name();
	# on older than snv_129 versions of pkg,
	# if the package is in an incorporation that we updated,
	# only try to install the updated incorporation, it will
	# pull the updated package, if the package is installed
	if (defined ($incorporated{$pn})) {
	    if (not defined ($incorps{$incorporated{$pn}})) {
		$all_pkgs = "$all_pkgs " . 
		    $all_incorporations{$incorporated{$pn}}->get_fmri();
		# don't add the same incorporation name to the args again
		$incorps{$incorporated{$pn}} = 1;
	    }
	}
	# on newer version of pkg, we also need to specify the package names
	if ($os_build >= 129) {
	    msg_info (2, "New IPS, add package name to command line");
	    $all_pkgs = "$all_pkgs pkg://$auth/$pn\@$version";
	}
    }

    my $msg;
    if ($all_pkgs ne "") {
	msg_info (1, "Running pfexec pkg refresh $auth");
	$msg=`pfexec pkg refresh $auth 2>&1`;
	if ( $? > 0 ) {
	    msg_error "pkg refresh failed when installing package: $msg";
	    $build_status[$spec_id] = 'FAILED';
	    $status_details[$spec_id] = $msg;
	    return 0;
	} else {
	    my @msg_lines = split /\n/, $msg;
	    foreach my $msg_line (@msg_lines) {
		msg_info (1, $msg_line);
	    }
	}
	msg_info (1, "Running pfexec pkg install --no-refresh $all_pkgs");
	$msg=`pfexec pkg install --no-refresh $all_pkgs 2>&1`;
	if ( $? > 0 ) {
	    msg_error "failed to update IPS incorporations: $msg";
	    $build_status[$spec_id] = 'FAILED';
	    $status_details[$spec_id] = $msg;
	    return 0;
	} else {
	    my @msg_lines = split /\n/, $msg;
	    foreach my $msg_line (@msg_lines) {
		msg_info (1, $msg_line);
	    }
	}
    }

    $all_pkgs = "";
    # now install all packages that are not already installed
    # (already installed packages are updated when we update
    # the incorporations)
    foreach my $pkg (@pkgs) {
	# subpackages are merged in the main package
	next if ($pkg->is_subpkg());
	my $pn = $pkg->get_ips_name();
	if ($ips_utils->is_installed($pn)) {
	    next;
	}
	my $prefix = $ips_utils->get_publisher_setting ($auth, 'prefix');
	# FIXME: hack?
	$prefix = $auth unless defined ($prefix);
	$all_pkgs = "$all_pkgs pkg://$auth/$pn\@$version";
    }
    if ($all_pkgs ne "") {
	msg_info (1, "Running pfexec pkg refresh $auth");
	$msg=`pfexec pkg refresh $auth 2>&1`;
	if ( $? > 0 ) {
	    msg_error "pkg refresh failed when installing package: $msg";
	    $build_status[$spec_id] = 'FAILED';
	    $status_details[$spec_id] = $msg;
	    return 0;
	} else {
	    my @msg_lines = split /\n/, $msg;
	    foreach my $msg_line (@msg_lines) {
		msg_info (1, $msg_line);
	    }
	}
	msg_info (1, "Running pfexec pkg install --no-refresh $all_pkgs");
	$msg=`pfexec pkg install --no-refresh $all_pkgs 2>&1`;
	if ( $? > 0 ) {
	    msg_error "failed to update IPS packages: $msg";
	    $build_status[$spec_id] = 'FAILED';
	    $status_details[$spec_id] = $msg;
	    return 0;
	} else {
	    my @msg_lines = split /\n/, $msg;
	    foreach my $msg_line (@msg_lines) {
		msg_info (1, $msg_line);
	    }
	}
    }
	
    return 1;
}

sub print_rpms ($) {
    my $spec_id = shift;
    my $spec = $specs_to_build[$spec_id];
    my @rpms;

    if ($full_path) {
	@rpms = $spec->get_rpm_paths ();
    } else {
	@rpms = $spec->get_rpms ();
    }
    map print("$_\n"), @rpms;
}

sub print_pkgs ($) {
    my $spec_id = shift;
    my $spec = $specs_to_build[$spec_id];

    my @pkgs = $spec->get_package_names ($ds);
    if ($full_path) {
	my $pkgdir = $spec->get_value_of ('_topdir') . "/PKGS";
	map print("$pkgdir/$_\n"), @pkgs;
    } else {
	map print("$_\n"), @pkgs;
    }
}

sub print_spec ($) {
    my $spec_id = shift;
    my $spec = $specs_to_build[$spec_id];

    print $spec->get_file_name() . "\n";
}

sub push_to_pkg_list ($) {
    my $spec_id = shift;
    my $spec = $specs_to_build[$spec_id];

    my @pkgs =  $spec->get_package_names ($ds);
    foreach my $pkg (@pkgs) {
	my @d = ($spec_id, $pkg);
	unshift (@pkg_list, \@d);
    }
}

# --------- dependency checking code ---------------------------------------
sub warn_always ($$$) {
    my $spec_name = shift;
    my $dep = shift;
    my $reason = shift;

    if ($reason eq "DEP_FAILED") {
	msg_warning (0, "skipping package $spec_name: required package $dep failed");
    } elsif ($reason eq "NOT_FOUND") {
	msg_warning (0, "skipping package $spec_name: required package $dep not installed");
	msg_warning (0, "and no spec file specified on the command line provides it");
    }
}

sub warn_once ($$$) {
    my $spec_name = shift;
    my $dep = shift;
    my $reason = shift;

    if ($reason eq "DEP_FAILED") {
	# should not happen
	msg_error ("assertion failed: warn_once / DEP_FAILED");
    } elsif ($reason eq "NOT_FOUND") {
	if (not defined ($warned_about{$dep})) {
	    msg_warning (0, "$dep is required but not found");
	    $warned_about{$dep}=1;
	}
    }
}

sub warn_never ($$$) {
}

sub get_dependencies ($$@) {
    my $spec_id = shift;
    my $build_only = shift;
    my @packages = @_;
    my $spec = $specs_to_build[$spec_id];

    my @dependencies = ();
    my @this_pkg_requires;
    if (not $build_only) {
	foreach my $pkg (@packages) {
	    @this_pkg_requires = $pkg->get_array ('requires');
	    next if not @this_pkg_requires or not defined $this_pkg_requires[0];
	    msg_debug (3, "adding \"@this_pkg_requires\" to the dependencies of $spec");
	    push (@dependencies, @this_pkg_requires);
	}
    }

    foreach my $pkg (@packages) {
	@this_pkg_requires = $pkg->get_array ('buildrequires');
	next if not @this_pkg_requires or not defined $this_pkg_requires[0];
	msg_debug (3, "adding \"@this_pkg_requires\" to the dependencies of $spec");
	push (@dependencies, @this_pkg_requires);
    }

    if (not $build_only) {
	foreach my $pkg (@packages) {
	    @this_pkg_requires = $pkg->get_array ('prereq');
	    next if not @this_pkg_requires or not defined $this_pkg_requires[0];
	    msg_debug (3, "adding \"@this_pkg_requires\" to the dependencies of $spec");
	    push (@dependencies, @this_pkg_requires);
	}
    }

    return @dependencies;
}

sub check_dependency ($$&&@) {
    my $spec_id = shift;
    my $capability = shift;
    my $recursive_callback = shift;
    my $warning_callback = shift;
    my @rec_cb_opts = @_;
    my $spec_name = $specs_to_build[$spec_id]->get_name ();

    my $result = 1;

    if (not defined ($capability)) {
	return 1;
    }

    if ((defined ($provider{$capability})) and
	($provider{$capability} == $spec_id)) {
	return 1;
    }

    if (defined $provider{$capability}) {
	if ($build_status[$provider{$capability}] eq "PASSED") {
	    return 1;
	} elsif ($build_status[$provider{$capability}] eq "SKIPPED") {
	    if (!is_provided ($capability)) {
		&$warning_callback ($spec_name, $capability, "DEP_FAILED");
		return 0;
	    }
	    return 1;
	} elsif ($build_status[$provider{$capability}] ne "NOT_BUILT") {
	    if (!is_provided ($capability)) {
		&$warning_callback ($spec_name, $capability, "DEP_FAILED");
		return 0;
	    }
	    return 1;
	}

	msg_info (1, "$spec_name requires $capability");
	my $save_log_name = $current_log;
	my $result = &$recursive_callback ($provider{$capability}, @rec_cb_opts);
	open_log ($save_log_name);
	return $result;
    } elsif (!is_provided ($capability)) {
	return 0 if $capability =~ /^\//;
	my $autodeps = $defaults->get ('autodeps');
	my $autospec;
	if ($autodeps) {
	    # try to find a spec file based on the package name
	    msg_info (1, "Trying to find spec file for $capability");

	    my @the_spec_dirlist = split /[:]/, $defaults->get ('specdirs');
	    my $specname = $capability . ".spec";

	    msg_info (2, "Looking for $specname");
	    foreach my $specdir (@the_spec_dirlist) {
		if (-f "$specdir/$specname") {
		    msg_info (3, "    ... in $specdir");
		    $autospec = "$specdir/$specname";
		    last;
		}
	    }

	    if (not defined ($autospec) and $capability =~ /-(devel|root)$/) {
		msg_info (2, "Looking for $specname");
		foreach my $specdir (@the_spec_dirlist) {
		    if (-f "$specdir/$specname") {
			msg_info (3, "    ... in $specdir");
			$autospec = "$specdir/$specname";
			last;
		    }
		}
	    }
	    
	    if (defined ($autospec)) {
		msg_info (1, "Found $autospec");
		my $spec = rpm_spec->new ($autospec, \@predefs);
		# ignore duplicate specs
		if (not defined ($all_specs{$spec->get_file_name()})) {
		    my $this_spec_id = $spec_counter ++;
		    $specs_to_build[$this_spec_id] = $spec;
		    $build_status[$this_spec_id] = 'NOT_BUILT';
		    $status_details[$this_spec_id] = '';
		    $all_specs{$spec->get_file_name ()} = $this_spec_id;
		    my $fname = $spec->get_file_name ();
		    my @used_specs = get_specs_used ($fname);
		    if (@used_specs) {
			foreach my $subspec0 (@used_specs) {
			    my $subspec = find_spec ($subspec0);
			    if (not defined ($subspec)) {
				$build_status[$this_spec_id] = 'FAILED';
				$status_details[$this_spec_id] = "Spec file used by $fname not found: $subspec0\n";
				msg_error ("Spec file used by $fname not found: $subspec0");
				last;
			    }
			    if (!defined $specs_copied{$subspec}) {
				$specs_copied{$subspec} = 1;
				copy_spec ($spec, $subspec) or $specs_copied{$subspec} = 0;
			    }
			}
		    }
		    if ($build_status[$this_spec_id] eq "NOT_BUILT") {
			msg_info (2, "Processing $autospec");
			process_spec ($this_spec_id);
			if (defined ($provider{$capability}) and
			    $provider{$capability} == $this_spec_id) {
			    msg_info (2, "$autospec defines $capability");
			    msg_warning (0, "Added $autospec to the build to satisfy dependencies");
			    my $result = &$recursive_callback ($provider{$capability}, @rec_cb_opts);
			    return $result;
			} else {
			    msg_info (1, "Unfortunately, $autospec does not define $capability");
			    return 0;
			}
		    } else {
			return 0;
		    }
		} else {
		    msg_info (1, "$autospec was already loaded");
		    return 0;
		}
	    } else {
		&$warning_callback ($spec_name, $capability, "NOT_FOUND");
		msg_info (0, "No spec file for $capability found.");
		msg_info (0, "Try specifying additional spec file directories");
		msg_info (0, "using the --specdirs option");
		return 0;
	    }
	} else {
	    &$warning_callback ($spec_name, $capability, "NOT_FOUND");
	    msg_info (0, "Hint: use the --autodeps to locate spec files for dependencies automatically");
	    return 0;
	}
    } else {
	return 1;
    }

    # should not happen
    msg_error("Assertion failed: check_dependency: return 0");
    return 0;
}

# --------- copy build spec files, tarballs, patches -----------------------
sub find_spec ($) {
    my $fname = shift;

    my $spec_file;
    if (not ($fname =~ /^\// or -f $fname)) {
	msg_info (3, "Looking for spec file $fname");
	my @the_spec_dirlist = split /[:]/, $defaults->get ('specdirs');
	foreach my $sdir (@the_spec_dirlist) {
	    my $spath = "$sdir/$fname";
	    if (! -f "$spath") {
		msg_info (3, "   $fname not found in $sdir");
	    } else {
		msg_info (3, "   found in $sdir");
		$spec_file = "$spath";
		last;
	    }
	}
	if (not defined ($spec_file)) {
	    msg_warning (1, "spec file $fname not found");
	}
    } else {
	$spec_file = $fname;
    }

    return $spec_file;
}

sub copy_spec ($$) {
    my $spec = shift;
    my $spec_file = shift;
    my $spec_id = $all_specs{$spec->get_file_name ()};

    my $base_name = $spec_file;
    $base_name =~ s/^.*\/([^\/]+)/$1/;
    mkdir_p ("$topdir/SPECS") or
	$build_status[$spec_id]="ERROR",
	$status_details[$spec_id]="failed to create directory $topdir/SPECS",
	msg_error ("Failed to create directory $topdir/SPECS"),
	return 0;
    my $target = "$topdir/SPECS/$base_name";

    msg_info (2, "copying spec file $spec_file to the SPECS dir");

    my $is_nightly = $defaults->get ('nightly');
    if ($is_nightly and ($os eq "linux")) {
	my $the_nightly_date_format = $defaults->get ('date_format');
	my $the_nightly_date = `date "+$the_nightly_date_format"`;
	if ($? > 0) {
	    msg_error ("incorrect date format: $the_nightly_date_format");
	    msg_error ("Assertion failed: copy_spec: can't find spec_id for $spec_file") unless defined $spec_id;
	    $build_status[$spec_id] = 'ERROR';
	    $status_details[$spec_id] = "incorrect nightly date format: $the_nightly_date";
	    return 0;
	}

	open SPEC_OUT, ">$target";
	if (not copy ($spec_file, "/tmp/.pkgtool.tmp.$$")) {
	    msg_error ("failed to copy $spec_file to $target");
	    msg_error ("Assertion failed: copy_spec: can't find spec_id for $spec_file (2)") unless defined $spec_id;
	    $build_status[$spec_id] = 'FAILED';
	    $status_details[$spec_id] = "cound not copy spec file to $target";
	    return 0;
	}
	open SPEC_IN, "</tmp/.pkgtool.tmp.$$";
	my $line;
	while (1) {
	    $line = <SPEC_IN>;
	    if (not defined ($line)) {
		last;
	    }
	    if ($line =~ /^Release\s*:/) {
		$line =~ s/\s*$/.$the_nightly_date/;
	    }
	    print SPEC_OUT $line;
	}
	close SPEC_OUT;
	close SPEC_IN;

	`rm -f /tmp/.pkgtool.tmp.$$`;
	my @prdefs = ();
	my $rpm_target = $defaults->get ('target');
	if (defined $rpm_target) {
	    @prdefs = ("_target $rpm_target");
	}
	delete ($all_specs{$spec->get_file_name ()});
	msg_error ("Assertion failed: copy_spec: can't find spec_id for $spec_file (3)") unless defined $spec_id;
	my $spec = rpm_spec->new ($target, $rpm_target);
	$all_specs{$spec->get_file_name ()} = $spec_id;
	$specs_to_build[$spec_id] = $spec;
    } else {
	`cmp -s $spec_file $target`;
	if ($? != 0) {  # the files differ
	    if (not copy ($spec_file, $target)) {
		msg_error "failed to copy $spec_file to $target";
		msg_error ("Assertion failed: copy_spec: can't find spec_id for $spec_file (3)") unless defined $spec_id;
		$build_status[$spec_id] = 'FAILED';
		$status_details[$spec_id] = "failed to copy spec file to $target";
		return 0;
	    }
	} else {
	    msg_info (3, "   $spec_file and $target are identical; not copying");
	}
    }
    `chmod a+r $target`;
    
    return 1;
}

sub find_source ($$) {
    my $spec_id = shift;
    my $src = shift;
    my $is_tarball = 0;
    my $src_path;

    my @the_tarball_dirlist = split /[:]/, $defaults->get ('tarballdirs');
    if ($src =~ /\.(tar\.gz|tgz|tar\.bz2|tar\.bzip2|zip|jar)$/) {
	$is_tarball = 1;

	foreach my $srcdir (@the_tarball_dirlist) {
	    $src_path = "$srcdir/$src";
	    if (! -f "$src_path") {
		msg_info (3, "   $src not found in $srcdir");
	    } else {
		msg_info (3, "   $src found in $srcdir");
		return "$src_path";
	    }
	}
    }
    my @the_source_dirlist = split /[:]/, $defaults->get ('sourcedirs');
    foreach my $extsrcdir (@the_source_dirlist) {
	$src_path = "$extsrcdir/$src";
	if (! -f "$src_path") {
	    msg_info (3, "   $src not found in $extsrcdir");
	} else {
	    msg_info (3, "   $src found in $extsrcdir");
	    return "$src_path";
	}
    }

    if (!$is_tarball) {
	msg_info (3, "   trying the tarball directories");
	foreach my $srcdir (@the_tarball_dirlist) {
	    $src_path = "$srcdir/$src";
	    if (! -f "$src_path") {
		msg_info (3, "   $src not found in $srcdir");
	    } else {
		msg_info (3, "   $src found in $srcdir");
		return $src_path;
	    }
	}
    }

    return undef;
}

sub wget_source ($$$) {
    my $spec_id = shift;
    my $src = shift;
    my $target = shift;
    my $protocol;

    if ($src =~ /^(http|ftp):\/\//) {
	$protocol = $1;
    } else {
	return 0
    }

    my $wget_opts = " -nd -nH --tries=1 -T 60 ";
    my $mirrors = $defaults->get('source_mirrors');
    my $base_src = basename ($src);
    my @mirror_list;
    if (defined $mirrors) {
	@mirror_list = split /,\s*/, $mirrors;
    }

    my $retval;

    if ($download_test) {
	my $wget_command = "$wget $wget_opts -S --spider -O- $src 2>&1";
	msg_info (0, "Verifying url $src");
	msg_info (2, "Running $wget_command");
	my $wget_output = `$wget_command`;
	$retval = $?;
	chomp ($wget_output);

	if ($retval != 0) {
	    # download test unsuccessful, try mirrors
	    if (defined ($mirrors)) {
		$wget_output =~ s/\n/\nWARNING: wget: /g;
		msg_log ("WARNING: wget: $wget_output");
		msg_warning (0, "Download test from primary site failed");
		foreach my $mirror (@mirror_list) {
		    msg_info (0, "Testing download from source mirror $mirror");
		    $wget_command = "$wget $wget_opts -S --spider $mirror/$base_src 2>&1";
		    msg_info (2, "Running $wget_command");
		    $wget_output = `$wget_command`;
		    chomp ($wget_output);
		    $retval = $?;
		    if ($retval == 0) {
			$wget_output =~ s/\n/\nINFO: wget: /g;
			msg_info (0, "Download test from mirror $mirror successful");
			msg_log ("INFO: wget: $wget_output");
			last;
		    } else {
			$wget_output =~ s/\n/\nWARNING: wget: /g;
			msg_warning (0, "Download test from mirror $mirror failed");
			msg_log ("WARNING: wget: $wget_output");
		    }
		}
	    } else {
		$wget_output =~ s/\n/\nERROR: wget: /g;
		msg_error ("wget: $wget_output");
	    }
	}
	return !$retval;
    }

    my $download_dir = $defaults->get ("download_to");
    $download_dir = "$target" unless defined ($download_dir);

    msg_info (0, "Downloading source $src");

    # make sure that the download dir exists
    if (not mkdir_p ($download_dir)) {
	msg_error ("Failed to create download directory $download_dir");
	$build_status[$spec_id] = 'ERROR';
	$status_details[$spec_id] = "cannot create download directory";
	return 0;
    }

    # download to temporary file .$base_src
    unlink "$download_dir/.$base_src";
    my $wget_command = "$wget $wget_opts -O $download_dir/.$base_src $src 2>&1";
    msg_info (2, "Running $wget_command");
    my $wget_output = `$wget_command`;
    chomp ($wget_output);
    $retval = $?;

    if ($retval != 0) {
	# download unsuccessful, delete temporary file if exists
	unlink (".$base_src");
        if (defined ($mirrors)) {
	    $wget_output =~ s/\n/\nWARNING: wget: /g;
	    msg_log ("WARNING: wget: $wget_output");
	    msg_warning (0, "Download failed from primary site");
	    foreach my $mirror (@mirror_list) {
		msg_info (0, "Trying source mirror $mirror");
		$wget_command = "$wget $wget_opts -O $download_dir/.$base_src $mirror/$base_src 2>&1";
		msg_info (2, "Running $wget_command");
		$wget_output = `$wget_command`;
		chomp ($wget_output);
		$retval = $?;
		if ($retval == 0) {
		    $wget_output =~ s/\n/\nINFO: wget: /g;
		    msg_info (0, "Download successful from mirror $mirror");
		    msg_log ("INFO: wget: $wget_output");
		    last;
		} else {
		    $wget_output =~ s/\n/\nWARNING: wget: /g;
		    msg_warning (0, "Download failed from mirror $mirror");
		    msg_log ("WARNING: wget: $wget_output");
		}
	    }
        } else {
	    $wget_output =~ s/\n/\nERROR: wget: /g;
	    msg_log ("ERROR: wget: $wget_output");
        }
    } else {
	$wget_output =~ s/\n/\nINFO: wget: /g;
	msg_log ("INFO: wget: $wget_output");
    }

    if ($retval == 0) {
	# wget succeeded, now rename temporary file to the real name
	rename ("$download_dir/.$base_src", "$download_dir/$base_src");
    } else {
	if (not defined ($ENV{"${protocol}_proxy"})) {
	    msg_info (0, "Hint: if you are behind a firewall, you need to set the ${protocol}_proxy");
	    msg_info (0, "environment variable.  See man -M /usr/sfw/man wget for details")
	}
	msg_error ("Download failed: $src");
	$build_status[$spec_id] = 'ERROR';
	$status_details[$spec_id] = "Download failed: $src";
	return 0;
    }

    if ($download_dir ne $target) {
	my $src_path = $src;
	$src_path =~ s/^.*\/([^\/]+)/$1/;
	msg_info (1, "Source $src_path saved in $download_dir");
	$src_path = "$download_dir/$src_path";
	if (not copy ($src_path, $target)) {
	    msg_warning (0, "failed to copy $src_path to $target");
	} else {
	    # FIXME
	    `chmod a+r $target`;
	}
    }

    return 1;
}

sub copy_sources ($) {
    my $spec_id = shift;
    my $spec = $specs_to_build[$spec_id];

    my @sources = $spec->get_sources ();
    my $src_path;
    my $target;

    my @packages = $spec->get_packages ();
    foreach my $pkg (@packages) {
	next if not defined $pkg;
	my $cp_file = $pkg->get_tag ('sunw_copyright');
	next if not defined $cp_file;
	push (@sources, $cp_file);
    }

    my @class_scripts = $spec->get_class_script_names ();
    push (@sources, @class_scripts);

    msg_info (0, "Finding sources");
    msg_info (2, "copying sources to $topdir/SOURCES");

    foreach my $src (@sources) {
	next if not defined ($src);
	my $base_src = basename ($src);
        $src_path = find_source ($spec_id, $base_src);
	if (not defined ($src_path)) {
	    if ($defaults->get ("download") and wget_in_path ()) {
		wget_source ($spec_id, $src, "$topdir/SOURCES") and next;
	    }
	    if (not $defaults->get ("download")) {
		msg_info (0, "Hint: you need to use the --download option to enable automatic downloads");
		$status_details[$spec_id] = "Source $base_src not found";
	    } else {
		$status_details[$spec_id] = "Source $base_src download failed";
	    }
	    $build_status[$spec_id] = 'FAILED';
	    msg_error ($specs_to_build[$spec_id] . ": Source file $base_src not found");
	    return 0;
	}

	msg_info (3, "   copying $base_src");

	$target = "$topdir/SOURCES/$base_src";

	`cmp -s $src_path $target`;

	if ($? != 0) {  # the files differ
	    mkdir_p ("$topdir/SOURCES") or
		$build_status[$spec_id]="ERROR",
		$status_details[$spec_id]="failed to create directory $topdir/SOURCES",
		msg_error ("Failed to create directory $topdir/SOURCES"),
		return 0;
	    if (not copy ($src_path, $target)) {
		msg_error ("failed to copy $src_path to $target");
		$build_status[$spec_id] = 'ERROR';
		$status_details[$spec_id] = "failed to copy $src_path to $target";
		return 0;
	    }
	}

	`chmod a+r $target`;
    
    }

    return 1;
}

sub copy_patches ($) {
    my $spec_id = shift;
    my $spec = $specs_to_build[$spec_id];

    my @patches = $spec->get_patches ();
    my $patch_path;
    my $target;
    foreach my $patch (@patches) {
	next if not defined $patch;
	my $patch_base_name = basename ($patch);
	msg_info (2, "looking for patch $patch_base_name");

	my @the_patch_dirlist = split /[:]/, $defaults->get ('patchdirs');
	foreach my $the_patch_dir (@the_patch_dirlist) {
	    $patch_path = "$the_patch_dir/$patch_base_name";
	    
	    if (! -f "$patch_path") {
		msg_info (3, "   not found in $patch_path");
	    } else {
		msg_info (3, "   found in $patch_path");
		last;
	    }
	}

	if (! -f "$patch_path") {
	    # patch not found in the patch dirlist
	    # try to download it from the net
	    if ($defaults->get ("download") and wget_in_path ()) {
		wget_source ($spec_id, $patch, "$topdir/SOURCES") and next;
	    }
	    # download unsuccessful
	    $build_status[$spec_id] = 'ERROR';
	    $status_details[$spec_id] = "Patch $patch_base_name not found";
	    msg_error ("Patch $patch_base_name not found");
	    return 0;
	}

	$target = "$topdir/SOURCES/$patch_base_name";

	`cmp -s $patch_path $target`;

	if ($? != 0) {  # the files differ
	    mkdir_p ("$topdir/SOURCES") or
		$build_status[$spec_id]="ERROR",
		$status_details[$spec_id]="failed to create directory $topdir/SOURCES",
		msg_error ("Failed to create directory $topdir/SOURCES"),
		return 0;
	    if (not copy ($patch_path, $target)) {
		msg_error "failed to copy $patch_path to $target";
		$build_status[$spec_id] = 'ERROR';
		$status_details[$spec_id] = "failed to copy $patch_path to $target";
		return 0;
	    }
	} else {
	    msg_info (3, "   $patch_path and $target are identical; not copying");
	}

	`chmod a+r $target`;
    
    }

    return 1;
}

# --------- build command --------------------------------------------------
# optional args: build_only {1 = yes, 0 = no} == ! install
#                prep_only  {1 = yes, 0 = no}
sub do_build (;$$) {
    my $build_only = shift;
    my $prep_only = shift;
    $build_only = 0 unless defined $build_only;
    $prep_only = 0 unless defined $prep_only;

    for (my $i = 0; $i <= $#specs_to_build; $i++) {
	if ($build_status[$i] eq "NOT_BUILT") {
	    if (! build_spec ($i, $build_only, $prep_only)) {
		if (defined ($the_good_build_dir) and not $build_only) {
		    msg_info (0, "Attempting to use a known good package");
		    install_good_pkgs ($i);
		}
	    }
	    print_live_status;
	}
	my $notify = $defaults->get ('notify');
	if ($build_status[$i] ne "PASSED" and $build_status[$i] ne "SKIPPED") {
	    $exit_val++;
	    if ($notify) {
		my $notify_msg = "pkgbuild:\n$specs_to_build[$i] failed:\n$status_details[$i]";
		system ("notify-send --icon 'dialog-warning' '$notify_msg'");
	    }
	    if (($build_status[$i] ne "SKIPPED") and
		($build_status[$i] ne "DEP_FAILED")) {
		mail_log ($i);
	    }
	} else {
	    if ($notify) {
		my $notify_msg = "pkgbuild:\n$specs_to_build[$i] passed";
		system ("notify-send --icon 'dialog-ok' '$notify_msg'");
	    }
	}
    }

    print_live_status;

    my $verbose = $defaults->get ('verbose');
    if ($verbose > 0) {
	print_status;
    }
}

# given 2 package names and versions in the form of
#     <name>-<version>
# return 1 if the first version is higher than the 2nd one
# version is . separated list of numbers
sub is_newer ($$) {
    my $p1 = shift;
    my $p2 = shift;

    # cut the matching package name and dash
    if ("$p1 -compare- $p2" =~ /^(.*)-([0-9.]+) -compare- \1-([0-9.]+)$/) {
	my $v1 = $2;
	my $v2 = $3;
	# compare the first token
	while ($v1 =~ /^([0-9]+)\.*(.*)/) {
	    my $v10 = $1;
	    $v1 = $2;
	    # do both version strings have tokens left?
	    if ($v2 =~ /^([0-9]+)\.*(.*)/) {
		my $v20 = $1;
		$v2 = $2;
		if ($v10 > $v20) {
		    return 1;
		} elsif ($v10 < $v20) {
		    return 0;
		}
		# first token is the same, continue with the next
		next;
	    } else {
		# v2 has no more tokens, so v1 is higher
		return 1;
	    }
	}
	# loop ended, which means v1 has no more tokens
	if ($v2 =~ /^[0-9]+/) {
	    # v2 has more tokens, so it's the higher version
	    return 0;
	} else {
	    # the are the same
	    return 0;
	}
    }

    # the package names did not match
    return 0;
}

sub build_spec ($$$) {
    my $spec_id = shift;
    my $build_only = shift;
    my $prep_only = shift;
    my $spec = $specs_to_build[$spec_id];

    my $rmlog = $defaults->get ('rmlog');
    my $logname = $defaults->get ('logdir') . "/" . get_log_name ($spec_id);

    msg_debug (0, "Trying to build $spec");
    open_log ($logname);

    if ($build_status[$spec_id] ne "NOT_BUILT") {
	if ($build_status[$spec_id] eq "DEP") {
	    $build_status[$spec_id]="ERROR";
	    $status_details[$spec_id]="Circular dependency detected";
	    msg_error ("Circular dependency detected " .
		"while trying to build $spec");
	}
	return 0;
    }

    my @packages = $spec->get_packages ();
    msg_debug (3, "packages: " . $spec->get_file_name () . " defines: @packages");

    if ($rmlog > 0) {
        `rm -f $logname`
    }

    my $check_deps = $defaults->get ('deps');
    if (not $build_only) {
	foreach my $pkg (@packages) {
	    if (not defined ($pkg)) {
		next;
	    }
	    my $svr4_pkg = $pkg->get_svr4_name();
	    msg_info (3, "Checking if $svr4_pkg is installed");
	    if (is_provided ("$svr4_pkg") and $check_deps) {
		if ($defaults->get ('update_always')) {
		    msg_info (0, "Updating installed package ${pkg}");
		    next;
		}
		my $provider = what_provides ("$svr4_pkg");
		my $pkgver = $pkg->eval ("%version");
		if ($provider eq "${svr4_pkg}-${pkgver}") {
		    if ($defaults->get ('update_if_newer') and
			is_newer ("${svr4_pkg}-${pkgver}", $provider)) {
			msg_info (0, "Updating ${pkg} to version ${pkgver} (from $provider)");
			next;
		    } else {
			msg_warning (0, "skipping package ${pkg}-${pkgver}: $provider already installed");
			$status_details[$spec_id]="$provider installed";
			$build_status[$spec_id]='SKIPPED';
			return 1;
		    }
		}
	    }
	}
    }

    if ($check_deps) {
	my @dependencies = get_dependencies ($spec_id, $build_only, @packages);

	my $this_result;
	my $result = 1;

	$build_status[$spec_id] = "DEP";
	$status_details[$spec_id] = "building dependencies first";
	
	msg_info (1, "Checking dependencies of $spec");

	foreach my $dep (@dependencies) {
	    $dep =~ s/ .*//;
	    
	    if (not $prep_only) {
		$this_result = check_dependency ($spec_id, $dep,
						 \&build_spec, \&warn_always, ($build_only, $prep_only));
	    } else {
		$this_result = 1;
	    }
	    if (!$this_result) {
		if (defined ($the_good_build_dir)) {
		    msg_info (0, "Attempting to use a known good package");
		    $this_result = install_good_pkgs ($dep);
		}
		if (! $this_result) {
		    msg_warning (1, "$spec won't be built as it requires $dep");
		}
	    }
	    $result = ($this_result and $result);
	}
	
	if (! $result) {
	    $build_status[$spec_id]="DEP_FAILED";
	    $status_details[$spec_id]="Dependency check failed";
	    return 0;
	}
    }

    copy_sources ($spec_id) || return 0;
    copy_patches ($spec_id) || return 0;

    if ($live_summary) {
	$build_status[$spec_id] = 'BEING_BUILT';
	$status_details[$spec_id] = "$build_engine_name running";
	print_live_status;
    }

    if ($prep_only) {
	if ($prep_only == 2) {
	    run_build ($spec_id, "-bs") || return 0;
	} else {
	    run_build ($spec_id, "-bp") || return 0;
	}
    } else {
	run_build ($spec_id) || return 0;
    }

    if (not $build_only) {
	if ($os eq "solaris") {
	    if (defined ($ips)) {
		update_incorporations ($spec_id);
		install_pkgs_ips ($spec_id) || return 0;
	    } elsif (defined ($svr4)) {
		install_pkgs_svr4 ($spec_id) || return 0;
	    } else {
		msg_error ("Internal error: either IPS or SVr4 should be selected");
		return 0;
	    }
	} else {
	    install_rpms ($spec_id) || return 0;
	}
    }

    $build_status[$spec_id] = "PASSED";
    $status_details[$spec_id] = "";

    close_log;

    return 1;

}

sub run_build ($;$) {
    my $spec_id = shift;
    my $build_mode = shift;
    $build_mode = "-ba" unless defined $build_mode;
    my $spec = $specs_to_build[$spec_id];
    my $spec_file = $spec->get_file_name ();
    my $base_name = $spec->get_base_file_name ();
    my $log_name = "$base_name";
    $log_name = get_log_name ($spec_id);

    my $builddir = $spec->get_value_of ("_builddir");
    mkdir_p ($builddir) or
	msg_error ("failed to create directory $builddir"),
	$build_status[$spec_id]="ERROR",
	$status_details[$spec_id]="failed to create directory $builddir",
	return 0;
    my $build_user = getpwuid ((stat($builddir))[4]);
    foreach my $dir ("PKGS", "SPKGS", "PKGMAPS/copyright", "PKGMAPS/depend",
		     "PKGMAPS/pkginfo", "PKGMAPS/proto", "PKGMAPS/scripts") {
	mkdir_p ("$topdir/$dir") or
	    msg_error ("failed to create directory $topdir/$dir"),
	    $build_status[$spec_id]="ERROR",
	    $status_details[$spec_id]="failed to create directory $topdir/$dir",
	    return 0;
    }
    chomp (my $id = `id`);
    $id =~ s/^uid=([0-9]*).*/$1/;
    my $running_user = getpwuid ($id);    
    my $command;

    msg_info (0, "Running $build_engine_name $build_mode [...] $base_name ($spec)");
    my $the_log_dir = $defaults->get ('logdir');
    msg_info (1, "Log file: $the_log_dir/$log_name");

    my $the_command = $build_engine;
    my $check_deps = $defaults->get ('deps');
    if (not $check_deps) {
	$the_command = "$build_engine --nodeps";
    }
    my $interactive_mode = $defaults->get ('interactive');
    if ($interactive_mode and ($build_engine_name eq "pkgbuild")) {
	$the_command = "$the_command --interactive";
    }
    if ($build_engine_name eq "pkgbuild") {
	my $pkgformat = $defaults->get ('pkgformat');
	if (defined ($ips)) {
	    $pkgformat = "ips";
	}
	if ($pkgformat ne 'filesystem' and $pkgformat ne 'fs') {
	    $the_command = "$the_command --pkgformat $pkgformat";
	}
    }
    foreach my $def (@predefs) {
	next if not defined $def;
	$the_command = "$the_command --define '$def'";
    }

    if ($build_mode eq "-bp") {
	$the_command = "$the_command --nodeps";
    }

# FIXME: ExclusiveArch?
    my $rpm_target = $defaults->get ('target');
    if (defined($rpm_target)) {
        $command = "$the_command --target $rpm_target $build_mode $spec_file";
    } else {
        $command = "$the_command $build_mode $spec_file";
    }
    if ($running_user eq "root") {
	$command = "/bin/su $build_user -c \"$command\"";
    }

    my $save_log_name = $current_log;
    msg_log ("INFO: Build command: \"$command\"");
    msg_log ("INFO: Starting $build_engine_name build engine at " . `date`);
    my $tempfile = "/tmp/$build_engine_name.out.$$";
    msg_log ("INFO: Build engine output is written to $tempfile");
    msg_log ("INFO: and will be appended to this log when completed.");
    log_cmdout_starts ();

    my $build_result;
    if ($interactive_mode) {
#	system ("( $command 2>&1 ; echo $? > /tmp/.pkgbuild.status.$$) | tee $tempfile");
#	$build_result = `cat /tmp/.pkgbuild.status.$$ && rm -f /tmp/pkgbuild.status.$$`
	system ($command);
	$build_result = $?;
    } else {
	`$command > $tempfile 2>&1`;
	$build_result = $?; 
    }
    system ("sed -e 's/^/$build_engine_name: /' $tempfile >> $the_log_dir/$log_name 2>&1; rm -f $tempfile");
    log_cmdout_ends ($save_log_name);
    msg_log ("INFO: $build_engine_name $build_mode finished at " . `date`);

    if ($build_result) {
	msg_error ("$spec FAILED");
	msg_info (0, "Check the build log in $save_log_name for details");
	$build_status[$spec_id] = "FAILED";
	$status_details[$spec_id] = "$build_engine_name build failed";
	return 0;
    }

    msg_info (0, "$spec PASSED");
    return 1;
}

# --------- build-order and install-order commands -------------------------
sub do_build_order () {
    for (my $i = 0; $i <= $#specs_to_build; $i++) {
	print_order ($i, \&print_spec);
    }
}

sub do_install_order () {
    for (my $i = 0; $i <= $#specs_to_build; $i++) {
	if ($os eq "solaris") {
	    print_order ($i, \&print_pkgs);
	} else {
	    print_order ($i, \&print_rpms);
	}
    }
}

sub print_order ($&) {
    my $spec_id = shift;
    my $print_command = shift;
    my $spec = $specs_to_build[$spec_id];

    if ($build_status[$spec_id] ne "NOT_BUILT") {
	if ($build_status[$spec_id] eq "DEP") {
	    $build_status[$spec_id]="ERROR";
	    $status_details[$spec_id]="Circular dependency detected";
	    msg_error ("Circular dependency detected " .
		"while checking dependencies of $spec");
	}
	return 0;
    }

    my $check_deps = $defaults->get ('deps');
    if ($check_deps) {
	my @packages = $spec->get_packages ();
	my @dependencies = get_dependencies ($spec_id, 0, @packages);

	if (@dependencies) {
	    $build_status[$spec_id] = "DEP";
	
	    msg_info (1, "Checking dependencies of $spec");
	    
	    foreach my $dep (@dependencies) {
		$dep =~ s/ .*//;	
    
		check_dependency ($spec_id, $dep, \&print_order, 
				  \&warn_never, ($print_command));
	    }
	}
    }
    
    $build_status[$spec_id] = "PASSED";
    &$print_command ($spec_id);
}

sub do_download () {
    $defaults->set ("download", 1);
    for (my $spec_id = 0; $spec_id <= $#specs_to_build; $spec_id++) {
	my $spec = $specs_to_build[$spec_id];
	$build_status[$spec_id] = "DONE";
	msg_info (1, "Downloading sources for $spec");
	my @sources = $spec->get_sources ();
	foreach my $src (@sources) {
	    if (not defined ($src)) {
		next;
	    }
	    my $base_src = $src;
	    $base_src =~ s/^.*\/([^\/]+)/$1/;
	    my $src_path = find_source ($spec_id, $base_src);
	    if (not defined ($src_path)) {
		if (wget_in_path ()) {
		    wget_source ($spec_id, $src, "$topdir/SOURCES") and next;
		}
		$build_status[$spec_id] = 'FAILED';
		$status_details[$spec_id] = "Source $src not found";
		msg_error ($specs_to_build[$spec_id] . ": Source file $src not found");
		next;
	    }
	}
    }
    print_status ();
}

sub make_admin_file ($) {
    my $fname = shift;

    open ADMIN_FILE, ">$fname" or
	msg_error ("Failed to create pkgadd/pkgrm admin file $fname"),
	return undef;

    print ADMIN_FILE "mail=\n";
    print ADMIN_FILE "instance=unique\n";
    print ADMIN_FILE "conflict=quit\n";
    print ADMIN_FILE "setuid=nocheck\n";
    print ADMIN_FILE "action=nocheck\n";
    print ADMIN_FILE "partial=quit\n";
    print ADMIN_FILE "idepend=nocheck\n";
    print ADMIN_FILE "rdepend=nocheck\n";
    print ADMIN_FILE "space=quit\n";

    close ADMIN_FILE;
}

# --------- uninstall-pkgs command -----------------------------------------
sub do_uninstall_pkgs () {
    my $save_ds = $ds;
    # undefine $ds (meaning the pkg is datastream) so that only the
    # package names are added to the pkg list not the datastream file names 
    $ds = undef;
    for (my $i = 0; $i <= $#specs_to_build; $i++) {
	print_order ($i, \&push_to_pkg_list);
    }
    $ds = $save_ds;

    my $adminfile = "/tmp/pkg.admin.$$";
    my $command;
    my $pkgrm;
    my $suffix = '';
    if ($os eq "solaris") {
	make_admin_file ($adminfile);
	$command = "pfexec /usr/sbin/pkgrm -a $adminfile -n 2>&1";
	$pkgrm = "pkgrm";
	$suffix = "'.*'"
    } else {
	$command = "rpm -v --erase --nodeps 2>&1";
	$pkgrm = "rpm";
    }
    my $verbose = $defaults->get ('verbose');
    my $remove_status;
    foreach my $ref (@pkg_list) {
	my ($spec_id, $pkg_to_remove) = @$ref;
	if (is_installed ($pkg_to_remove)) {
	    msg_info (0, "Uninstalling $pkg_to_remove");
	    my $cmd_out = `$command $pkg_to_remove$suffix 2>&1`;
	    chomp ($cmd_out);
	    $remove_status = $?;
	    if ($remove_status > 0) {
		$build_status[$spec_id] = "FAILED";
		$status_details[$spec_id] = $cmd_out;
		$exit_val++;
	    } else {
		msg_info (1, "Successfully uninstalled $pkg_to_remove");
		if ($build_status[$spec_id] ne "FAILED") {
		    $build_status[$spec_id] = "UNINSTALLED";
		}
	    }
	    if ($cmd_out ne "") {
		if ($remove_status > 0) {
		    $cmd_out =~ s/\n(.)/\nERROR: $pkgrm: $1/;
		} else {
		    $cmd_out =~ s/\n(.)/\nINFO: $pkgrm: $1/;
		}
		if ($verbose > 1 or ($verbose > 0 and $remove_status > 0)) {
		    if ($remove_status > 0) {
			msg_error ($cmd_out);
		    } else {
			msg_info (1, $cmd_out);
		    }
		}
	    }
	} else {
	    msg_info (0, "Package $pkg_to_remove is not installed");
	    if ($build_status[$spec_id] eq "PASSED") {
		$build_status[$spec_id] = "NOT INSTLD";
	    }
	}
    }
    if ($os eq "solaris") {
	unlink ($adminfile);
    }
    if ($verbose > 0) {
	print_status;
    }
}

sub do_install_pkgs () {
    for (my $i = 0; $i <= $#specs_to_build; $i++) {
	print_order ($i, \&push_to_pkg_list);
    }

    @pkg_list = reverse (@pkg_list);
    my $adminfile = "/tmp/pkg.admin.$$";
    my $pkgsdir = $specs_to_build[0]->get_value_of ("_topdir") . "/PKGS";    
    my $command;
    my $pkgadd;
    if ($os eq "solaris") {
	make_admin_file ($adminfile);
	$command = "pfexec /usr/sbin/pkgadd -a $adminfile -n -d $pkgsdir";
	$pkgadd = "pkgadd";
    } else {
	$command = "rpm -v --upgrade";
	$pkgadd = "rpm";
    }
    my $verbose = $defaults->get ('verbose');
    my $install_status;
    foreach my $ref (@pkg_list) {
	my ($spec_id, $pkg_to_install) = @$ref;
	if (is_installed ($pkg_to_install)) {
	    msg_info (0, "$pkg_to_install is already installed.");
	    if ($build_status[$spec_id] eq "PASSED") {
		$build_status[$spec_id] = "SKIPPED";
	    }
	} else {
	    msg_info (0, "Installing package $pkg_to_install");
	    my $cmd_out;
	    if (defined ($ds)) {
		$cmd_out = `$command/$pkg_to_install all 2>&1`;
	    } else {
		$cmd_out = `$command $pkg_to_install 2>&1`;
	    }
	    chomp ($cmd_out);
	    $install_status = $?;
	    if ($install_status > 0) {
		$build_status[$spec_id] = "FAILED";
		$status_details[$spec_id] = $cmd_out;
		$exit_val++;
	    } else {
		msg_info (1, "Successfully installed $pkg_to_install");
		if ($build_status[$spec_id] ne "FAILED") {
		    $build_status[$spec_id] = "INSTALLED";
		}
	    }
	    if ($cmd_out ne "") {
		if ($install_status > 0) {
		    $cmd_out =~ s/\n(.)/\nERROR: $pkgadd: $1/;
		} else {
		    $cmd_out =~ s/\n(.)/\nINFO: $pkgadd: $1/;
		}
		if ($verbose > 1 or ($verbose > 0 and $install_status > 0)) {
		    if ($install_status > 0) {
			msg_error ($cmd_out);
		    } else {
			msg_info (1, $cmd_out);
		    }
		}
	    }
	}
    }
    if ($os eq "solaris") {
	unlink ($adminfile);
    }
    if ($verbose > 0) {
	print_status;
    }
}

sub get_specs_used ($);

sub get_specs_used ($) {
    my $fname = shift;
    my @ret = ();

    msg_info (3, "Looking for files %included or %used by $fname");

    open SPEC_IN1, "<$fname" or 
	msg_warning (0, "Couldn't open $fname for reading"), 
	return @ret;
    my @lines = <SPEC_IN1>;
    close SPEC_IN1;

    my @includes = grep /^\s*%include\s/, @lines;
    foreach my $inc (@includes) {
	$inc =~ s/^\s*%include\s+(\S+)\s*$/$1/ or next;
	if ($inc =~ /^"(.*)"$/) {
	    $inc = $1;
	}
	push (@ret, $inc);
	$inc = find_spec ($inc);
	next if not defined $inc;
	my @subspecs = get_specs_used ($inc);
	if (@subspecs) {
	    push (@ret, @subspecs);
	}
    }

    my @uses = grep /^\s*%use\s/, @lines;
    foreach my $use (@uses) {
	if ($use =~ /^\s*%use\s+([a-zA-Z_][a-zA-Z0-9_]*)\s*=\s*(\S+)\s*$/) {
	    $use = $2;
	    if ($use =~ /^"(.*)"$/) {
		$use = $1;
	    }
	    push (@ret, $use);
	    $use = find_spec ($use);
	    next if not defined $use;
	    my @subspecs = get_specs_used ($use);
	    if (@subspecs) {
		push (@ret, @subspecs);
	    }
	}
    }

    return @ret;
}

sub copy_specs () {
    msg_info (0, "Copying %use'd or %include'd spec files to SPECS directory");
    for (my $spec_id = 0; $spec_id <= $#specs_to_build; $spec_id++) {
        my $spec = $specs_to_build[$spec_id];
	my $fname = $spec->get_file_name ();
	if ($defaults->get ('nightly')) {
	    my $fullfname = find_spec ($fname);
	    if (not defined ($fullfname)) {
		$build_status[$spec_id] = 'FAILED';
		$status_details[$spec_id] = "Spec file not found: $fname\n";
		next;
	    }
	    if (not defined ($specs_copied{$fullfname})) {
		$specs_copied{$fullfname} = 1;
		copy_spec ($spec, $fullfname) or $specs_copied{$fullfname} = 0;
	    }
	}
	my @used_specs = get_specs_used ($fname);
	next if not @used_specs;
	foreach my $subspec0 (@used_specs) {
	    my $subspec = find_spec ($subspec0);
	    if (not defined ($subspec)) {
		$build_status[$spec_id] = 'FAILED';
		$status_details[$spec_id] = "Spec file used by $fname not found: $subspec0\n";
		next;
	    }
	    if (!defined $specs_copied{$subspec}) {
		$specs_copied{$subspec} = 1;
		copy_spec ($spec, $subspec) or $specs_copied{$subspec} = 0;
	    }
	}
    }
}

sub process_spec ($) {
    my $spec_id = shift;
		  
    my $spec = $specs_to_build[$spec_id];
    return if ($build_status[$spec_id] ne "NOT_BUILT");
    msg_info (1, "Processing spec file " . $spec->get_base_file_name ());
    my @packages = $spec->get_packages ();
    if (defined $spec->{error}) {
	$build_status[$spec_id] = 'ERROR';
	$status_details[$spec_id] = $spec->{error};
	msg_warning (0, "Failed to process spec file $spec: $spec->{error}");
	return;
    }
    foreach my $pkg (@packages) {
	next if not defined $pkg;
	my $pkgname = $pkg->get_tag ('SUNW_Pkg');
	$pkgname = "$pkg" unless $pkgname;
	if (defined ($provider{$pkgname})) {
	    my $prev_spec = $specs_to_build[$provider{$pkgname}];
	    msg_warning (0, "skipping spec file " . 
			 $spec->get_base_file_name() . 
			 ": $pkgname already defined by spec file " . 
			 $prev_spec->get_file_name ());
	    $build_status[$spec_id] = "ERROR";
	    $status_details[$spec_id] = "$pkgname is already defined by spec file " .
		$prev_spec->get_file_name ();
	} else {
	    $provider{$pkgname} = $spec_id;
	    msg_debug (2, "$pkgname is provided by spec $spec");
	}
	my @provides = $pkg->get_array ('provides');
	foreach my $prov (@provides) {
	    if (not defined $prov) {
		next;
	    }
	    $prov =~ s/ .*//;
	    if (defined ($provider{$prov}) and
		($provider{$pkgname} != $spec_id)) {
		my $prev_spec = $specs_to_build[$provider{$prov}];
		msg_warning (0, "skipping spec file " .
			     $spec->get_base_file_name() .
			     ": $pkgname is already defined by spec file "
			     . $prev_spec->get_file_name ());
		if ($build_status[$spec_id] ne "ERROR") {
		    $build_status[$spec_id] = "ERROR";
		    $status_details[$spec_id] = "$pkgname is already defined by spec file " . $prev_spec->get_file_name ();
		}
	    } else {
		$provider{$prov} = $spec_id;
	    }
	    msg_debug (2, "$prov is provided by spec $spec");
	}
    }
}

sub process_specs () {
    msg_info (0, "Processing spec files\n");
    for (my $spec_id = 0; $spec_id <= $#specs_to_build; $spec_id++) {
	process_spec ($spec_id);
    }
}


# merge new entries into the .pkgnames file
sub pkgnames_merge ($$) {
    my $pkgnames_file = shift;
    my $new_entries = shift;

    my %entries;

    if (! open PKGNAMES, "<$pkgnames_file") {
        msg_error ("Could not open file $pkgnames_file for reading");
	return;
    }

    my $line;
    # read entries from old .pkgnames file first
    while ($line = <PKGNAMES>) {
	chomp ($line);
	if ($line =~ /^([^:]+):(.*)$/) {
	    $entries{$1} = $2;
	} else {
	    msg_warning (1, "Invalid entry in .pkgnames file: $line");
	}
    }

    close PKGNAMES;

    if (! open PKGNAMES, "<$new_entries") {
        msg_warning (0, "Could not open file $new_entries for reading");
	return;
    }

    # New entries overwrite the old entries.
    # This is important when package summaries change.
    while ($line = <PKGNAMES>) {
	chomp ($line);
	if ($line =~ /^([^:]+):(.*)$/) {
	    $entries{$1} = $2;
	} else {
	    msg_warning (1, "Invalid entry in .pkgnames file: $line");
	}
    }

    close PKGNAMES;

    # Write merged data in temp file first
    if (! open PKGNAMES, ">${new_entries}") {
	msg_warning (0, "Failed to open package names file ${pkgnames_file}.$$ for writing");
	return;
    }

    foreach my $pkg (sort keys %entries) {
	print PKGNAMES "$pkg:$entries{$pkg}\n";
    }

    close PKGNAMES;

    # replace the real pkgnames file
    system ("mv $new_entries $pkgnames_file");
}

# write a list of package names and summaries so that pkgbuild can
# generate proper depend info for packages that are not currently installed
sub write_pkgnames () {
    my $pkgnames_file = "$topdir/.pkgnames";
    # just in case
    unlink ("${pkgnames_file}.$$");
    # build a temporary pkgnames file first
    if (! open PKGNAMES, ">${pkgnames_file}.$$") {
	msg_warning (0, "Failed to open package names file ${pkgnames_file}.$$ for writing");
	return;
    }
    for (my $spec_id = 0; $spec_id <= $#specs_to_build; $spec_id++) {
	my @packages = $specs_to_build[$spec_id]->get_packages ();
	my $summary;
	my $pkg_name;
	foreach my $pkg (@packages) {
	    next if not defined $pkg;
	    $summary = $pkg->eval('%summary');
	    $pkg_name = $pkg->get_name();
	    next if not defined $summary;
	    next if not defined $pkg_name;
	    print PKGNAMES "$pkg_name:$summary\n";
	}
    }
    close PKGNAMES;
    # merge the old and the new pkgnames:
    if (-f $pkgnames_file) {
	pkgnames_merge ($pkgnames_file, "$pkgnames_file.$$");
    } else {
	# .pkgnames is created for the first time.
	system ("mv $pkgnames_file.$$ $pkgnames_file");
    }
}

# --------- main program ---------------------------------------------------
sub main {
    my $dbus = `svcprop -c -p restarter/state svc:/system/dbus:default 2>/dev/null`;
    chomp ($dbus);
    if (($dbus eq "online") and find_in_path ('notify-send')) {
	$can_notify = 1;
    }

    process_defaults ();

    process_options ();

    $ds = $defaults->get('pkgformat');
    if ($ds eq 'ds' or $ds eq 'datastream') {
	$ds = 1;
    } else {
	$ds = undef;
    }

    my $summary_log = $defaults->get ('summary_log');
    if (defined ($summary_log) and -f $summary_log) {
	msg_error ("Summary build report $summary_log already exists")
	    or exit (1);
    }

    if (not defined ($specs_to_build[0])) {
	msg_info (0, "No spec files specified: nothing to do.");
	exit (0);
    }

    copy_specs ();
    process_specs ();

    # write of list of package names and summaries for pkgbuild
    # to pick up names of dependenct packages from
    write_pkgnames ();

    if (not defined ($build_command)) {
	usage (1);
    }
    
    if ($build_command eq "build") {
	can_install || cannot_install_error ("uninstall-pkgs", "installing",
					     "build-only");
	do_build;
    } elsif ($build_command eq "build-install") {
	can_install || cannot_install_error ("uninstall-pkgs", "installing",
					     "build-only");
	do_build;
    } elsif ($build_command eq "build-only") {
	do_build (1);
    } elsif ($build_command eq "prep") {
	# ignore dependencies if only doing %prep
	$defaults->set ('deps', 0);
	do_build (1, 1);
    } elsif ($build_command eq "spkg") {
	do_build (1, 2);
    } elsif ($build_command eq "uninstall-pkgs") {
	can_install || cannot_install_error ("uninstall-pkgs", "removing");
	do_uninstall_pkgs;
    } elsif ($build_command eq "install-pkgs") {
	can_install || cannot_install_error ("uninstall-pkgs", "installing");
	do_install_pkgs;
    } elsif ($build_command eq "publish-pkgs") {
	if (defined ($ips)) {
	    $exit_val = do_publish_pkgs();
	} else {
	    msg_error ("The publish-pkgs mode only works when using IPS");
	}
    } elsif ($build_command eq "build-order") {
	do_build_order;
    } elsif ($build_command eq "install-order") {
	do_install_order;
    } elsif ($build_command eq "download") {
	do_download;
    }

    exit ($exit_val);
}

$pkgbuild_path = shift (@ARGV);
$build_engine = $pkgbuild_path;

init;
main;
