/*
 * Copyright (c) 2007, to-do. All rights reserved.
 */
#include "util.h"
#include "var.h"
#include "_sym.h"
#include "_ext.h"

/*
sleep
alarm
fork
exec (execl, execve)
*/

/************************************************************
 *
 ************************************************************/
static int _global_getenv(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *s;
	warnif(argc != 1, "invalid arguments");
	warnif(!(s = var_toChars(argv)), "invalid argument-0");
	if (!(s = getenv(s))) {
		return 0;
	}
	new_string_copy(ex, res, s, strlen(s));
	return 1;
}
static int _global_setenv(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *name, *val = NULL;
	warnif(argc < 1, "invalid arguments");
	warnif(!(name = var_toChars(argv)), "invalid argument-0");
	if (argc >= 2) {
		val = var_toChars(argv + 1);
	}
	if (!val) {
		unsetenv(name);
		return 0;
	}
	new_bool(res, setenv(name, val, 1) == 0);
	return 1;
}

static int _global_system(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *s;
	warnif(argc != 1,  "invalid arguments");
	warnif(!(s = var_toChars(argv)),  "invalid argument-0");
	new_int(res, system(s));
	return 1;
}

static int _global_popen(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	FILE* fp;
	char *cmd, *mode="r";
	warnif(argc < 1, "invalid arguments");
	warnif(!(cmd = var_toChars(argv)), "invalid argument-0");
	warnif((argc >= 2) && !(mode = var_toChars(argv)), "invalid argument-1");
	if (!(fp = popen(cmd, mode))) {
		warns("open error: %s %s\n", cmd, strerror(errno));
		return 0;
	}
	new_file(res, fp);
	return 1;
}

static int _global_pclose(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	warnif(argc != 1, "invalid arguments");
	warnif(argv->type != V_file, "invalid argument-0");
	if (pclose((FILE*)argv->u.p) != 0) {
		warns("pclose error: %s\n", strerror(errno));
		return 0;
	}
	new_bool(res, 1);
	return 1;
}

/************************************************************
 *
 ************************************************************/
static int _global_load(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	var_t v;
	char *fn;
	char *bc;
	int bn, k, r;
	
	warnif(argc != 1,  "invalid arguments");
	warnif(!(fn = var_toChars(argv)),  "invalid argument-0");
	
	bc = NULL;
	bn = 0;
	r = compile_file_to_bytes(fn, &bc, &bn);
	if (r != 0) {
		if (bc) free(bc);
		/* compile error */
		return 0;
	}
	
	r = call_bytes(ex, "load", bc, 0, NULL, res);
	
	if (!r) {
		free(bc);
	} else {
		k = strtoh(fn, strlen(fn));
		new_string_with(ex, &v, bc, bn);
		frame_setLocal(ex, ex->fp, k, &v);
	}
	
	return r;
}
static int _global_eval(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *s;
	int n, r;
	
	warnif(argc != 1,  "invalid arguments");
	warnif (!var_toString(argv, &s, &n), "invalid argument-0");
	
	r = eval_bytes(ex, s, n, res);
	
	return r;
}

/************************************************************
 *
 ************************************************************/
static int _global_print(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *s;
	int i,n,m=0;
	warnif(argc < 1,  "invalid arguments");
	for (i = 0; i < argc; i++) {
		if (var_toString(argv + i, &s, &n)) {
			m += fwrite(s, 1, n, g_stdout);
		}
	}
	new_int(res, m);
	return 1;
}
static int _global_println(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *s;
	int i,n,m=0;
	warnif(argc < 1,  "invalid arguments");
	for (i = 0; i < argc; i++) {
		if (var_toString(argv + i, &s, &n)) {
			m += fwrite(s, 1, n, g_stdout);
		}
		m += fputc('\n', g_stdout) > 0;
	}
	new_int(res, m);
	return 1;
}
static int _global_printf(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *s;
	int n;
	warnif(argc < 1,  "invalid arguments");
	warnif(!(s = var_toChars(argv)),  "invalid argument-0");
	n = printf_to_file(g_stdout, s, argv + 1, argc - 1);
	new_int(res, n);
	return 1;
}
static int _global_sprintf(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	string_t* str;
	char *s;
	warnif(argc < 1,  "invalid arguments");
	warnif(!(s = var_toChars(argv)),  "invalid argument-0");
	str = string_alloc(ex);
	str->n = printf_to_bytes(&str->s, s, argv + 1, argc - 1);
	new_string(ex, res, str);
	return 1;
}
static int _global_echo(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *s;
	int n;
	
	warnif (argc != 1, "invalid arguments");
	warnif (!var_toString(argv, &s, &n), "invalid argument-0");
	
	n = echo_to_file(ex, g_stdout, s, n);
	
	new_int(res, n);
	
	return 1;
}
static int _global_printfile(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	FILE* fp;
	char buf[BUFSIZ + 1];
	char *s;
	int n, m;
	
	warnif(argc != 1, "invalid arguments");
	warnif(!(s = var_toChars(argv)), "invalid argument-0");
	
	if (!(fp = fopen(s, "r"))) {
		warns("fopen error: %s %s\n", s, strerror(errno));
		return 0;
	}
	n = 0;
	while ((m = fread(buf, 1, BUFSIZ, fp)) > 0) {
		fwrite(buf, 1, m, g_stdout);
		n += m;
	}
	fclose(fp);
	new_int(res, n);
	return 1;
}

/************************************************************
 *
 ************************************************************/
static int _global_serial(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *s;
	int n;
	
	warnif(argc != 1,  "invalid arguments");
	
	if (!serial_to_bytes(ex, argv, &s, &n)) {
		return 0;
	}
	new_string_with(ex, res, s, n);
	return 1;
}

static int _global_unserial(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *s;
	int n;
	
	warnif(argc != 1,  "invalid arguments");
	warnif(!var_toString(argv, &s, &n),  "invalid argument-0");
	
	n = unserial_from_bytes(ex, res, s, n);
	return (n > 0);
}

static int _global_trace(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *s;
	int n;
	
	warnif(argc != 1,  "invalid arguments");
	if (!trace_to_bytes(ex, argv, &s, &n)) {
		return 0;
	}
	new_string_with(ex, res, s, n);
	return 1;
}

static int _global_untrace(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *s;
	int n;
	warnif(argc != 1,  "invalid arguments");
	warnif(!var_toString(argv, &s, &n),  "invalid argument-0");
	n = untrace_from_bytes(ex, res, s, n);
	return (n > 0);
}

/************************************************************
 *
 ************************************************************/
static int _global_pack(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *fmt, *dst;
	int len;
	
	warnif(argc < 2,  "invalid arguments");
	warnif(!(fmt = var_toChars(argv)),  "invalid argument-0");
	
	len = pack_to_bytes(&dst, fmt, argv + 1, argc - 1);
	if (!len) {
		return 0;
	}
	new_string_with(ex, res, dst, len);
	return 1;
}
static int _global_unpack(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *fmt, *src;
	int len;
	
	warnif(argc < 2,  "invalid arguments");
	warnif(!(fmt = var_toChars(argv)),  "invalid argument-0");
	warnif(!var_toString(argv + 1, &src, &len),  "invalid argument-1");
	
	len = unpack_from_bytes(ex, fmt, src, len, res);
	
	return (len > 0);
}

static int _global_jconv(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *s, *d;
	int n, q;
	q = 0;
	warnif(argc < 1,  "invalid arguments");
	warnif(!var_toString(argv, &s, &n),  "invalid argument-0");
	if (argc >= 2) {
		q = var_toChar(argv + 1);
	}
	if (!(n = jconv(&d, s, n, q))) {
		return 0;
	}
	new_string_with(ex, res, d, n);
	return 1;
}
static int _global_quote(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *s, *d;
	int n,q;
	q = 0;
	warnif(argc < 1,  "invalid arguments");
	warnif(!var_toString(argv, &s, &n),  "invalid argument-0");
	if (argc >= 2) {
		q = var_toChar(argv + 1);
	}
	if (!(n = quote(&d, s, n, q))) {
		return 0;
	}
	new_string_with(ex, res, d, n);
	return 1;
}
static int _global_unquote(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *s, *d;
	int n;
	warnif(argc != 1,  "invalid arguments");
	warnif(!var_toString(argv, &s, &n),  "invalid argument-0");
	if (!(n = unquote(&d, s, n))) {
		return 0;
	}
	new_string_with(ex, res, d, n);
	return 1;
}
static int _global_encode(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *s, *d;
	int n,q;
	q = 0;
	warnif(argc < 1,  "invalid arguments");
	warnif(!var_toString(argv, &s, &n),  "invalid argument-0");
	if (argc >= 2) {
		q = var_toChar(argv + 1);
	}
	if (!(n = encode(&d, s, n, q))) {
		return 0;
	}
	new_string_with(ex, res, d, n);
	return 1;
}
static int _global_decode(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *s, *d;
	int n,q;
	q = 0;
	warnif(argc < 1,  "invalid arguments");
	warnif(!var_toString(argv, &s, &n),  "invalid argument-0");
	if (argc >= 2) {
		q = var_toChar(argv + 1);
	}
	if (!(n = decode(&d, s, n, q))) {
		return 0;
	}
	new_string_with(ex, res, d, n);
	return 1;
}
static int _global_crypt(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *s, *k;
	char buf[10];
	warnif(argc < 1,  "invalid arguments");
	warnif(!(s = var_toChars(argv)),  "invalid argument-0");
	if (argc >= 2) {
		k = var_toChars(argv + 1);
	} else {
		strncpy(k = buf, s, 2);
		k[2] = 0;
	}
	if (!(s = strcrypt(s, k))) {
		warns("crypt error: %s\n", strerror(errno));
		return 0;
	}
	new_string_copy(ex, res, s, strlen(s));
	return 1;
}
static int _global_md5(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char buf[33];
	char *s;
	int n;
	warnif(argc != 1,  "invalid arguments");
	warnif(!var_toString(argv, &s, &n),  "invalid argument-0");
	if (!(n = strnmd5(buf, s, n))) {
		warns("strnmd5 error: %s\n", s);
		return 0;
	}
	new_string_copy(ex, res, buf, n);
	return 1;
}

/************************************************************
 *
 ************************************************************/
static int _global_atan2(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	warnif(argc != 2,  "invalid arguments");
	new_float(res, atan2(var_toFloat(argv), var_toFloat(argv + 1)));
	return 1;
}
static int _global_pow(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	warnif(argc != 2,  "invalid arguments");
	new_float(res, pow(var_toFloat(argv), var_toFloat(argv + 1)));
	return 1;
}
static int _global_sin(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	warnif(argc != 1,  "invalid arguments");
	new_float(res, sin(var_toFloat(argv)));
	return 1;
}
static int _global_asin(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	warnif(argc != 1,  "invalid arguments");
	new_float(res, asin(var_toFloat(argv)));
	return 1;
}
static int _global_cos(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	warnif(argc != 1,  "invalid arguments");
	new_float(res, cos(var_toFloat(argv)));
	return 1;
}
static int _global_acos(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	warnif(argc != 1,  "invalid arguments");
	new_float(res, acos(var_toFloat(argv)));
	return 1;
}
static int _global_tan(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	warnif(argc != 1,  "invalid arguments");
	new_float(res, tan(var_toFloat(argv)));
	return 1;
}
static int _global_atan(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	warnif(argc != 1,  "invalid arguments");
	new_float(res, atan(var_toFloat(argv)));
	return 1;
}
static int _global_sqrt(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	warnif(argc != 1,  "invalid arguments");
	new_float(res, sqrt(var_toFloat(argv)));
	return 1;
}
static int _global_exp(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	warnif(argc != 1,  "invalid arguments");
	new_float(res, exp(var_toFloat(argv)));
	return 1;
}
static int _global_log(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	warnif(argc != 1,  "invalid arguments");
	new_float(res, log(var_toFloat(argv)));
	return 1;
}
static int _global_log10(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	warnif(argc != 1,  "invalid arguments");
	new_float(res, log10(var_toFloat(argv)));
	return 1;
}
static int _global_ceil(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	warnif(argc != 1,  "invalid arguments");
	new_int(res, (int)ceil(var_toFloat(argv)));
	return 1;
}
static int _global_floor(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	warnif(argc != 1,  "invalid arguments");
	new_int(res, (int)floor(var_toFloat(argv)));
	return 1;
}
static int _global_round(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	warnif(argc != 1,  "invalid arguments");
	new_int(res, (int)round(var_toFloat(argv)));
	return 1;
}
static int _global_abs(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	double f;
	int i;
	warnif(argc != 1,  "invalid arguments");
	if (var_toNumber(argv, &f, &i) > 1) {
		new_float(res, fabs(f));
	} else {
		new_int(res, abs(i));
	}
	return 1;
}
static int _global_rand(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	double f, r;
	int i;
	static long seed = 0;
	if (!seed) {
		srand(seed = 1);
	}
	warnif(argc > 1,  "invalid arguments");
	r = rand() / (double)RAND_MAX;
	if (argc < 1) {
		new_float(res, r);
	} else {
		if (var_toNumber(argv, &f, &i) > 1) {
			new_float(res, r * f);
		} else {
			new_int(res, (int)(r * i));
		}
	}
	return 1;
}
static int _global_min(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	int i;
	warnif(argc < 2,  "invalid arguments");
	copy_var(res, argv);
	for (i = 1; i < argc; i++) {
		if (var_cmpAsNumber(res, argv + i) > 0) {
			copy_var(res, argv + i);
		}
	}
	return 1;
}
static int _global_max(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	int i;
	warnif(argc < 2,  "invalid arguments");
	copy_var(res, argv);
	for (i = 1; i < argc; i++) {
		if (var_cmpAsNumber(res, argv + i) < 0) {
			copy_var(res, argv + i);
		}
	}
	return 1;
}
static int _global_isfinite(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	warnif(argc != 1,  "invalid arguments");
	new_bool(res, (argv->type == V_int) || ((argv->type == V_float) && finite(argv->u.f)));
	return 1;
}
static int _global_isinf(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	warnif(argc != 1,  "invalid arguments");
	new_bool(res, (argv->type == V_float) && isinf(argv->u.f));
	return 1;
}
static int _global_isnan(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	warnif(argc != 1,  "invalid arguments");
	new_bool(res, (argv->type == V_float) && isnan(argv->u.f));
	return 1;
}

/************************************************************
 *
 ************************************************************/
static int _global_readfile(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	FILE* fp;
	char buf[BUFSIZ + 1];
	char *s;
	int n, m;
	warnif(argc != 1,  "invalid arguments");
	warnif(!(s = var_toChars(argv)), "invalid argument-0");
	if (!(fp = fopen(s, "r"))) {
		warns("fopen error: %s %s\n", s, strerror(errno));
		return 0;
	}
	s = NULL;
	n = 0;
	while ((m = fread(buf, 1, BUFSIZ, fp)) > 0) {
		s = realloc(s, n + m + 1);
		memcpy(s + n, buf, m);
		n += m;
	}
	fclose(fp);
	if (!n) {
		return 0;
	}
	s[n] = 0;
	new_string_with(ex, res, s, n);
	return 1;
}
static int _global_writefile(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	FILE* fp;
	char *s;
	int n;
	warnif(argc != 2,  "invalid arguments");
	warnif(!(s = var_toChars(argv)), "invalid argument-0");
	if (!(fp = fopen(s, "w"))) {
		warns("fopen error: %s %s\n", s, strerror(errno));
		return 0;
	}
	if (!var_toString(argv + 1, &s, &n)) {
		warns("invalid argument: %s\n", typeof_var(argv + 1));
		n = 0;
	} else {
		n = fwrite(s, 1, n, fp);
	}
	fclose(fp);
	new_int(res, n);
	return 1;
}
static int _global_fopen(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	FILE* fp;
	char *path=NULL, *mode = "r";
	
	warnif(argc < 1,  "invalid arguments");
	warnif (!(path = var_toChars(argv + 0)), "invalid argument-0");
	warnif ((argc >= 2) && !(mode = var_toChars(argv + 1)), "invalid argument-1");
	
	fp = fopen(path, mode);
	if (!fp) {
		warns("fopen error: %s %s\n", path, strerror(errno));
		return 0;
	}
	new_file(res, fp);
	return 1;
}
static int _global_tmpfile(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	FILE* fp;
	warnif(argc != 0,  "invalid arguments");
	if (!(fp = tmpfile())) {
		warns("tmpfile error: %s\n", strerror(errno));
		return 0;
	}
	new_file(res, fp);
	return 1;
}
static int _global_tempnam(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char buf[FILENAME_MAX];
	char *s1, *s2, *s3;
	struct stat st;
	int i;
	s1 = s2 = s3 = NULL;
	
	if (argc >= 1) s1 = var_toChars(argv + 0);
	if (argc >= 2) s2 = var_toChars(argv + 1);
	if (argc >= 3) s3 = var_toChars(argv + 2);
	if (!s1) s1 = P_tmpdir;
	if (!s2) s2 = "";
	if (!s3) s3 = "";
	
	if (!s1 || !*s1 || (stat(s1, &st) != 0) || !(st.st_mode & S_IFDIR)) {
		warns("invalid directory: %s\n", s1);
		return 0;
	}
	if ((strlen(s1) + strlen(s2) + strlen(s3) + 10) >= FILENAME_MAX) {
		warns("too large file-name: %s %s %s\n", s1, s2, s3);
		return 0;
	}
	s1 = strchr(strcpy(buf, s1), 0);
	if (*(s1 - 1) != '/') {
		*s1++ = '/';
	}
	if (*s2) {
		s1 = strchr(strcpy(s1, s2), 0);
	}
	
	i = 0;
	do {
		sprintf(s1, "%04x%s", i++, s3);
	} while ((stat(buf, &st) == 0) && (i < 0xFFFF));
	
	if (errno != ENOENT) {
		warns("tempnam error: %s\n", buf);
		return 0;
	}
	
	new_string_copy(ex, res, buf, strlen(buf));
	
	return 1;
}
static int _global_chmod(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *path;
	int mode = 0777;
	warnif(argc < 1,  "invalid arguments");
	warnif(!(path = var_toChars(argv)),  "invalid argument-0");
	if (argc > 1) {
		mode = var_toInt(argv + 1);
	}
	if (chmod(path, mode)) {
		warns("chmod error: %s %s\n", path, strerror(errno));
		return 0;
	}
	new_bool(res, 1);
	return 1;
}
static int _global_mkdir(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *path;
	int mode = 0777;
	warnif(argc < 1,  "invalid arguments");
	warnif(!(path = var_toChars(argv)),  "invalid argument-0");
	if (argc > 1) {
		mode = var_toInt(argv + 1);
	}
	if (mkdir(path, mode)) {
		warns("mkdir error: %s %s\n", path, strerror(errno));
		return 0;
	}
	new_bool(res, 1);
	return 1;
}
static int _global_touch(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	FILE* fp;
	char *path;
	warnif(argc != 1,  "invalid arguments");
	warnif(!(path = var_toChars(argv)),  "invalid argument-0");
	if (!(fp = fopen(path, "r"))) {
		if (!(fp = fopen(path, "w"))) {
			warns("fopen error: %s %s\n", path, strerror(errno));
			return 0;
		}
		fclose(fp);
	} else {
		fclose(fp);
		if (utimes(path, NULL)) {
			warns("utime error: %s %s\n", path, strerror(errno));
		}
	}
	new_bool(res, 1);
	return 1;
}
static int _global_chdir(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *path;
	warnif(argc != 1,  "invalid arguments");
	warnif(!(path = var_toChars(argv)),  "invalid argument-0");
	if (chdir(path)) {
		warns("mkdir error: %s %s\n", path, strerror(errno));
		return 0;
	}
	new_bool(res, 1);
	return 1;
}
static int _global_getcwd(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *s;
	warnif(argc != 0,  "invalid arguments");
	if (!(s = getcwd(NULL, 0))) {
		warns("getcwd error: %s %s\n", strerror(errno));
	}
	new_string_with(ex, res, s, strlen(s));
	return 1;
}
static int _global_rename(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *f1, *f2;
	warnif(argc < 2,  "invalid arguments");
	warnif(!(f1 = var_toChars(argv + 0)),  "invalid argument-0");
	warnif(!(f2 = var_toChars(argv + 1)),  "invalid argument-1");
	if (rename(f1, f2)) {
		warns("rename error: %s %s\n", f1, strerror(errno));
		return 0;
	}
	new_bool(res, 1);
	return 1;
}
static int _global_symlink(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *f1, *f2;
	warnif(argc < 2,  "invalid arguments");
	warnif(!(f1 = var_toChars(argv + 0)),  "invalid argument-0");
	warnif(!(f2 = var_toChars(argv + 1)),  "invalid argument-1");
	if (symlink(f1, f2)) {
		warns("symlink error: %s %s\n", f1, strerror(errno));
		return 0;
	}
	new_bool(res, 1);
	return 1;
}
static int _global_flink(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *f1, *f2;
	warnif(argc < 2,  "invalid arguments");
	warnif(!(f1 = var_toChars(argv + 0)),  "invalid argument-0");
	warnif(!(f2 = var_toChars(argv + 1)),  "invalid argument-1");
	if (link(f1, f2)) {
		warns("link error: %s\n", strerror(errno));
		return 0;
	}
	new_bool(res, 1);
	return 1;
}
static int _global_fcopy(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	FILE *p1, *p2;
	char *f1, *f2;
	char buf[BUFSIZ];
	size_t sz;
	warnif(argc < 2,  "invalid arguments");
	warnif(!(f1 = var_toChars(argv + 0)),  "invalid argument-0");
	warnif(!(f2 = var_toChars(argv + 1)),  "invalid argument-1");
	if (!(p1 = fopen(f1, "r"))) {
		warns("fopen error: %s %s\n", f1, strerror(errno));
		return 0;
	}
	if (!(p2 = fopen(f2, "w"))) {
		fclose(p1);
		warns("fopen error: %s %s\n", f2, strerror(errno));
		return 0;
	}
	while ((sz = fread(buf, 1, BUFSIZ, p1)) > 0) {
		fwrite(buf, 1, sz, p2);
	}
	fclose(p2);
	fclose(p1);
	new_bool(res, 1);
	return 1;
}
static int _global_truncate(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *path;
	size_t sz = 0;
	warnif(argc < 1,  "invalid arguments");
	warnif(!(path = var_toChars(argv)),  "invalid argument-0");
	if (argc > 1) {
		sz = var_toInt(argv + 1);
	}
	if (truncate(path, sz)) {
		warns("truncate error: %s %s\n", path, strerror(errno));
	}
	new_bool(res, 1);
	return 1;
}

/************************************************************
 *
 ************************************************************/
static int _global_glob(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	glob_t gl;
	var_t tmp;
	char *s;
	int i;
	
	warnif(argc != 1,  "invalid arguments");
	warnif(!(s = var_toChars(argv)),  "invalid argument-0");
	
	memset(&gl, 0, sizeof(glob_t));
	/* append '/' if directory */
	if (glob(s, GLOB_MARK, NULL, &gl)) {
		warns("glob error: %s %s\n", s, strerror(errno));
		return 0;
	}
	new_array(ex, res, NULL);
	for (i = 0; i < gl.gl_pathc; i++) {
		s = gl.gl_pathv[i];
		new_string_copy(ex, &tmp, s, strlen(s));
		array_push(ex, res->u.p, &tmp);
	}
	globfree(&gl);
	return 1;
}
static int _global_remove(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	glob_t gl;
	char *s;
	int i, j, n;
	
	warnif(argc < 1,  "invalid arguments");
	
	n = 0;
	for (i = 0; i < argc; i++) {
		if (!(s = var_toChars(argv + i))) {
			warns("invalid argument: %d %s\n", i, typeof_var(argv + i));
		} else if (strchr(s, '*')) {
			memset(&gl, 0, sizeof(glob_t));
			/* do not sort */
			if (glob(s, GLOB_NOSORT, NULL, &gl)) {
				warns("glob error: %s %s\n", s, strerror(errno));
			} else {
				for (j = 0; j < gl.gl_pathc; j++) {
					if (remove(s = gl.gl_pathv[j])) {
						warns("remove error: %s %s\n", s, strerror(errno));
					} else {
						n++;
					}
				}
				globfree(&gl);
			}
		} else {
			if (remove(s)) {
				warns("remove error: %s %s\n", s, strerror(errno));
			} else {
				n++;
			}
		}
	}
	new_int(res, n);
	return 1;
}
static int _global_unlink(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	glob_t gl;
	char *s;
	int i, j, n;
	
	warnif(argc < 1,  "invalid arguments");
	
	n = 0;
	for (i = 0; i < argc; i++) {
		if (!(s = var_toChars(argv + i))) {
			warns("invalid argument: %d %s\n", i, typeof_var(argv + i));
		} else if (strchr(s, '*')) {
			memset(&gl, 0, sizeof(glob_t));
			/* do not sort */
			if (glob(s, GLOB_NOSORT, NULL, &gl)) {
				warns("glob error: %s %s\n", s, strerror(errno));
			} else {
				for (j = 0; j < gl.gl_pathc; j++) {
					if (unlink(s = gl.gl_pathv[j])) {
						warns("unlink error: %s %s\n", s, strerror(errno));
					} else {
						n++;
					}
				}
				globfree(&gl);
			}
		} else {
			if (unlink(s)) {
				warns("unlink error: %s %s\n", s, strerror(errno));
			} else {
				n++;
			}
		}
	}
	new_int(res, n);
	return 1;
}

/************************************************************
 *
 ************************************************************/
static int _global_fmode(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	struct stat st;
	char *path;
	warnif(argc != 1,  "invalid arguments");
	warnif (!(path = var_toChars(argv)), "invalid argument-0");
	if (stat(path, &st)) {
		warns("stat error: %s %s\n", path, strerror(errno));
		return 0;
	}
	new_int(res, st.st_mode);
	return 1;
}
static int _global_fsize(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	struct stat st;
	char *path;
	warnif(argc != 1,  "invalid arguments");
	warnif (!(path = var_toChars(argv)), "invalid argument-0");
	if (stat(path, &st)) {
		warns("stat error: %s %s\n", path, strerror(errno));
		return 0;
	}
	new_int(res, st.st_size);
	return 1;
}
static int _global_fmtime(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	struct stat st;
	char *path;
	warnif(argc != 1,  "invalid arguments");
	warnif (!(path = var_toChars(argv)), "invalid argument-0");
	if (stat(path, &st)) {
		warns("stat error: %s %s\n", path, strerror(errno));
		return 0;
	}
	new_int(res, st.st_mtime);
	return 1;
}
static int _global_fctime(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	struct stat st;
	char *path;
	warnif(argc != 1,  "invalid arguments");
	warnif (!(path = var_toChars(argv)), "invalid argument-0");
	if (stat(path, &st)) {
		warns("stat error: %s %s\n", path, strerror(errno));
		return 0;
	}
	new_int(res, st.st_ctime);
	return 1;
}
static int _global_fatime(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	struct stat st;
	char *path;
	warnif(argc != 1,  "invalid arguments");
	warnif (!(path = var_toChars(argv)), "invalid argument-0");
	if (stat(path, &st)) {
		warns("stat error: %s %s\n", path, strerror(errno));
		return 0;
	}
	new_int(res, st.st_atime);
	return 1;
}

/************************************************************
 *
 ************************************************************/
#include <pwd.h>
#include <grp.h>
static int _global_chown(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *path, *us, *gs;
	struct passwd *pw;
	struct group *gr;
	path = us = gs = NULL;
	pw = NULL;
	gr = NULL;
	warnif(argc < 1,  "invalid arguments");
	warnif (!(path = var_toChars(argv)), "invalid argument-0");
	warnif((argc >= 2) && !(us = var_toChars(argv + 1)), "invalid argument-1");
	warnif((argc >= 3) && !(gs = var_toChars(argv + 2)), "invalid argument-2");
	if (us && !(pw = getpwnam(us))) {
		warns("getpwnam error: %s\n", us);
	}
	if (gs && !(gr = getgrnam(gs))) {
		warns("getgrnam error: %s\n", gs);
	}
	if (chown(path, pw ? pw->pw_uid : -1, gr ? gr->gr_gid : -1)) {
		warns("chown error: %s %s %s\n", us, gs, strerror(errno));
		return 0;
	}
	new_bool(res, 1);
	return 1;
}
static int _global_fuid(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	struct stat st;
	char *path;
	struct passwd *pw;
	warnif(argc != 1,  "invalid arguments");
	warnif (!(path = var_toChars(argv)), "invalid argument-0");
	if (stat(path, &st)) {
		warns("stat error: %s %s\n", path, strerror(errno));
		return 0;
	}
	if (!(pw = getpwuid(st.st_uid))) {
		warns("getpwuid error: %s %d\n", path, st.st_uid);
		return 0;
	}
	new_string_copy(ex, res, pw->pw_name, strlen(pw->pw_name));
	return 1;
}
static int _global_fgid(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	struct stat st;
	char *path;
	struct group *gr;
	
	warnif(argc != 1,  "invalid arguments");
	warnif (!(path = var_toChars(argv)), "invalid argument-0");
	if (stat(path, &st)) {
		warns("stat error: %s %s\n", path, strerror(errno));
		return 0;
	}
	if (!(gr = getgrgid(st.st_gid))) {
		warns("getgrgid error: %s %d\n", path, st.st_gid);
	}
	new_string_copy(ex, res, gr->gr_name, strlen(gr->gr_name));
	return 1;
}

/************************************************************
 *
 ************************************************************/
static int _global_dbopen(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	void *p;
	char *path=NULL, *mode = NULL;
	
	warnif(argc < 1,  "invalid arguments");
	warnif (!(path = var_toChars(argv)), "invalid argument-0");
	warnif ((argc >= 2) && !(mode = var_toChars(argv)), "invalid argument-1");
	
	p = dbm_open(path, mode);
	if (!p) {
		return 0;
	}
	new_dbm(res, p);
	return 1;
}

/************************************************************
 *
 ************************************************************/
static int _global_time(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *s;
	int n, t;
	if (argc < 1) {
		t = time(NULL);
	} else if (argv->type == V_string) {
		warnif(!var_toString(argv, &s, &n),  "invalid argument-0");
		t = strtotime(s, n, 0);
	} else {
		t = var_toInt(argv);
	}
	new_int(res, t);
	return 1;
}
static int _global_gmtime(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *s;
	int n, t;
	if (argc < 1) {
		t = time(NULL);
	} else if (argv->type == V_string) {
		warnif(!var_toString(argv, &s, &n),  "invalid argument-0");
		t = strtotime(s, n, 1);
	} else {
		t = var_toInt(argv);
	}
	new_int(res, t);
	return 1;
}
static int _global_date(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char dst[BUFSIZ];
	char *s;
	int n, t;
	if (argc < 2) {
		t = time(NULL);
	} else if ((argv + 1)->type == V_string) {
		warnif(!var_toString(argv + 1, &s, &n),  "invalid argument-1");
		t = strtotime(s, n, 0);
	} else {
		t = var_toInt(argv + 1);
	}
	if (argc < 1) {
		s = "f";
		n = 1;
	} else {
		warnif(!var_toString(argv, &s, &n) || (n >= 256),  "invalid argument-0");
	}
	n = timetostr(dst, s, n, t, 0);
	if (!n) {
		return 0;
	}
	new_string_copy(ex, res, dst, n);
	return 1;
}
static int _global_gmdate(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char dst[BUFSIZ];
	char *s;
	int n, t;
	if (argc < 2) {
		t = time(NULL);
	} else if (argv->type == V_string) {
		warnif(!var_toString(argv + 1, &s, &n),  "invalid argument-1");
		t = strtotime(s, n, 1);
	} else {
		t = var_toInt(argv + 1);
	}
	if (argc < 1) {
		s = "f";
		n = 1;
	} else {
		warnif(!var_toString(argv, &s, &n) || (n >= 256),  "invalid argument-0");
	}
	n = timetostr(dst, s, n, t, 0);
	if (!n) {
		return 0;
	}
	new_string_copy(ex, res, dst, n);
	return 1;
}

/************************************************************
 *
 ************************************************************/
int _global_length(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	warnif(argc != 1,  "invalid arguments");
	return var_getLength(ex, argv, res);
}
int _global_isNull(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	warnif(argc != 1,  "invalid arguments");
	new_bool(res, argv->type == V_null);
	return 1;
}
int _global_isBool(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	warnif(argc != 1,  "invalid arguments");
	new_bool(res, argv->type == V_bool);
	return 1;
}
int _global_isChar(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	warnif(argc != 1,  "invalid arguments");
	new_bool(res, argv->type == V_char);
	return 1;
}
int _global_isNumber(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	warnif(argc != 1,  "invalid arguments");
	new_bool(res, (argv->type == V_int) || (argv->type == V_float));
	return 1;
}
int _global_isInt(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	warnif(argc != 1,  "invalid arguments");
	new_bool(res, argv->type == V_int);
	return 1;
}
int _global_isFloat(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	warnif(argc != 1,  "invalid arguments");
	new_bool(res, argv->type == V_float);
	return 1;
}
int _global_isString(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	warnif(argc != 1,  "invalid arguments");
	new_bool(res, argv->type == V_string);
	return 1;
}
int _global_isArray(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	warnif(argc != 1,  "invalid arguments");
	new_bool(res, argv->type == V_array);
	return 1;
}
int _global_isObject(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	warnif(argc != 1,  "invalid arguments");
	new_bool(res, argv->type == V_object);
	return 1;
}
int _global_toBool(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	warnif(argc != 1,  "invalid arguments");
	new_bool(res, var_toBool(argv));
	return 1;
}
int _global_toChar(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	warnif(argc != 1,  "invalid arguments");
	new_char(res, var_toChar(argv));
	return 1;
}
int _global_toInt(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	warnif(argc != 1,  "invalid arguments");
	new_int(res, var_toInt(argv));
	return 1;
}
int _global_toFloat(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	warnif(argc != 1,  "invalid arguments");
	new_float(res, var_toFloat(argv));
	return 1;
}
int _global_toNumber(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	double f;
	int i;
	warnif(argc != 1,  "invalid arguments");
	if (var_toNumber(argv, &f, &i) > 1) {
		new_float(res, f);
	} else {
		new_int(res, i);
	}
	return 1;
}
int _global_toString(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *s;
	int n;
	warnif(argc != 1,  "invalid arguments");
	warnif(!var_toString(argv, &s, &n), "invalid argument-0");
	new_string_copy(ex, res, s, n);
	return 1;
}

/************************************************************
 *
 ************************************************************/
typedef int (*global_func)(exec_t*, int, var_t*, var_t*);
static global_func is_global_func(int sym)
{
	switch (sym) {
#include "_global_call.h"
	}
	return NULL;
}
int global_callFunc(exec_t* ex, int sym, int argc, var_t* argv, var_t* res)
{
	global_func f = is_global_func(sym);
	if (f) {
		return (*f)(ex, argc, argv, res);
	}
	return -1;
}
