/*
 * Copyright (c) 2007, to-do. All rights reserved.
 */
#include "util.h"
#include "var.h"
#include "_sym.h"
#include "_ext.h"

/************************************************************
 *
 ************************************************************/
static int issymbols(char *s, int n)
{
	if (!n-- || !(isalpha(*s) || (*s == '_'))) return 0;
	for (s++; n-- > 0; s++) {
		if (!(isalnum(*s) || (*s == '_'))) return 0;
	}
	return 1;
}
static int skip_space(char** s, char *e)
{
	while ((*s < e) && isspace(**s)) (*s)++;
	return ((*s < e) ? **s : 0);
}
static void put_tabs(char **s, int *n, int d)
{
	(*s)[(*n)++] = '\n';
	while (d-- > 0) (*s)[(*n)++] = '\t';
}

/************************************************************
 *
 ************************************************************/
static int trace_length(var_t* v, int d)
{
	object_t* obj;
	array_t* arr;
	each_t e;
	char *s;
	int i, n, m;
	switch (v->type) {
	case V_object:
		obj = v->u.p;
		n = 2 + (obj->n * (d+3));
		for (i = 0; i < obj->n; i++) {
			if (object_each(obj, i, &e)) {
				n += symbol_to_string(e.k, &s, &m);
				n += trace_length(&e.v, d+1);
			}
		}
		return n;
	case V_array:
		arr = v->u.p;
		n = 2 + (arr->n * (d+2));
		for (i = 0; i < arr->n; i++) {
			n += trace_length(arr->v + i, d+1);
		}
		return n;
	case V_string:
		return (2 + (((string_t*)v->u.p)->n * 2));
	case V_char:
		return 4;
	case V_float:
		return 32;
	case V_int:
		return 16;
	case V_bool:
		return 1;
	default:
		break;
	}
	return 0;
}

static int var_to_trace(var_t* v, char **s, int *n, int d);

static int char_to_trace(int c, char **s, int *n)
{
	(*s)[(*n)++] = '\'';
	if (metac(&c, '\'')) {
		(*s)[(*n)++] = '\\';
	}
	(*s)[(*n)++] = c;
	(*s)[(*n)++] = '\'';
	(*s)[(*n)] = 0;
	return 1;
}
static int string_to_trace(string_t* str, char **s, int *n)
{
	int i, c;
	(*s)[(*n)++] = '"';
	for (i = 0; i < str->n; i++) {
		c = str->s[i];
		if (metac(&c, '"')) {
			(*s)[(*n)++] = '\\';
		}
		(*s)[(*n)++] = c;
	}
	(*s)[(*n)++] = '"';
	(*s)[(*n)] = 0;
	return 1;
}
static int symbol_to_trace(int k, char** s, int *n)
{
	string_t str;
	if (!symbol_to_string(k, &str.s, &str.n)) return 0;
	if (issymbols(str.s, str.n)) {
		/* strncpy */
		memcpy(*s + *n, str.s, str.n);
		*(*s + (*n += str.n)) = 0;
	} else {
		if (!string_to_trace(&str, s, n)) return 0;
	}
	return 1;
}
static int object_to_trace(object_t* obj, char **s, int *n, int d)
{
	each_t e;
	int i, num, len;
	(*s)[(*n)++] = '{';
	num = 0;
	for (i = 0; i < obj->n; i++) {
		len = (*n);
		if (num) (*s)[len++] = ',';
		put_tabs(s, &len, d+1);
		if (object_each(obj, i, &e)) {
			if (symbol_to_trace(e.k, s, &len)) {
				(*s)[len++] = ':';
				if (var_to_trace(&e.v, s, &len, d+1)) {
					(*n) = len;
					num++;
				}
			}
		}
	}
	if (num) put_tabs(s, n, d);
	(*s)[(*n)++] = '}';
	(*s)[(*n)] = 0;
	return 1;
}
static int array_to_trace(array_t* arr, char **s, int *n, int d)
{
	int i, num, len;
	(*s)[(*n)++] = '[';
	num = 0;
	for (i = 0; i < arr->n; i++) {
		len = (*n);
		if (num) (*s)[len++] = ',';
		put_tabs(s, &len, d+1);
		if (var_to_trace(arr->v + i, s, &len, d+1)) {
			(*n) = len;
			num++;
		}
	}
	if (num) put_tabs(s, n, d);
	(*s)[(*n)++] = ']';
	(*s)[(*n)] = 0;
	return 1;
}
static int float_to_trace(double f, char **s, int *n)
{
	(*n) += sprintf((*s) + (*n), "%g", f);
	return 1;
}
static int int_to_trace(int v, char **s, int *n)
{
	(*n) += sprintf((*s) + (*n), "%d", v);
	return 1;
}
static int var_to_trace(var_t* v, char **s, int *n, int d)
{
	switch (v->type) {
	case V_object:
		return object_to_trace(v->u.p, s, n, d);
	case V_array:
		return array_to_trace(v->u.p, s, n, d);
	case V_string:
		return string_to_trace(v->u.p, s, n);
	case V_float:
		return float_to_trace(v->u.f, s, n);
	case V_char:
		return char_to_trace(v->u.i, s, n);
	case V_int:
	case V_bool:
		return int_to_trace(v->u.i, s, n);
	default:
		break;
	}
	return 0;
}
int trace_to_bytes(exec_t* ex, var_t* v, char **s, int *n)
{
	int r;
	*s = malloc(trace_length(v, 0) + 1);
	*n = 0;
	r = var_to_trace(v, s, n, 0);
	if (!r) {
		if (*s) free(*s);
		*s = NULL;
		*n = 0;
		return 0;
	}
	assert((*s != NULL) && (*n > 0));
	return 1;
}

/************************************************************
 *
 ************************************************************/
static int trace_to_var(exec_t* ex, var_t* v, char** s, char *e);

static int trace_to_object(exec_t* ex, var_t* v, char** s, char *e)
{
	object_t* obj;
	int k;
	new_object(ex, v, NULL);
	obj = v->u.p;
	while (trace_to_var(ex, v, s, e)) {
		if (skip_space(s, e) != ':') break;
		(*s)++;
		k = var_toSymbol(v);
		if (!trace_to_var(ex, v, s, e)) break;
		object_set(ex, obj, k, v);
		if (skip_space(s, e) != ',') break;
		(*s)++;
	}
	if (*(*s)++ != '}') {
		object_free(ex, obj);
		return 0;
	}
	new_object(ex, v, obj);
	return 1;
}
static int trace_to_array(exec_t* ex, var_t* v, char** s, char* e)
{
	array_t* arr;
	new_array(ex, v, NULL);
	arr = v->u.p;
	while (trace_to_var(ex, v, s, e)) {
		array_push(ex, arr, v);
		if (skip_space(s, e) != ',') break;
		(*s)++;
	}
	if (*(*s)++ != ']') {
		array_free(ex, arr);
		return 0;
	}
	new_array(ex, v, arr);
	return 1;
}
static int trace_to_string(exec_t* ex, var_t* v, char** s, char* e)
{
	char *f = (*s);
	int c;
	string_t* str;
	while (((*s) < e) && (*(*s) != '"')) {
		if (*(*s)++ == '\\') (*s)++;
	}
	if (*(*s)++ != '"') {
		return 0;
	}
	new_string(ex, v, NULL);
	str = v->u.p;
	str->s = calloc((*s) - f, 1);
	while ((f < e) && (*f != '"')) {
		if ((c = *f++) == '\\') {
			c = unmetac(&f, 0);
		}
		str->s[str->n++] = c;
	}
	str->s[str->n] = 0;
	return 1;
}
static int trace_to_char(var_t* v, char** s, char* e)
{
	int c = 0;
	if (((*s) < e) && (*(*s) != '\'')) {
		if ((c = *(*s)++) == '\\') {
			c = unmetac(s, 0);
		}
	}
	if (*(*s)++ != '\'') {
		return 0;
	}
	new_char(v, c);
	return 1;
}
static int trace_to_symbol(var_t* v, char** s, char *e)
{
	char *f = (*s);
	while (((*s) < e) && (isalnum(*(*s)) || (*(*s)=='_'))) (*s)++;
	new_symbol(v, string_to_symbol(f, (*s) - f));
	return 1;
}
static int trace_to_number(var_t* v, char** s, char *e)
{
	char *f = (*s);
	while (((*s) < e) && isalnum(*(*s))) (*s)++;
	if (*(*s) == '.') {
		new_float(v, strtod(f, s));
	} else {
		new_int(v, strtol(f, NULL, 0));
	}
	return 1;
}

static int trace_to_var(exec_t* ex, var_t* v, char** s, char *e)
{
	int c = skip_space(s, e);
	if (c == '{') {
		(*s)++;
		return trace_to_object(ex, v, s, e);
	} else if (c == '[') {
		(*s)++;
		return trace_to_array(ex, v, s, e);
	} else if (c == '"') {
		(*s)++;
		return trace_to_string(ex, v, s, e);
	} else if (c == '\'') {
		(*s)++;
		return trace_to_char(v, s, e);
	} else if (isalpha(c) || (c == '_')) {
		return trace_to_symbol(v, s, e);
	} else if (isdigit(c) || (c == '.')) {
		return trace_to_number(v, s, e);
	}
	return 0;
}

int untrace_from_bytes(exec_t* ex, var_t* v, char *s, int n)
{
	int r;
	char *p = s;
	r = trace_to_var(ex, v, &p, s + n);
	if (!r) {
		new_null(v);
	}
	return r;
}
