/*
 * Copyright (c) 2007, to-do. All rights reserved.
 */
#include "util.h"
#include "var.h"
#include "_sym.h"
#include "_ext.h"

/************************************************************
 *
 ************************************************************/
name_t* g_names = NULL;

int string_to_symbol(char *s, int n)
{
	int k,i;
	k = strtoh(s, n);
	if (!k) return 0;
	switch (k) {
#include "_sym2sym.h"	
	}
	i = 0;
	if (g_names) {
		for (; g_names[i].s; i++) {
			if (g_names[i].k == k) {
				return k;
			}
		}
	}
	g_names = realloc(g_names, (i + 2) * sizeof(name_t));
	g_names[i].k = k;
	g_names[i].s = strndupok(s, n);
	g_names[i].n = n;
	memset(g_names + i + 1, 0, sizeof(name_t));
	return k;
}
int chars_to_symbol(char* s)
{
	return string_to_symbol(s, strlen(s));
}
int symbol_to_string(int k, char** s, int *n)
{
	int i;
	switch (k) {
#include "_sym2str.h"	
	}
	i = 0;
	if (g_names) {
		for (; g_names[i].s; i++) {
			if (g_names[i].k == k) {
				*s = g_names[i].s;
				*n = g_names[i].n;
				return *n;
			}
		}
	}
	*s = NULL;
	*n = 0;
	return 0;
}
char* symbol_to_chars(int k)
{
	char *s;
	int n;
	if (symbol_to_string(k, &s, &n)) {
		return s;
	}
	return NULL;
}
void free_names()
{
	int i = 0;
	if (g_names) {
		for (; g_names[i].s; i++) {
			free(g_names[i].s);
		}
		free(g_names);
		g_names = NULL;
	}
}

/************************************************************
 *
 ************************************************************/
void* alloc_link(exec_t* ex, size_t sz, proto_t* proto)
{
	link_t* p;
	p = calloc(1, sizeof(link_t) + sz);
	p->proto = proto;
	p->prev = ex->gc;
	if (ex->gc) {
		ex->gc->next = p;
	}
	ex->gc = p;
	return ((void*)p + sizeof(link_t));
}
void trace_link(link_t* p)
{
	void *v = (void*)p + sizeof(link_t);
	trace("%p: %d: %s\n", p, p->lock, p->proto->typestr);
	if (p->proto->trace) {
		(*p->proto->trace)(v);
	}
}
void trace_links(exec_t* ex)
{
	link_t* p = ex->gc;
	while (p) {
		trace_link(p);
		p = p->prev;
	}
}
void* free_link(exec_t* ex, void *v)
{
	link_t *p, *f;
	p = v - sizeof(link_t);
	if (p->proto && p->proto->clear) {
		(*p->proto->clear)(ex, v);
	}
	if ((f = p->prev) != NULL) {
		f->next = p->next;
	}
	if (p->next) {
		p->next->prev = f;
	} else {
		ex->gc = f;
	}
	free(p);
	return f;
}
void free_links(exec_t* ex, void *v)
{
	link_t *p;
	p = v - sizeof(link_t);
	while (p->next) {
		if (p->next->lock > 0) {
			p = p->next;
		} else {
			p = free_link(ex, (void*)p->next + sizeof(link_t));
		}
	}
	assert(p == ex->gc);
	free_link(ex, v);
	if (!ex->gc) {
		free_names();
	}
}

/************************************************************
 *
 ************************************************************/
void trace_var(var_t* v)
{
	trace("%s: %s\n", all_protos[v->type]->typestr, var_toChars(v));
	if (all_protos[v->type]->trace) {
		(*all_protos[v->type]->trace)(v->u.p);
	}
}
int lock_var(exec_t* ex, var_t* v)
{
	if (all_protos[v->type]->lock) {
		((link_t*)(v->u.p - sizeof(link_t)))->lock++;
		return 1;
	}
	return 0;
}
int unlock_var(exec_t* ex, var_t* v)
{
	if (all_protos[v->type]->lock) {
		((link_t*)(v->u.p - sizeof(link_t)))->lock--;
		return 1;
	}
	return 0;
}
int delete_var(exec_t* ex, var_t* v)
{
	if (all_protos[v->type]->lock) {
		if (((link_t*)(v->u.p - sizeof(link_t)))->lock <= 0) {
			free_link(ex, v->u.p);
			return 1;
		}
	}
	return 0;
}
void copy_var(var_t* v1, var_t* v2)
{
	memcpy(v1, v2, sizeof(var_t));
}
char* typeof_var(var_t* v)
{
	return all_protos[v->type]->typestr;
}

/************************************************************
 *
 ************************************************************/
int var_add(exec_t* ex, var_t* v1, var_t* v2, var_t* v3)
{
	if (v1->type == V_string || v2->type == V_string) {
		char *s1, *s2;
		int n1, n2;
		new_string(ex, v3, NULL);
		if (var_toString(v1, &s1, &n1)) {
			string_add(v3->u.p, s1, n1);
		}
		if (var_toString(v2, &s2, &n2)) {
			string_add(v3->u.p, s2, n2);
		}
	} else 
	if (v1->type == V_float || v2->type == V_float) {
		double r = var_toFloat(v1) + var_toFloat(v2);
		new_float(v3, r);
	} else {
		int r = var_toInt(v1) + var_toInt(v2);
		new_int(v3, r);
	}
	return 1;
}

/************************************************************
 *
 ************************************************************/
int var_cmpAsNumber(var_t* v1, var_t* v2)
{
	double f1, f2;
	f1 = var_toFloat(v1);
	f2 = var_toFloat(v2);
	return ((f1 < f2) ? -1 : (f1 > f2));
}
int var_cmpAsString(var_t* v1, var_t* v2)
{
	char *s1,*s2;
	int n1, n2;
	if (v1->u.p == v2->u.p) return 0;
	if (!var_toString(v1, &s1, &n1)
	|| !var_toString(v2, &s2, &n2)) {
		return (n1 < n2) ? -1 : (n1 > n2);
	}
	return strnncmp(s1, n1, s2, n2);
}
int var_cmp(var_t* v1, var_t* v2)
{
	if (v1->type == V_string || v2->type == V_string) {
		return var_cmpAsString(v1, v2);
	} else if (v1->type == V_float || v2->type == V_float) {
		return var_cmpAsNumber(v1, v2);
	}		
	return (var_toInt(v1) - var_toInt(v2));
}

/************************************************************
 *
 ************************************************************/
int var_toBool(var_t* v)
{
	if (all_protos[v->type]->toBool) {
		return (*all_protos[v->type]->toBool)(v);
	}
	return (v->type != 0);
}
int var_toChar(var_t* v)
{
	if (all_protos[v->type]->toChar) {
		return (*all_protos[v->type]->toChar)(v);
	}
	return 0;
}
int var_toInt(var_t* v)
{
	if (all_protos[v->type]->toInt) {
		return (*all_protos[v->type]->toInt)(v);
	}
	return 0;
}
double var_toFloat(var_t* v)
{
	if (all_protos[v->type]->toFloat) {
		return (*all_protos[v->type]->toFloat)(v);
	}
	return 0;
}
int var_toString(var_t* v, char **s, int *n)
{
	if (all_protos[v->type]->toString) {
		return (*all_protos[v->type]->toString)(v, s, n);
	}
	*s = NULL;
	*n = 0;
	return 0;
}
int var_toNumber(var_t* v, double *f, int *i)
{
	if (v->type == V_float) {
		*i = *f = v->u.f;
		return 2;
	} else if (all_protos[v->type]->toNumber) {
		return (*all_protos[v->type]->toNumber)(v, f, i);
	} else if (all_protos[v->type]->toInt) {
		*f = *i = (*all_protos[v->type]->toInt)(v);
		return 1;
	}
	*f = *i = 0;
	return 0;
}
char* var_toChars(var_t* v)
{
	char *s;
	int n;
	if (var_toString(v, &s, &n)) {
		return s;
	}
	return NULL;
}
int var_toSymbol(var_t* v)
{
	char *s;
	int n;
	if (v->type == V_symbol) {
		return v->u.i;
	} else if (var_toString(v, &s, &n)) {
		return string_to_symbol(s, n);
	}
	return 0;
}

/************************************************************
 *
 ************************************************************/
int var_getLength(exec_t* ex, var_t* v, var_t* res)
{
	if (all_protos[v->type]->getLength) {
		new_int(res, (*all_protos[v->type]->getLength)(ex, v->u.p));
		return 1;
	}
	new_null(res);
	warns("invalid getLength: %s\n", typeof_var(v));
	return 0;
}
int var_setLength(exec_t* ex, var_t* v, var_t* val)
{
	warnif (val->type != V_int, "invalid value");
	if (all_protos[v->type]->setLength) {
		return (*all_protos[v->type]->setLength)(ex, v->u.p, val->u.i);
	}
	warns("invalid setLength: %s\n", typeof_var(v));
	return 0;
}
int var_getItem(exec_t* ex, var_t* v, var_t* key, var_t* res)
{
	if ((key->type == V_symbol) && (key->u.i == K_length)) {
		return var_getLength(ex, v, res);
	}
	if (all_protos[v->type]->getItem) {
		return (*all_protos[v->type]->getItem)(ex, v->u.p, key, res);
	}
	new_null(res);
	return 0;
}
int var_setItem(exec_t* ex, var_t* v, var_t* key, var_t* val)
{
	if ((key->type == V_symbol) && (key->u.i == K_length)) {
		return var_setLength(ex, v, val);
	}
	if (all_protos[v->type]->setItem) {
		return (*all_protos[v->type]->setItem)(ex, v->u.p, key, val);
	}
	warns("invalid setItem: %s\n", typeof_var(v));
	return 0;
}
int var_delItem(exec_t* ex, var_t* v, var_t* key, var_t* res)
{
	if (all_protos[v->type]->delItem) {
		return (*all_protos[v->type]->delItem)(ex, v->u.p, key, res);
	}
	if (res) new_null(res);
	warns("invalid delItem: %s\n", typeof_var(v));
	return 0;
}
int var_eachItem(exec_t* ex, var_t* v, int i, var_t* key, var_t* val)
{
	if (all_protos[v->type]->eachItem) {
		return (*all_protos[v->type]->eachItem)(ex, v->u.p, i, key, val);
	}
	warns("invalid eachItem: %s\n", typeof_var(v));
	return 0;
}
int var_callFunc(exec_t* ex, var_t* v, int sym, int argc, var_t* argv, var_t* res)
{
	int r;
	if (!v) {
		r = global_callFunc(ex, sym, argc, argv, res);
		if (r >= 0) return r;
		warns("no such function: %s(%d)\n", symbol_to_chars(sym), argc);
	} else {
		if (all_protos[v->type]->callFunc) {
			r = (*all_protos[v->type]->callFunc)(ex, v->u.p, sym, argc, argv, res);
			if (r >= 0) return r;
		}
		if (argc == 0) {
			r = global_callFunc(ex, sym, 1, v, res);
			if (r >= 0) return r;
		}
		warns("no such function: %s.%s(%d)\n", typeof_var(v), symbol_to_chars(sym), argc);
	}
	return -1;
}

/************************************************************
 *
 ************************************************************/
void new_null (var_t* v)
{
	v->type = 0;
}
void new_bool(var_t* v, int i)
{
	v->type = V_bool;
	v->u.i = i != 0;
}
void new_char (var_t* v, int i)
{
	v->type = V_char;
	v->u.i = i & 0xFF;
}
void new_int (var_t* v, int i)
{
	v->type = V_int;
	v->u.i = i;
}
void new_symbol (var_t* v, int i)
{
	v->type = V_symbol;
	v->u.i = i;
}
void new_float (var_t* v, double f)
{
	v->type = V_float;
	v->u.f = f;
}
void new_label (var_t* v, char *p)
{
	v->type = V_label;
	v->u.p = p;
}
void new_function (var_t* v, char *p)
{
	v->type = V_function;
	v->u.p = p;
}
void new_class (var_t* v, char *p)
{
	v->type = V_class;
	v->u.p = p;
}
void new_file(var_t* v, FILE* fp)
{
	v->type = V_file;
	v->u.p = fp;
}
void new_dbm(var_t* v, void *p)
{
	v->type = V_dbm;
	v->u.p = p;
}
void new_string_with (exec_t* ex, var_t* v, char *s, int n)
{
	string_t* str;
	v->type = V_string;
	str = v->u.p = string_alloc(ex);
	if (s && n) {
		str->s = s;
		str->n = n;
	}
}
void new_string_copy (exec_t* ex, var_t* v, char *s, int n)
{
	string_t* str;
	v->type = V_string;
	str = v->u.p = string_alloc(ex);
	if (s && n) {
		str->s = strndupok(s, n);
		str->n = n;
	}
}
void new_string (exec_t* ex, var_t* v, void *p)
{
	if (!p) p = string_alloc(ex);
	v->type = V_string;
	v->u.p = p;
}
void new_regex (exec_t* ex, var_t* v, char *s, int n, int i)
{
	v->type = V_regex;
	v->u.p = regex_alloc(ex);
	if (!regex_comp(v->u.p, s, n, i)) {
		/* make null */;
		regex_free(ex, v->u.p);
		v->type = 0;
	}
}
void new_array (exec_t* ex, var_t* v, void *p)
{
	if (!p) p = array_alloc(ex);
	v->type = V_array;
	v->u.p = p;
}
void new_refer (exec_t* ex, var_t* v, void *p)
{
	if (!p) p = refer_alloc(ex);
	v->type = V_refer;
	v->u.p = p;
}
void new_object (exec_t* ex, var_t* v, void* p)
{
	if (!p) p = object_alloc(ex);
	v->type = V_object;
	v->u.p = p;
}
void new_frame (exec_t* ex, var_t* v, void *p)
{
	if (!p) p = frame_alloc(ex);
	v->type = V_frame;
	v->u.p = p;
}

/************************************************************
 *
 ************************************************************/
static int int_toBool(var_t* v)
{
	return (v->u.i != 0);
}
static int int_toChar(var_t* v)
{
	return (v->u.i & 0xFF);
}
static int int_toInt(var_t* v)
{
	return v->u.i;
}
static double int_toFloat(var_t* v)
{
	return (double)v->u.i;
}

/************************************************************
 *
 ************************************************************/
static int float_toBool(var_t* v)
{
	return (v->u.f != 0);
}
static int float_toChar(var_t* v)
{
	return ((int)v->u.f & 0xFF);
}
static int float_toInt(var_t* v)
{
	return (int)v->u.f;
}
static double float_toFloat(var_t* v)
{
	return v->u.f;
}

/************************************************************
 *
 ************************************************************/
static int bool_toString(var_t* v, char **s, int *n)
{
	if (v->u.i) {
		*s = "true";
		*n = 4;
	} else {
		*s = "false";
		*n = 5;
	}
	return *n;
}
static int char_toString(var_t* v, char **s, int *n)
{
	return (*n = sprintf(*s = tmpbuf(), "%c", (v->u.i & 0xFF)));
}
static int int_toString(var_t* v, char **s, int *n)
{
	return (*n = sprintf(*s = tmpbuf(), "%d", v->u.i));
}
static int float_toString(var_t* v, char **s, int *n)
{
	return (*n = sprintf(*s = tmpbuf(), "%g", v->u.f));
}
static int symbol_toString(var_t* v, char **s, int *n)
{
	return symbol_to_string(v->u.i, s, n);
}

/************************************************************
 *
 ************************************************************/
proto_t null_proto = {
"null", /* typestr */
0, /* lock */
NULL, /* trace */
NULL, /* clear */
NULL, /* toBool */
NULL, /* toChar */
NULL, /* toInt */
NULL, /* toFloat */
NULL, /* toString */
NULL, /* toNumber */
NULL, /* getLength */
NULL, /* setLegnth */
NULL, /* callFunc */
NULL, /* getItem */
NULL, /* setItem */
NULL, /* delItem */
NULL, /* eachItem */
0
};

proto_t bool_proto = {
"bool", /* typestr */
0, /* lock */
NULL, /* trace */
NULL, /* clear */
int_toBool, /* toBool */
int_toChar, /* toChar */
int_toInt, /* toInt */
int_toFloat, /* toFloat */
bool_toString, /* toString */
NULL, /* toNumber */
NULL, /* getLength */
NULL, /* setLegnth */
NULL, /* callFunc */
NULL, /* getItem */
NULL, /* setItem */
NULL, /* delItem */
NULL, /* eachItem */
0
};

proto_t char_proto = {
"char", /* typestr */
0, /* lock */
NULL, /* trace */
NULL, /* clear */
int_toBool, /* toBool */
int_toChar, /* toChar */
int_toInt, /* toInt */
int_toFloat, /* toFloat */
char_toString, /* toString */
NULL, /* toNumber */
NULL, /* getLength */
NULL, /* setLegnth */
NULL, /* callFunc */
NULL, /* getItem */
NULL, /* setItem */
NULL, /* delItem */
NULL, /* eachItem */
0
};

proto_t int_proto = {
"int", /* typestr */
0, /* lock */
NULL, /* trace */
NULL, /* clear */
int_toBool, /* toBool */
int_toChar, /* toChar */
int_toInt, /* toInt */
int_toFloat, /* toFloat */
int_toString, /* toString */
NULL, /* toNumber */
NULL, /* getLength */
NULL, /* setLegnth */
NULL, /* callFunc */
NULL, /* getItem */
NULL, /* setItem */
NULL, /* delItem */
NULL, /* eachItem */
0
};

proto_t symbol_proto = {
"symbol", /* typestr */
0, /* lock */
NULL, /* trace */
NULL, /* clear */
NULL, /* toBool */
NULL, /* toChar */
NULL, /* toInt */
NULL, /* toFloat */
symbol_toString, /* toString */
NULL, /* toNumber */
NULL, /* getLength */
NULL, /* setLegnth */
NULL, /* callFunc */
NULL, /* getItem */
NULL, /* setItem */
NULL, /* delItem */
NULL, /* eachItem */
0
};

proto_t float_proto = {
"float", /* typestr */
0, /* lock */
NULL, /* trace */
NULL, /* clear */
float_toBool, /* toBool */
float_toChar, /* toChar */
float_toInt, /* toInt */
float_toFloat, /* toFloat */
float_toString, /* toString */
NULL, /* toNumber */
NULL, /* getLength */
NULL, /* setLegnth */
NULL, /* callFunc */
NULL, /* getItem */
NULL, /* setItem */
NULL, /* delItem */
NULL, /* eachItem */
0
};

proto_t string_proto = {
"string", /* typestr */
1, /* lock */
string_trace, /* trace */
string_clear, /* clear */
string_toBool, /* toBool */
string_toChar, /* toChar */
string_toInt, /* toInt */
string_toFloat, /* toFloat */
string_toString, /* toString */
string_toNumber, /* toNumber */
string_getLength, /* getLength */
string_setLength, /* setLegnth */
string_callFunc, /* callFunc */
string_getItem, /* getItem */
string_setItem, /* setItem */
string_delItem, /* delItem */
string_eachItem, /* eachItem */
0
};

proto_t regex_proto = {
"regex", /* typestr */
1, /* lock */
regex_trace, /* trace */
regex_clear, /* clear */
NULL, /* toBool */
NULL, /* toChar */
NULL, /* toInt */
NULL, /* toFloat */
NULL, /* toString */
NULL, /* toNumber */
NULL, /* getLength */
NULL, /* setLegnth */
NULL, /* callFunc */
NULL, /* getItem */
NULL, /* setItem */
NULL, /* delItem */
NULL, /* eachItem */
0
};

proto_t array_proto = {
"array", /* typestr */
1, /* lock */
array_trace, /* trace */
array_clear, /* clear */
array_toBool, /* toBool */
array_toChar, /* toChar */
array_toInt, /* toInt */
array_toFloat, /* toFloat */
array_toString, /* toString */
NULL, /* toNumber */
array_getLength, /* getLength */
array_setLength, /* setLegnth */
array_callFunc, /* callFunc */
array_getItem, /* getItem */
array_setItem, /* setItem */
array_delItem, /* delItem */
array_eachItem, /* eachItem */
0
};

proto_t refer_proto = {
"refer", /* typestr */
1, /* lock */
refer_trace, /* trace */
refer_clear, /* clear */
NULL, /* toBool */
NULL, /* toChar */
NULL, /* toInt */
NULL, /* toFloat */
NULL, /* toString */
NULL, /* toNumber */
NULL, /* getLength */
NULL, /* setLegnth */
NULL, /* callFunc */
NULL, /* getItem */
NULL, /* setItem */
NULL, /* delItem */
NULL, /* eachItem */
0
};

proto_t object_proto = {
"object", /* typestr */
1, /* lock */
object_trace, /* trace */
object_clear, /* clear */
object_toBool, /* toBool */
object_toChar, /* toChar */
object_toInt, /* toInt */
object_toFloat, /* toFloat */
object_toString, /* toString */
NULL, /* toNumber */
object_getLength, /* getLength */
object_setLength, /* setLegnth */
object_callFunc, /* callFunc */
object_getItem, /* getItem */
object_setItem, /* setItem */
object_delItem, /* delItem */
object_eachItem, /* eachItem */
0
};

proto_t frame_proto = {
"frame", /* typestr */
0, /* lock */
frame_trace, /* trace */
frame_clear, /* clear */
NULL, /* toBool */
NULL, /* toChar */
NULL, /* toInt */
NULL, /* toFloat */
NULL, /* toString */
NULL, /* toNumber */
NULL, /* getLength */
NULL, /* setLegnth */
NULL, /* callFunc */
frame_getItem, /* getItem */
frame_setItem, /* setItem */
frame_delItem, /* delItem */
NULL, /* eachItem */
0
};

proto_t file_proto = {
"file", /* typestr */
0, /* lock */
NULL, /* trace */
NULL, /* clear */
NULL, /* toBool */
NULL, /* toChar */
NULL, /* toInt */
NULL, /* toFloat */
NULL, /* toString */
NULL, /* toNumber */
NULL, /* getLength */
NULL, /* setLegnth */
file_callFunc, /* callFunc */
NULL, /* getItem */
NULL, /* setItem */
NULL, /* delItem */
NULL, /* eachItem */
0
};

proto_t dbm_proto = {
"dbm", /* typestr */
0, /* lock */
NULL, /* trace */
NULL, /* clear */
NULL, /* toBool */
NULL, /* toChar */
NULL, /* toInt */
NULL, /* toFloat */
NULL, /* toString */
NULL, /* toNumber */
NULL, /* getLength */
NULL, /* setLegnth */
dbm_callFunc, /* callFunc */
dbm_getItem, /* getItem */
dbm_setItem, /* setItem */
dbm_delItem, /* delItem */
dbm_eachItem, /* eachItem */
0
};

proto_t function_proto = {
"function", /* typestr */
0, /* lock */
NULL, /* trace */
NULL, /* clear */
NULL, /* toBool */
NULL, /* toChar */
NULL, /* toInt */
NULL, /* toFloat */
NULL, /* toString */
NULL, /* toNumber */
NULL, /* getLength */
NULL, /* setLegnth */
NULL, /* callFunc */
NULL, /* getItem */
NULL, /* setItem */
NULL, /* delItem */
NULL, /* eachItem */
0
};

proto_t class_proto = {
"class", /* typestr */
0, /* lock */
NULL, /* trace */
NULL, /* clear */
NULL, /* toBool */
NULL, /* toChar */
NULL, /* toInt */
NULL, /* toFloat */
NULL, /* toString */
NULL, /* toNumber */
NULL, /* getLength */
NULL, /* setLegnth */
NULL, /* callFunc */
NULL, /* getItem */
NULL, /* setItem */
NULL, /* delItem */
NULL, /* eachItem */
0
};

proto_t label_proto = {
"label", /* typestr */
0, /* lock */
NULL, /* trace */
NULL, /* clear */
NULL, /* toBool */
NULL, /* toChar */
NULL, /* toInt */
NULL, /* toFloat */
NULL, /* toString */
NULL, /* toNumber */
NULL, /* getLength */
NULL, /* setLegnth */
NULL, /* callFunc */
NULL, /* getItem */
NULL, /* setItem */
NULL, /* delItem */
NULL, /* eachItem */
0
};

proto_t* all_protos[] = {
	& null_proto,
	& bool_proto,
	& char_proto,
	& int_proto,
	& symbol_proto,
	& float_proto,
	& file_proto,
	& dbm_proto,
	& function_proto,
	& class_proto,
	& label_proto,
	& frame_proto,
	& string_proto,
	& regex_proto,
	& array_proto,
	& refer_proto,
	& object_proto,
	NULL
};

