/*
 * Copyright (c) 2007, to-do. All rights reserved.
 */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include "jis.h"

/************************************************************
 *
 ************************************************************/
static unsigned short _sjis_to_jis(int c)
{
	int a,b;
	a = c >> 8;
	b = c & 0xff;
	if (a >= 0xe0) a -= 0x40;
	if (b >= 0x80) b--;
	c = ((a - 0x81) * 0xbc) + (b - 0x40);
	a = (c / 0x5e) + 0x21;
	b = (c % 0x5e) + 0x21;
	return ((a << 8) | b);
}
static unsigned short _sjis_to_ucs(int c)
{
	return _jis_to_ucs[_sjis_to_jis(c)];
}
static unsigned short _jis_to_sjis(int c)
{
	int a,b;
	a = c >> 8;
	b = c & 0xff;
	c = ((a - 0x21) * 0x5e) + (b - 0x21);
	a = (c / 0xbc) + 0x81;
	b = (c % 0xbc) + 0x40;
	if (a >= 0xa0) a += 0x40;
	if (b >= 0x7f) b++;
	return ((a << 8) | b);
}
static inline unsigned short _ucs_to_sjis(int c)
{
	return _jis_to_sjis(_ucs_to_jis[c]);
}
static inline unsigned short _jis_to_euc(int c)
{
	return (c | 0x8080);
}
static inline unsigned short _euc_to_jis(int c)
{
	return (c & 0x7f7f);
}
static inline unsigned short _euc_to_sjis(int c)
{
	return _euc_to_jis(_jis_to_sjis(c));
}
static inline unsigned short _sjis_to_euc(int c)
{
	return _sjis_to_jis(_jis_to_euc(c));
}
static inline unsigned short _ucs_to_euc(int c)
{
	return _jis_to_euc(_ucs_to_jis[c]);
}
static inline unsigned short _euc_to_ucs(int c)
{
	return _jis_to_ucs[_euc_to_jis(c)];
}

/************************************************************
 *
 ************************************************************/
static int is_utf8(char *s, char *e)
{
	int c, n = 0;
	for (; (s < e) && ((c = *s) != 0); s++) {
		if (c <= 0x7e) break;
		if (c >= 0xe0) {
			c = *(++s);
			if (!(c >= 0x80 && c<0xc0)) return 0;
			c = *(++s);
			if (!(c >= 0x80 && c<0xc0)) return 0;
			n++;
		} else if (c >= 0xc0) {
			c = *(++s);
			if (!(c >= 0x80 && c<0xc0)) return 0;
			n++;
		} else {
			return 0;
		}
	}
	return (n > 0);
}
static int is_euc(char *s, char *e)
{
	int c, n = 0;
	for (; (s < e) && ((c = *s) != 0); s++) {
		if (c <= 0x7e) break;
		if ((c >= 0xa1) && (c <= 0xfe)) {
			c = *(++s);
			if (!(c >= 0xa1 && c<=0xfe)) return 0;
			n++;
		} else if (c == 0x8e) {
			c = *(++s);
			if (!(c >= 0xa1 && c<=0xdf)) return 0;
			;
		} else {
			return 0;
		}
	}
	return (n > 0);
}
static int is_sjis(char *s, char *e)
{
	int c, n = 0;
	for (; (s < e) && ((c = *s) != 0); s++) {
		if (c <= 0x7e) break;
		if ((c>=0x81 && c<=0x9f) || (c>=0xe0 && c<=0xfc)) {
			c = *(++s);
			if (!(c>=0x40 && c<=0xfc)) return 0;
			n++;
		} else if (c >= 0xa1 && c<=0xdf) {
			;
		} else {
			return 0;
		}
	}
	return (n > 0);
}

/************************************************************
 *
 ************************************************************/
static int putjisesc(char *d, int j)
{
	d[0] = 0x1b;
	if (j == 2) {
		d[1] = '$', d[2] = 'B';
	} else if (j == 1) {
		d[1] = '(', d[2] = 'I';
	} else {
		d[1] = '(', d[2] = 'B';
	}
	return 3;
}
static int getjisesc(char **s)
{
	int c1, c2;
	c1 = *(*s)++;
	c2 = *(*s)++;
	return (((c1 == '$') << 1) | (c2 == 'I'));
}

/************************************************************
 *
 ************************************************************/
static int utf8_to_jis(char *d, char *s, int n)
{
	char *e;
	int c, j;
	e = s + n;
	n = j = 0;
	while ((s < e) && ((c = *s++) != 0)) {
		if (c >= 0xe0) {
			c =  (c    & 0x0f) << 12;
			c |= (*s++ & 0x3f) << 6;
			c |= (*s++ & 0x3f);
		} else if (c >= 0xc0) {
			c =  (c    & 0x1f) << 6;
			c |= (*s++ & 0x3f);
		}
		if ((c >= 0xff61) && (c <= 0xff9f)) {
			/* half kana */
			c &= 0x3f;
			if (!(j & 1)) {
				if (!(j >> 2)) {
					j |= c << 2;
					continue;
				}
				n += putjisesc(d + n, 1);
				d[n++] = j >> 2;
				j = 1;
			}
			d[n++]=c;
		} else if (c >= 0x100) {
			if (!(j & 2)) {
				n += putjisesc(d + n, j = 2);
			}
			c = _ucs_to_jis[c];
			d[n++] = c >> 8;
			d[n++] = c & 0xff;
		} else {
			if (j & 3) {
				n += putjisesc(d + n, j = 0);
			}
			d[n++]=c;
		}
	}
	if (j & 3) {
		n += putjisesc(d + n, 0);
	}
	d[n]=0;
	return n;
}
static int sjis_to_jis(char *d, char *s, int n)
{
	char *e;
	int c, j;
	e = s + n;
	n = j = 0;
	while ((s < e) && ((c = *s++) != 0)) {
		if ((c >= 0xa1) && (c <= 0xdf)) {
			/* half kana */
			c &= 0x7f;
			if (!(j & 1)) {
				if (!(j >> 2)) {
					j |= c << 2;
					continue;
				}
				n += putjisesc(d + n, 1);
				d[n++] = j >> 2;
				j = 1;
			}
			d[n++]=c;
		} else if (c >= 0x81) {
			c = (c << 8) | *s++;
			if (!(j & 2)) {
				n += putjisesc(d + n, j = 2);
			}
			c = _sjis_to_jis(c);
			d[n++] = c >> 8;
			d[n++] = c & 0xff;
		} else {
			if (j & 3) {
				n += putjisesc(d + n, j = 0);
			}
			d[n++]=c;
		}
	}
	if (j & 3) {
		n += putjisesc(d + n, 0);
	}
	d[n]=0;
	return n;
}
static int euc_to_jis(char *d, char *s, int n)
{
	char *e;
	int c, j;
	e = s + n;
	n = j = 0;
	while ((s < e) && ((c = *s++) != 0)) {
		if (c == 0x8e) {
			/* half kana */
			c = *s++ & 0x7f;
			if (!(j & 1)) {
				if (!(j >> 2)) {
					j |= c << 2;
					continue;
				}
				n += putjisesc(d + n, 1);
				d[n++] = j >> 2;
				j = 1;
			}
			d[n++]=c;
		} else if (c >= 0xa1) {
			c = (c << 8) | *s++;
			if (!(j & 2)) {
				n += putjisesc(d + n, j = 2);
			}
			c = _euc_to_jis(c);
			d[n++] = c >> 8;
			d[n++] = c & 0xff;
		} else {
			if (j & 3) {
				n += putjisesc(d + n, j = 0);
			}
			d[n++]=c;
		}
	}
	if (j & 3) {
		n += putjisesc(d + n, 0);
	}
	d[n]=0;
	return n;
}

/************************************************************
 *
 ************************************************************/
static int utf8_to_sjis(char *d, char *s, int n)
{
	char *e;
	int c;
	e = s + n;
	n = 0;
	while ((s < e) && ((c = *s++) != 0)) {
		if (c >= 0xe0) {
			c =  (c    & 0x0f) << 12;
			c |= (*s++ & 0x3f) << 6;
			c |= (*s++ & 0x3f);
		} else if (c >= 0xc0) {
			c =  (c    & 0x1f) << 6;
			c |= (*s++ & 0x3f);
		}
		if ((c >= 0xff61) && (c <= 0xff9f)) {
			/* half kana */
			c = (c & 0x3f) | 0x80;
			d[n++]=c;
		} else if (c >= 0x100) {
			c = _ucs_to_sjis(c);
			d[n++] = c >> 8;
			d[n++] = c & 0xff;
		} else {
			d[n++]=c;
		}
	}
	d[n]=0;
	return n;
}
static int euc_to_sjis(char *d, char *s, int n)
{
	char *e;
	int c;
	e = s + n;
	n = 0;
	while ((s < e) && ((c = *s++) != 0)) {
		if (c == 0x8e) {
			/* half kana */
			c = *s++ & 0x7f;
			d[n++]=c;
		} else if (c >= 0xa1) {
			c = (c << 8) | *s++;
			c = _euc_to_sjis(c);
			d[n++] = c >> 8;
			d[n++] = c & 0xff;
		} else {
			d[n++]=c;
		}
	}
	d[n]=0;
	return n;
}
static int jis_to_sjis(char *d, char *s, int n)
{
	char *e;
	int c, j;
	e = s + n;
	n = j = 0;
	while ((s < e) && ((c = *s++) != 0)) {
		if (c == 0x1b) {
			j = getjisesc(&s);
			c = *s++;
		}
		if (j & 2) {
			c = (c << 8) | *s++;
			c = _jis_to_sjis(c);
			d[n++] = c >> 8;
			d[n++] = c & 0xFF;
		} else if (j & 1) {
			c |= 0x80;
			d[n++] = c;
		} else {
			d[n++] = c;
		}
	}
	d[n] = 0;
	return n;
}

/************************************************************
 *
 ************************************************************/
static int utf8_to_euc(char *d, char *s, int n)
{
	char *e;
	int c;
	e = s + n;
	n = 0;
	while ((s < e) && ((c = *s++) != 0)) {
		if (c >= 0xe0) {
			c =  (c    & 0x0f) << 12;
			c |= (*s++ & 0x3f) << 6;
			c |= (*s++ & 0x3f);
		} else if (c >= 0xc0) {
			c =  (c    & 0x1f) << 6;
			c |= (*s++ & 0x3f);
		}
		if ((c >= 0xff61) && (c <= 0xff9f)) {
			/* half kana */
			c = (c & 0x3f) | 0x80;
			d[n++] = 0x8e;
			d[n++] = c;
		} else if (c >= 0x100) {
			c = _ucs_to_euc(c);
			d[n++] = c >> 8;
			d[n++] = c & 0xff;
		} else {
			d[n++]=c;
		}
	}
	d[n]=0;
	return n;
}
static int sjis_to_euc(char *d, char *s, int n)
{
	char *e;
	int c;
	e = s + n;
	n = 0;
	while ((s < e) && ((c = *s++) != 0)) {
		if ((c >= 0xa1) && (c <= 0xdf)) {
			/* half kana */
			d[n++]=0x8e;
			d[n++]=c;
		} else if (c >= 0x81) {
			c = (c << 8) | *s++;
			c = _sjis_to_euc(c);
			d[n++] = c >> 8;
			d[n++] = c & 0xff;
		} else {
			d[n++]=c;
		}
	}
	d[n]=0;
	return n;
}
static int jis_to_euc(char *d, char *s, int n)
{
	char *e;
	int c, j;
	e = s + n;
	n = j = 0;
	while ((s < e) && ((c = *s++) != 0)) {
		if (c == 0x1b) {
			j = getjisesc(&s);
			c = *s++;
		}
		if (j & 2) {
			c = (c << 8) | *s++;
			c = _jis_to_euc(c);
			d[n++] = c >> 8;
			d[n++] = c & 0xFF;
		} else if (j & 1) {
			c |= 0x80;
			d[n++] = 0x8e;
			d[n++] = c;
		} else {
			d[n++] = c;
		}
	}
	d[n] = 0;
	return n;
}

/************************************************************
 *
 ************************************************************/
static int jis_to_utf8(char *d, char *s, int n)
{
	char *e;
	int c, j;
	e = s + n;
	n = j = 0;
	while ((s < e) && ((c = *s++) != 0)) {
		if (c == 0x1b) {
			j = getjisesc(&s);
			continue;
		}
		if (j & 2) {
			c = (c << 8) | *s++;
			c = _jis_to_ucs[c];
		} else if (j & 1) {
			c |= 0xff40;
		}
		if (c >= 0x800) {
			d[n++] = 0xe0 | ((c >> 12) & 0x0f);
			d[n++] = 0x80 | ((c >>  6) & 0x3f);
			d[n++] = 0x80 | ((c >>  0) & 0x3f);
		} else if (c >= 0x80) {
			d[n++] = 0xc0 | ((c >>  6) & 0x1f);
			d[n++] = 0x80 | ((c >>  0) & 0x3f);
		} else {
			d[n++] = c;
		}
	}
	d[n] = 0;
	return n;
}
static int sjis_to_utf8(char *d, char *s, int n)
{
	char *e;
	int c;
	e = s + n;
	n = 0;
	while ((s < e) && ((c = *s++) != 0)) {
		if ((c>=0xa1) && (c<=0xdf)) {
			c = (c & 0x7f) | 0xff40;
		} else if (c >= 0x81) {
			c = (c << 8) | *s++;
			c = _sjis_to_ucs(c);
		}
		if (c >= 0x800) {
			d[n++] = 0xe0 | ((c >> 12) & 0x0f);
			d[n++] = 0x80 | ((c >>  6) & 0x3f);
			d[n++] = 0x80 | ((c >>  0) & 0x3f);
		} else if (c >= 0x80) {
			d[n++] = 0xc0 | ((c >>  6) & 0x1f);
			d[n++] = 0x80 | ((c >>  0) & 0x3f);
		} else {
			d[n++] = c;
		}
	}
	d[n] = 0;
	return n;
}
static int euc_to_utf8(char *d, char *s, int n)
{
	char *e;
	int c;
	e = s + n;
	n = 0;
	while ((s < e) && ((c = *s++) != 0)) {
		if (c == 0x8e) {
			c = (*s++ & 0x7f) | 0xff40;
		} else if (c >= 0xa1) {
			c = (c << 8) | *s++;
			c = _euc_to_ucs(c);
		}
		if (c >= 0x800) {
			d[n++] = 0xe0 | ((c >> 12) & 0x0f);
			d[n++] = 0x80 | ((c >>  6) & 0x3f);
			d[n++] = 0x80 | ((c >>  0) & 0x3f);
		} else if (c >= 0x80) {
			d[n++] = 0xc0 | ((c >>  6) & 0x1f);
			d[n++] = 0x80 | ((c >>  0) & 0x3f);
		} else {
			d[n++] = c;
		}
	}
	d[n] = 0;
	return n;
}

/************************************************************
 *
 ************************************************************/
int jcodeof(char *s, int n)
{
	char *e = s + n;
	int c;
	for (; (s < e) && ((c = *s) != 0); s++) {
		if (c == 0x1b) return 'j';
		else if (c >= 0x80) break;
	}
	if (!*s) return 0;
	if (is_utf8(s, e)) return 'u';
	if (is_euc(s, e)) return 'e';
	if (is_sjis(s, e)) return 's';
	return 's';
}
int to_euc(char *d, char *s, int n)
{
	int j;
	j = jcodeof(s, n);
	if (j == 'u') {
		return utf8_to_euc(d, s, n);
	} else if (j == 's') {
		return sjis_to_euc(d, s, n);
	} else if (j == 'j') {
		return jis_to_euc(d, s, n);
	}
	strncpy(d, s, n);
	return n;
}
int to_sjis(char *d, char *s, int n)
{
	int j;
	j = jcodeof(s, n);
	if (j == 'u') {
		return utf8_to_sjis(d, s, n);
	} else if (j == 'e') {
		return euc_to_sjis(d, s, n);
	} else if (j == 'j') {
		return jis_to_sjis(d, s, n);
	}
	strncpy(d, s, n);
	return n;
}
int to_jis(char *d, char *s, int n)
{
	int j;
	j = jcodeof(s, n);
	if (j == 'u') {
		return utf8_to_jis(d, s, n);
	} else if (j == 's') {
		return sjis_to_jis(d, s, n);
	} else if (j == 'e') {
		return euc_to_jis(d, s, n);
	}
	strncpy(d, s, n);
	return n;
}
int to_utf8(char *d, char *s, int n)
{
	int j;
	j = jcodeof(s, n);
	if (j == 'j') {
		return jis_to_utf8(d, s, n);
	} else if (j == 's') {
		return sjis_to_utf8(d, s, n);
	} else if (j == 'e') {
		return euc_to_utf8(d, s, n);
	}
	strncpy(d, s, n);
	return n;
}

/************************************************************
 *
 ************************************************************/
int jconv(char **d, char *s, int n, int j)
{
	switch (tolower(j)) {
	case 'j':
		*d = malloc((n * 4) + 1);
		n = to_jis(*d, s, n);
		break;
	case 's':
		*d = malloc(n + 1);
		n = to_sjis(*d, s, n);
		break;
	case 'e':
		*d = malloc((n * 2) + 1);
		n = to_euc(*d, s, n);
		break;
	case 'u':
	default:
		*d = malloc((n * 2) + 1);
		n = to_utf8(*d, s, n);
		break;
	}
	if (!n && *d) {
		free(*d);
		*d = NULL;
	}
	return n;
}
