/*
 * Copyright (c) 2007, to-do. All rights reserved.
 */
#include "util.h"
#include "var.h"
#include "_sym.h"
#include "_ext.h"

/************************************************************
 *
 ************************************************************/
int string_to_symbol(exec_t* ex, char *s, int n)
{
	name_t* ns;
	int k,i;
	k = strtoh(s, n);
	if (!k) return 0;
	switch (k) {
#include "_sym2sym.h"	
	}
	i = 0;
	if ((ns = ex->ns) != NULL) {
		for (; ns->s; ns++, i++) {
			if (ns->k == k) {
				return k;
			}
		}
	}
	ex->ns = realloc(ex->ns, (i + 2) * sizeof(name_t));
	ns = ex->ns + i;
	ns->k = k;
	ns->s = strndupok(s, n);
	ns->n = n;
	memset(ns + 1, 0, sizeof(name_t));
	return k;
}
int chars_to_symbol(exec_t* ex, char* s)
{
	return string_to_symbol(ex, s, strlen(s));
}
int symbol_to_string(exec_t* ex, int k, char** s, int *n)
{
	name_t* ns;
	switch (k) {
#include "_sym2str.h"	
	}
	if ((ns = ex->ns) != NULL) {
		for (; ns->s; ns++) {
			if (ns->k == k) {
				*s = ns->s;
				*n = ns->n;
				return *n;
			}
		}
	}
	*s = NULL;
	*n = 0;
	return 0;
}
char* symbol_to_chars(exec_t* ex, int k)
{
	char *s;
	int n;
	if (symbol_to_string(ex, k, &s, &n)) {
		return s;
	}
	return NULL;
}
void free_names(exec_t* ex)
{
	name_t* ns;
	if ((ns = ex->ns) != NULL) {
		for (; ns->s; ns++) {
			free(ns->s);
		}
		free(ex->ns);
		ex->ns = NULL;
	}
}

/************************************************************
 *
 ************************************************************/
void trace_links(exec_t* ex)
{
	link_t* l = ex->gc;
	while (l) {
		fprintf(ex->out, "%p: %d: %s\n", l, l->lock, l->proto->typestr);
		if (l->proto->trace) {
			(*l->proto->trace)(ex, l + 1);
		}
		l = l->prev;
	}
}
void* alloc_link(exec_t* ex, size_t sz, proto_t* proto)
{
	link_t *l;
	l = calloc(1, sizeof(link_t) + sz);
	l->proto = proto;
	l->prev = ex->gc;
	if (ex->gc) {
		ex->gc->next = l;
	}
	ex->gc = l;
	return (void*)(l + 1);
}
static link_t* remove_link(exec_t* ex, link_t* l)
{
	link_t *prev, *next;
	if (l->proto && l->proto->clear) {
		(*l->proto->clear)(ex, l + 1);
		/* l->prev && l->next 
		can be modified
		 when clear object or array.
		 */
	}
	prev = l->prev;
	next = l->next;
	free(l);
	if (prev) {
		prev->next = next;
	}
	if (next) {
		next->prev = prev;
	} else {
		ex->gc = prev;
	}
	return prev;
}
void free_link(exec_t* ex, void *p)
{
	remove_link(ex, (link_t*)p - 1);
}
void free_links(exec_t* ex, void *p)
{
	link_t *l = (link_t*)p - 1;
	while (l->next) {
		if (l->next->lock > 0) {
			l = l->next;
		} else {
			l = remove_link(ex, l->next);
		}
	}
	assert(l == ex->gc);
	free_link(ex, p);
	if (!ex->gc) {
		free_names(ex);
	}
}

/************************************************************
 *
 ************************************************************/
void trace_var(exec_t* ex, var_t* v)
{
	fprintf(ex->out, "%s: %s\n", typeof_var(ex, v), var_toChars(ex, v));
	if (v->p && v->p->trace) {
		(*v->p->trace)(ex, v->u.p);
	}
}
int lock_var(exec_t* ex, var_t* v)
{
	if (v->p && v->p->lock) {
		((link_t*)(v->u.p - sizeof(link_t)))->lock++;
		return 1;
	}
	return 0;
}
int unlock_var(exec_t* ex, var_t* v)
{
	if (v->p && v->p->lock) {
		((link_t*)(v->u.p - sizeof(link_t)))->lock--;
		return 1;
	}
	return 0;
}
int delete_var(exec_t* ex, var_t* v)
{
	if (v->p && v->p->lock) {
		if (((link_t*)(v->u.p - sizeof(link_t)))->lock <= 0) {
			free_link(ex, v->u.p);
			return 1;
		}
	}
	/* else {
		if (v->p && v->p->clear) {
			(*v->p->clear)(ex, v->u.p);
			return 1;
		}
	}
	*/
	return 0;
}

/************************************************************
 * MEMCPY
 ************************************************************/
void copy_var(var_t* v1, var_t* v2)
{
	memcpy(v1, v2, sizeof(var_t));
}

/************************************************************
 * GET TYPESTR
 ************************************************************/
char* typeof_var(exec_t* ex, var_t* v)
{
	return (v->p ? v->p->typestr : "null");
}

/************************************************************
 * CONCAT
 ************************************************************/
int cat_vars(exec_t* ex, var_t* v1, var_t* v2, var_t* v3)
{
	if (v1->p == string_proto || v2->p == string_proto) {
		char *s1, *s2;
		int n1, n2;
		new_string(ex, v3, NULL);
		if (var_toString(ex, v1, &s1, &n1)) {
			string_add(v3->u.p, s1, n1);
		}
		if (var_toString(ex, v2, &s2, &n2)) {
			string_add(v3->u.p, s2, n2);
		}
	} else 
	if (v1->p == float_proto || v2->p == float_proto) {
		double r = var_toFloat(ex, v1) + var_toFloat(ex, v2);
		new_float(v3, r);
	} else {
		int r = var_toInt(ex, v1) + var_toInt(ex, v2);
		new_int(v3, r);
	}
	return 1;
}

/************************************************************
 * COMPARE
 ************************************************************/
int cmp_vars_asNumber(exec_t* ex, var_t* v1, var_t* v2)
{
	double f1, f2;
	f1 = var_toFloat(ex, v1);
	f2 = var_toFloat(ex, v2);
	return ((f1 < f2) ? -1 : (f1 > f2));
}
int cmp_vars_asString(exec_t* ex, var_t* v1, var_t* v2)
{
	char *s1,*s2;
	int n1, n2;
	if (v1->u.p == v2->u.p) return 0;
	if (!var_toString(ex, v1, &s1, &n1)
	|| !var_toString(ex, v2, &s2, &n2)) {
		return (n1 < n2) ? -1 : (n1 > n2);
	}
	return strnncmp(s1, n1, s2, n2);
}
int cmp_vars(exec_t* ex, var_t* v1, var_t* v2)
{
	if (v1->p == string_proto || v2->p == string_proto) {
		return cmp_vars_asString(ex, v1, v2);
	} else if (v1->p == float_proto || v2->p == float_proto) {
		return cmp_vars_asNumber(ex, v1, v2);
	}		
	return (var_toInt(ex, v1) - var_toInt(ex, v2));
}

/************************************************************
 *
 ************************************************************/
int var_toBool(exec_t* ex, var_t* v)
{
	if (v->p && v->p->toBool) {
		return (*v->p->toBool)(ex, v);
	}
	return (v->p != NULL);
}
int var_toChar(exec_t* ex, var_t* v)
{
	if (v->p && v->p->toChar) {
		return (*v->p->toChar)(ex, v);
	}
	return 0;
}
int var_toInt(exec_t* ex, var_t* v)
{
	if (v->p && v->p->toInt) {
		return (*v->p->toInt)(ex, v);
	}
	return 0;
}
double var_toFloat(exec_t* ex, var_t* v)
{
	if (v->p && v->p->toFloat) {
		return (*v->p->toFloat)(ex, v);
	}
	return 0;
}
int var_toString(exec_t* ex, var_t* v, char **s, int *n)
{
	if (v->p && v->p->toString) {
		return (*v->p->toString)(ex, v, s, n);
	}
	*s = NULL;
	*n = 0;
	return 0;
}
int var_toNumber(exec_t* ex, var_t* v, double *f, int *i)
{
	if (v->p) {
		if (v->p == float_proto) {
			*i = *f = v->u.f;
			return 2;
		} else if (v->p->toNumber) {
			return (*v->p->toNumber)(ex, v, f, i);
		} else if (v->p->toInt) {
			*f = *i = (*v->p->toInt)(ex, v);
			return 1;
		}
	}
	*f = *i = 0;
	return 0;
}
char* var_toChars(exec_t* ex,var_t* v)
{
	char *s;
	int n;
	if (var_toString(ex, v, &s, &n)) {
		return s;
	}
	return NULL;
}
int var_toSymbol(exec_t* ex, var_t* v)
{
	char *s;
	int n;
	if (v->p == symbol_proto) {
		return v->u.i;
	} else if (var_toString(ex, v, &s, &n)) {
		return string_to_symbol(ex, s, n);
	}
	return 0;
}

/************************************************************
 *
 ************************************************************/
int var_getLength(exec_t* ex, var_t* v, var_t* res)
{
	if (v->p && v->p->getLength) {
		new_int(res, (*v->p->getLength)(ex, v->u.p));
		return 1;
	}
	new_null(res);
	fprintf(ex->err, "bad getLength: %s\n", typeof_var(ex, v));
	return 0;
}
int var_setLength(exec_t* ex, var_t* v, var_t* val)
{
	if (val->p != int_proto) {
		fprintf(ex->err, "bad value: %s\n", typeof_var(ex, val));
		return 0;
	}
	if (v->p && v->p->setLength) {
		return (*v->p->setLength)(ex, v->u.p, val->u.i);
	}
	fprintf(ex->err, "bad setLength: %s\n", typeof_var(ex, v));
	return 0;
}
int var_getItem(exec_t* ex, var_t* v, var_t* key, var_t* res)
{
	if ((key->p == symbol_proto) && (key->u.i == K_length)) {
		return var_getLength(ex, v, res);
	}
	if (v->p && v->p->getItem) {
		return (*v->p->getItem)(ex, v->u.p, key, res);
	}
	new_null(res);
	return 0;
}
int var_setItem(exec_t* ex, var_t* v, var_t* key, var_t* val)
{
	if ((key->p == symbol_proto) && (key->u.i == K_length)) {
		return var_setLength(ex, v, val);
	}
	if (v->p && v->p->setItem) {
		return (*v->p->setItem)(ex, v->u.p, key, val);
	}
	fprintf(ex->err, "bad setItem: %s\n", typeof_var(ex, v));
	return 0;
}
int var_delItem(exec_t* ex, var_t* v, var_t* key, var_t* res)
{
	if (v->p && v->p->delItem) {
		return (*v->p->delItem)(ex, v->u.p, key, res);
	}
	if (res) new_null(res);
	fprintf(ex->err, "bad delItem: %s\n", typeof_var(ex, v));
	return 0;
}
int var_eachItem(exec_t* ex, var_t* v, int i, var_t* key, var_t* val)
{
	if (v->p && v->p->eachItem) {
		return (*v->p->eachItem)(ex, v->u.p, i, key, val);
	}
	fprintf(ex->err, "bad eachItem: %s\n", typeof_var(ex, v));
	return 0;
}
int var_callFunc(exec_t* ex, var_t* v, int sym, int argc, var_t* argv, var_t* res)
{
	int r;
	if (!v) {
		r = global_callFunc(ex, sym, argc, argv, res);
		if (r >= 0) return r;
		fprintf(ex->err, "no such function: %s(%d)\n", symbol_to_chars(ex, sym), argc);
	} else {
		if (v->p && v->p->callFunc) {
			r = (*v->p->callFunc)(ex, v->u.p, sym, argc, argv, res);
			if (r >= 0) return r;
		}
		if (argc == 0) {
			r = global_callFunc(ex, sym, 1, v, res);
			if (r >= 0) return r;
		}
		fprintf(ex->err, "no such function: %s.%s(%d)\n",
			typeof_var(ex, v), symbol_to_chars(ex, sym), argc);
	}
	ex->jmp = K_catch;
	return 0;
}

/************************************************************
 *
 ************************************************************/
void new_null (var_t* v)
{
	v->p = NULL;
}
void new_bool(var_t* v, int i)
{
	v->p = bool_proto;
	v->u.i = i != 0;
}
void new_char (var_t* v, int i)
{
	v->p = char_proto;
	v->u.i = i & 0xFF;
}
void new_int (var_t* v, int i)
{
	v->p = int_proto;
	v->u.i = i;
}
void new_symbol (var_t* v, int i)
{
	v->p = symbol_proto;
	v->u.i = i;
}
void new_float (var_t* v, double f)
{
	v->p = float_proto;
	v->u.f = f;
}
void new_label (var_t* v, char *p)
{
	v->p = label_proto;
	v->u.p = p;
}
void new_function (var_t* v, char *p)
{
	v->p = function_proto;
	v->u.p = p;
}
void new_class (var_t* v, char *p)
{
	v->p = class_proto;
	v->u.p = p;
}
void new_file(var_t* v, FILE* fp)
{
	v->p = file_proto;
	v->u.p = fp;
}
void new_string_with (exec_t* ex, var_t* v, char *s, int n)
{
	string_t* str;
	v->p = string_proto;
	str = v->u.p = string_alloc(ex);
	if (s && n) {
		str->s = s;
		str->n = n;
	}
}
void new_string_copy (exec_t* ex, var_t* v, char *s, int n)
{
	string_t* str;
	v->p = string_proto;
	str = v->u.p = string_alloc(ex);
	if (s && n) {
		str->s = strndupok(s, n);
		str->n = n;
	}
}
void new_string (exec_t* ex, var_t* v, void *p)
{
	if (!p) p = string_alloc(ex);
	v->p = string_proto;
	v->u.p = p;
}
void new_regex (exec_t* ex, var_t* v, char *s, int n, int i)
{
	v->p = regex_proto;
	v->u.p = regex_alloc(ex);
	if (!regex_comp(ex, v->u.p, s, n, i)) {
		fprintf(ex->err, "regex compile error: %.*s\n", n, s);
		regex_free(ex, v->u.p);
		/* make null */;
		v->p = NULL;
	}
}
void new_array (exec_t* ex, var_t* v, void *p)
{
	if (!p) p = array_alloc(ex);
	v->p = array_proto;
	v->u.p = p;
}
void new_refer (exec_t* ex, var_t* v, void *p)
{
	if (!p) p = refer_alloc(ex);
	v->p = refer_proto;
	v->u.p = p;
}
void new_object (exec_t* ex, var_t* v, void* p)
{
	if (!p) p = object_alloc(ex);
	v->p = object_proto;
	v->u.p = p;
}
void new_frame (exec_t* ex, var_t* v, void *p)
{
	if (!p) {
		p = frame_alloc(ex);
	}
	v->p = frame_proto;
	v->u.p = p;
}

/************************************************************
 *
 ************************************************************/
static int int_toBool(exec_t* ex, var_t* v)
{
	return (v->u.i != 0);
}
static int int_toChar(exec_t* ex, var_t* v)
{
	return (v->u.i & 0xFF);
}
static int int_toInt(exec_t* ex, var_t* v)
{
	return v->u.i;
}
static double int_toFloat(exec_t* ex, var_t* v)
{
	return (double)v->u.i;
}

/************************************************************
 *
 ************************************************************/
static int float_toBool(exec_t* ex, var_t* v)
{
	return (v->u.f != 0);
}
static int float_toChar(exec_t* ex, var_t* v)
{
	return ((int)v->u.f & 0xFF);
}
static int float_toInt(exec_t* ex, var_t* v)
{
	return (int)v->u.f;
}
static double float_toFloat(exec_t* ex, var_t* v)
{
	return v->u.f;
}

/************************************************************
 *
 ************************************************************/
static int bool_toString(exec_t* ex, var_t* v, char **s, int *n)
{
	if (v->u.i) {
		*s = "true";
		*n = 4;
	} else {
		*s = "false";
		*n = 5;
	}
	return *n;
}
static int char_toString(exec_t* ex, var_t* v, char **s, int *n)
{
	return (*n = sprintf(*s = tmpbuf(), "%c", (v->u.i & 0xFF)));
}
static int int_toString(exec_t* ex, var_t* v, char **s, int *n)
{
	return (*n = sprintf(*s = tmpbuf(), "%d", v->u.i));
}
static int float_toString(exec_t* ex, var_t* v, char **s, int *n)
{
	return (*n = sprintf(*s = tmpbuf(), "%g", v->u.f));
}
static int symbol_toString(exec_t* ex, var_t* v, char **s, int *n)
{
	return symbol_to_string(ex, v->u.i, s, n);
}

/************************************************************
 *
 ************************************************************/
proto_t _bool_proto_ = {
"bool", /* typestr */
0, /* lock */
NULL, /* trace */
NULL, /* clear */
int_toBool, /* toBool */
int_toChar, /* toChar */
int_toInt, /* toInt */
int_toFloat, /* toFloat */
NULL, /* toNumber */
bool_toString, /* toString */
NULL, /* getLength */
NULL, /* setLegnth */
NULL, /* callFunc */
NULL, /* getItem */
NULL, /* setItem */
NULL, /* delItem */
NULL, /* eachItem */
0
};

proto_t _char_proto_ = {
"char", /* typestr */
0, /* lock */
NULL, /* trace */
NULL, /* clear */
int_toBool, /* toBool */
int_toChar, /* toChar */
int_toInt, /* toInt */
int_toFloat, /* toFloat */
NULL, /* toNumber */
char_toString, /* toString */
NULL, /* getLength */
NULL, /* setLegnth */
NULL, /* callFunc */
NULL, /* getItem */
NULL, /* setItem */
NULL, /* delItem */
NULL, /* eachItem */
0
};

proto_t _int_proto_ = {
"int", /* typestr */
0, /* lock */
NULL, /* trace */
NULL, /* clear */
int_toBool, /* toBool */
int_toChar, /* toChar */
int_toInt, /* toInt */
int_toFloat, /* toFloat */
NULL, /* toNumber */
int_toString, /* toString */
NULL, /* getLength */
NULL, /* setLegnth */
NULL, /* callFunc */
NULL, /* getItem */
NULL, /* setItem */
NULL, /* delItem */
NULL, /* eachItem */
0
};

proto_t _symbol_proto_ = {
"symbol", /* typestr */
0, /* lock */
NULL, /* trace */
NULL, /* clear */
NULL, /* toBool */
NULL, /* toChar */
NULL, /* toInt */
NULL, /* toFloat */
NULL, /* toNumber */
symbol_toString, /* toString */
NULL, /* getLength */
NULL, /* setLegnth */
NULL, /* callFunc */
NULL, /* getItem */
NULL, /* setItem */
NULL, /* delItem */
NULL, /* eachItem */
0
};

proto_t _float_proto_ = {
"float", /* typestr */
0, /* lock */
NULL, /* trace */
NULL, /* clear */
float_toBool, /* toBool */
float_toChar, /* toChar */
float_toInt, /* toInt */
float_toFloat, /* toFloat */
NULL, /* toNumber */
float_toString, /* toString */
NULL, /* getLength */
NULL, /* setLegnth */
NULL, /* callFunc */
NULL, /* getItem */
NULL, /* setItem */
NULL, /* delItem */
NULL, /* eachItem */
0
};

proto_t _string_proto_ = {
"string", /* typestr */
1, /* lock */
string_trace, /* trace */
string_clear, /* clear */
string_toBool, /* toBool */
string_toChar, /* toChar */
string_toInt, /* toInt */
string_toFloat, /* toFloat */
string_toNumber, /* toNumber */
string_toString, /* toString */
string_getLength, /* getLength */
string_setLength, /* setLegnth */
string_callFunc, /* callFunc */
string_getItem, /* getItem */
string_setItem, /* setItem */
string_delItem, /* delItem */
string_eachItem, /* eachItem */
0
};

proto_t _regex_proto_ = {
"regex", /* typestr */
1, /* lock */
regex_trace, /* trace */
regex_clear, /* clear */
NULL, /* toBool */
NULL, /* toChar */
NULL, /* toInt */
NULL, /* toFloat */
NULL, /* toNumber */
NULL, /* toString */
NULL, /* getLength */
NULL, /* setLegnth */
NULL, /* callFunc */
NULL, /* getItem */
NULL, /* setItem */
NULL, /* delItem */
NULL, /* eachItem */
0
};

proto_t _array_proto_ = {
"array", /* typestr */
1, /* lock */
array_trace, /* trace */
array_clear, /* clear */
array_toBool, /* toBool */
array_toChar, /* toChar */
array_toInt, /* toInt */
array_toFloat, /* toFloat */
NULL, /* toNumber */
array_toString, /* toString */
array_getLength, /* getLength */
array_setLength, /* setLegnth */
array_callFunc, /* callFunc */
array_getItem, /* getItem */
array_setItem, /* setItem */
array_delItem, /* delItem */
array_eachItem, /* eachItem */
0
};

proto_t _refer_proto_ = {
"refer", /* typestr */
1, /* lock */
refer_trace, /* trace */
refer_clear, /* clear */
NULL, /* toBool */
NULL, /* toChar */
NULL, /* toInt */
NULL, /* toFloat */
NULL, /* toNumber */
NULL, /* toString */
NULL, /* getLength */
NULL, /* setLegnth */
NULL, /* callFunc */
NULL, /* getItem */
NULL, /* setItem */
NULL, /* delItem */
NULL, /* eachItem */
0
};

proto_t _object_proto_ = {
"object", /* typestr */
1, /* lock */
object_trace, /* trace */
object_clear, /* clear */
object_toBool, /* toBool */
object_toChar, /* toChar */
object_toInt, /* toInt */
object_toFloat, /* toFloat */
NULL, /* toNumber */
object_toString, /* toString */
object_getLength, /* getLength */
object_setLength, /* setLegnth */
object_callFunc, /* callFunc */
object_getItem, /* getItem */
object_setItem, /* setItem */
object_delItem, /* delItem */
object_eachItem, /* eachItem */
0
};

proto_t _frame_proto_ = {
"frame", /* typestr */
0, /* lock */
frame_trace, /* trace */
frame_clear, /* clear */
NULL, /* toBool */
NULL, /* toChar */
NULL, /* toInt */
NULL, /* toFloat */
NULL, /* toNumber */
NULL, /* toString */
NULL, /* getLength */
NULL, /* setLegnth */
NULL, /* callFunc */
frame_getItem, /* getItem */
frame_setItem, /* setItem */
frame_delItem, /* delItem */
NULL, /* eachItem */
0
};

proto_t _file_proto_ = {
"file", /* typestr */
0, /* lock */
NULL, /* trace */
NULL, /* clear */
NULL, /* toBool */
NULL, /* toChar */
NULL, /* toInt */
NULL, /* toFloat */
NULL, /* toNumber */
NULL, /* toString */
NULL, /* getLength */
NULL, /* setLegnth */
file_callFunc, /* callFunc */
NULL, /* getItem */
NULL, /* setItem */
NULL, /* delItem */
NULL, /* eachItem */
0
};

proto_t _dbm_proto_ = {
"dbm", /* typestr */
0, /* lock */
NULL, /* trace */
NULL, /* clear */
NULL, /* toBool */
NULL, /* toChar */
NULL, /* toInt */
NULL, /* toFloat */
NULL, /* toNumber */
NULL, /* toString */
NULL, /* getLength */
NULL, /* setLegnth */
dbm_callFunc, /* callFunc */
dbm_getItem, /* getItem */
dbm_setItem, /* setItem */
dbm_delItem, /* delItem */
dbm_eachItem, /* eachItem */
0
};

proto_t _xml_proto_ = {
"xml", /* typestr */
1, /* lock */
xml_trace, /* trace */
xml_clear, /* clear */
NULL, /* toBool */
NULL, /* toChar */
NULL, /* toInt */
NULL, /* toFloat */
NULL, /* toNumber */
xml_toString, /* toString */
xml_getLength, /* getLength */
NULL, /* setLegnth */
xml_callFunc, /* callFunc */
NULL, /* getItem */
NULL, /* setItem */
NULL, /* delItem */
NULL, /* eachItem */
0
};

proto_t _function_proto_ = {
"function", /* typestr */
0, /* lock */
NULL, /* trace */
NULL, /* clear */
NULL, /* toBool */
NULL, /* toChar */
NULL, /* toInt */
NULL, /* toFloat */
NULL, /* toNumber */
NULL, /* toString */
NULL, /* getLength */
NULL, /* setLegnth */
NULL, /* callFunc */
NULL, /* getItem */
NULL, /* setItem */
NULL, /* delItem */
NULL, /* eachItem */
0
};

proto_t _class_proto_ = {
"class", /* typestr */
0, /* lock */
NULL, /* trace */
NULL, /* clear */
NULL, /* toBool */
NULL, /* toChar */
NULL, /* toInt */
NULL, /* toFloat */
NULL, /* toNumber */
NULL, /* toString */
NULL, /* getLength */
NULL, /* setLegnth */
NULL, /* callFunc */
NULL, /* getItem */
NULL, /* setItem */
NULL, /* delItem */
NULL, /* eachItem */
0
};

proto_t _label_proto_ = {
"label", /* typestr */
0, /* lock */
NULL, /* trace */
NULL, /* clear */
NULL, /* toBool */
NULL, /* toChar */
NULL, /* toInt */
NULL, /* toFloat */
NULL, /* toNumber */
NULL, /* toString */
NULL, /* getLength */
NULL, /* setLegnth */
NULL, /* callFunc */
NULL, /* getItem */
NULL, /* setItem */
NULL, /* delItem */
NULL, /* eachItem */
0
};

proto_t* bool_proto      = &_bool_proto_;
proto_t* char_proto      = &_char_proto_;
proto_t* int_proto       = &_int_proto_;
proto_t* symbol_proto    = &_symbol_proto_;
proto_t* float_proto     = &_float_proto_;
proto_t* file_proto      = &_file_proto_;
proto_t* dbm_proto       = &_dbm_proto_;
proto_t* xml_proto       = &_xml_proto_;
proto_t* function_proto  = &_function_proto_;
proto_t* class_proto     = &_class_proto_;
proto_t* label_proto     = &_label_proto_;
proto_t* frame_proto     = &_frame_proto_;
proto_t* string_proto    = &_string_proto_;
proto_t* regex_proto     = &_regex_proto_;
proto_t* array_proto     = &_array_proto_;
proto_t* refer_proto     = &_refer_proto_;
proto_t* object_proto    = &_object_proto_;
