/*
 * Copyright (c) 2007, to-do. All rights reserved.
 */
#include "util.h"
#include "var.h"
#include "_sym.h"
#include "_ext.h"
#include "_op.h"

/************************************************************
 *
 ************************************************************/
 /* LEX FLAG */
#define LF_MACRO      0x01
#define LF_ECHO       0x02
#define LF_RIGHT      0x10
#define LF_LEFT       0x20

/* VALUE FLAG */
#define VF_OBJECT     0x0002
#define VF_NEW        0x0004
#define VF_CONST      0x0008
#define VF_RESOLV     0x00FF
#define VF_SYMBOL     0x0100
#define VF_REFER      0x0200
#define VF_UNRESOLV   0xFF00

/* ALLOW FLAG */
#define AF_BREAK      0x01
#define AF_CONTINUE   0x02
#define AF_RETURN     0x04
#define AF_THROW      0x08
#define AF_THIS       0x10
#define AF_BLOCK      0x0100
#define AF_FUNCTION   0x0400
#define AF_CLASS      0x0800
#define AF_FILE       0x1000
#define AF_NAMES      0x2000
#define AF_GLOBAL     0x4000

/************************************************************
 *
 ************************************************************/
typedef struct {
	int k;
	char *s;
	int n;
} cp_def_t;

typedef struct {
	int k;
	char *s;
	int n;
} cp_name_t;

typedef struct {
	int k;
	int c;
	int j;
} cp_var_t;

typedef struct cp_frame_t {
	struct cp_frame_t* p;
	int vi; /* locals index */
	int af; /* allow flag */
	int bi; /* bytecode index */
} cp_frame_t;

typedef struct {
	cp_def_t* dp; /* defines */
	int dn;
	cp_name_t* np; /* name space */
	int nn;
	cp_var_t* vp; /* locals pointer */
	int vn;
	cp_frame_t* fp; /* frame info */
	char *bc;
	int bn;
	/* info */
	char *fn; /* file-name */
	int ln; /* line-number */
	char *ss; /* start-of-source */
	char *es; /* end-of-source */
	char *sc; /* source-code */
	op_t lc; /* last charactor */
	int lf; /* lex flag */
	int vf; /* value flag */
	int nc; /* cols */
	int nr; /* rows */
	FILE* out;
	FILE* err;
	FILE* in;
} compile_t;

/************************************************************
 *
 ************************************************************/
static int seek_int(compile_t* cp, int v)
{
	int j = cp->bn;
	*(int*)(cp->bc + cp->bn) = v;
	cp->bn += sizeof(int);
	return j;
}
static int set_offset(compile_t* cp, int j)
{
	int v = *(int*)(cp->bc + j);
	*(int*)(cp->bc + j) = cp->bn - j;
	return v;
}
static void put_offset(compile_t* cp, int j)
{
	*(int*)(cp->bc + cp->bn) = j - cp->bn;
	cp->bn += sizeof(int);
}
static void put_c(compile_t* cp, op_t c)
{
	cp->bc[cp->bn++] = c;
}
static void put_ushort(compile_t* cp, int v)
{
	*(unsigned short*)(cp->bc + cp->bn) = v;
	cp->bn += sizeof(unsigned short);
}
static void put_int(compile_t* cp, int v)
{
	*(int*)(cp->bc + cp->bn) = v;
	cp->bn += sizeof(int);
}
static void put_c_int(compile_t* cp, op_t c, int v)
{
	cp->bc[cp->bn++] = c;
	*(int*)(cp->bc + cp->bn) = v;
	cp->bn += sizeof(int);
}
static void put_float(compile_t* cp, double f)
{
	*(double*)(cp->bc + cp->bn) = f;
	cp->bn += sizeof(double);
}
static void put_bytes(compile_t* cp, char *s, int n)
{
	memcpy(cp->bc + cp->bn, s, n);
	cp->bn += n;
}
static void put_chars(compile_t* cp, char *s, int n)
{
	/* strn0cpy */
	memcpy(cp->bc + cp->bn, s, n);
	cp->bn += n;
	cp->bc[cp->bn++] = 0;
}
static void put_regex(compile_t* cp, char *s, int n)
{
	int q, c;
	char *d, *e;
	d = cp->bc + cp->bn + 5;
	e = s + n;
	q = *s++;
	n = 0;
	while ((s < e) && (*s != q)) {
		if ((c = *s++) == '\\' && (*s > 0)) {
			if (*s == q) c = *s++;
			else if (*s == '\\') d[n++] = *s++;
		}
		d[n++] = c;
	}
	d[n] = 0;
	assert(*s++ == q);
	c = 0;
	while (islower(*s)) {
		c |= (1 << (*s++ - 'a'));
	}
	assert(s == e);
	put_c_int(cp, C_regex, c);
	cp->bn += n + 1;
}
static void put_string(compile_t* cp, char *s, int n)
{
	char *d, *e;
	int c, q;
	d = cp->bc + cp->bn + 5;
	e = s + n;
	q = *s++;
	n = 0;
	while ((s < e) && (*s != q)) {
		if ((c = *s++) == '\\') {
			c = meta_to_c(&s, q);
		}
		d[n++] = c;
	}
	d[n] = 0;
	assert((*s++ == q) && (s == e));
	if (q == '\'') {
		if (n == 1) {
			put_c_int(cp, C_char, *d);
			return;
		}
	}
	put_c_int(cp, C_string, n);
	cp->bn += n;
}
static void put_doc(compile_t* cp, char *s, int n, int str)
{
	if (str) { /* as string */
		if (n > 0) {
			put_c_int(cp, C_string, n);
			put_bytes(cp, s, n);
		} else {
			put_c(cp, C_null);
		}
	} else {
		if (n > 0) {
			put_c_int(cp, C_doc, n);
			put_bytes(cp, s, n);
		}
	}
}
static void put_lineno(compile_t* cp)
{
	put_c(cp, C_line);
	put_ushort(cp, cp->ln);
}

/************************************************************
 *
 ************************************************************/
static int is_macro(char *s, int *n)
{
#include "_is_M.h"
	return 0;
}
static int is_keyword(char *s, int *n)
{
#include "_is_K.h"
	return 0;
}
static int is_operand(char *s, int *n)
{
#include "_is_R.h"
	return 0;
}
static int seek_doc(compile_t* cp)
{
	char *s;
	int c;
	s = cp->sc;
	while ((c = *s++) > 0) {
		if ((c == '<') && (*s=='?') && !islower(*(s + 1))) {
			break;
		}
		if (iscrlf(c)) {
			if (*s == (c ^ 7)) s++;
			cp->nr++;
		}
	}
	cp->nc = --s - cp->sc;
	return C_doc;
}
static int seek_regex(compile_t* cp)
{
	char *s;
	int c, q;
	q = *cp->sc;
	s = cp->sc + 1;
	while (((c = *s++) > 0) && (c != q)) {
		if (c == '\\') c = *s++;
		if (iscrlf(c)) {
			if (*s == (c ^ 7)) s++;
			cp->nr++;
		}
	}
	exerrif(cp, c != q, "bad regex");
	cp->nc = s - cp->sc;
	while (islower(*s)) s++;
	cp->nc = s - cp->sc;
	return C_regex;
}
static int seek_string(compile_t* cp)
{
	char *s;
	int c,q;
	q = *cp->sc;
	s = cp->sc + 1;
	while (((c = *s++) > 0) && (c != q)) {
		if (c == '\\') c = *s++;
		if (iscrlf(c)) {
			if (*s == (c ^ 7)) s++;
			cp->nr++;
		}
	}
	exerrif(cp, c != q, "bad string");
	cp->nc = s - cp->sc;
	return C_string;
}
static int seek_number(compile_t* cp)
{
	char *s;
	int c, q;
	c = *cp->sc;
	s = cp->sc + 1;
	q = 0;
	if (c=='0' && *s=='x') {
		s++;
		while (isxdigit(*s)) s++;
	} else if (c=='0' && isoctal(*s)) {
		s++;
		while (isoctal(*s)) s++;
	} else {
		while (isdigit(*s)) s++;
		if (*s == '.') {
			q = *s++;
			while (isdigit(*s)) s++;
		}
		if (*s=='e' || *s=='E') {
			q = *s++;
			if (*s=='+' || *s=='-') s++;
			while (isdigit(*s)) s++;
		}
	}
	cp->nc = s - cp->sc;
	return (q ? C_float : C_int);
}
static int seek_symbol_only(compile_t* cp)
{
	char *s;
	s = cp->sc + 1;
	while (isalnum(*s) || (*s=='_')) s++;
	cp->nc = s - cp->sc;
	return C_symbol;
}
static int seek_symbol_or_keyword(compile_t* cp)
{
	char *s;
	int c;
	c = !islower(*cp->sc);
	s = cp->sc + 1;
	while (isalnum(*s) || (*s=='_')) {
		c += !islower(*s++);
	}
	cp->nc = s - cp->sc;
	if (!c) {
		if ((c = is_keyword(cp->sc, &cp->nc)) != 0) {
			return c;
		}
	}
	return C_symbol;
}
static int lex(compile_t* cp, int w)
{
	char *s;
	int c, q;
	if ((c = cp->lc) > 0) {  /* > EOF, ERR */
		cp->lc = 0;
		return c;
	}
	cp->sc += cp->nc;
	cp->ln += cp->nr;
	cp->nc = 0;
	cp->nr = 0;
	q = 0;
	s = cp->sc;
	while ((c = *s++) > 0) {
		if (s > cp->es) {
			c = 0;
			break;
		}
		else if (iscrlf(c)) {
			if (*s == (c ^ 7)) s++;
			cp->ln++;
			if (q != '*') q = 0;
			if (!q && (cp->lf & LF_MACRO)) {
				break;
			}
		}
		else if (isspace(c)) {
			;
		}
		else if (q) {
			if ((q == '*') && (c=='*') && (*s=='/')) {
				s++;
				q = 0;
			}
		}
		else if ((c == '/') && (*s=='/' || *s=='*')) {
			q = *s++;
		}
		else if (c == '#') {
			if (islower(*s) && (w & LF_LEFT)) {
				break;
			}
			q = c;
		}
		else {
			break;
		}
	}
	cp->sc = s - 1;
	if (c <= 0) {
		return -1; /* EOF */
	}
	else if (iscrlf(c)) {
		cp->sc = s;
		cp->lf &= (0xF ^ LF_MACRO);
		return C_crlf;
	}
	else if (c == '#') {
		while (islower(*s)) s++;
		cp->nc = s - cp->sc;
		cp->lf |= LF_MACRO;
		c = is_macro(cp->sc, &cp->nc);
		exerrif(cp, !c, "unsupported macro");
		return c;
	}
	else if ((c=='?') && (*s=='>')) {
		s++;
		if (!(cp->lf & LF_ECHO)) {
			while (iscrlf(c = *s)) {
				if (*(++s) == (c ^ 7)) s++;
				cp->nr++;
			}
		}
		cp->sc = s;
		return seek_doc(cp);
	}
	else if ((c == '<') && (*s == '?')) {
		if (*(++s) == '=') {
			s++;
			cp->lf |= LF_ECHO;
		} else {
			cp->lf &= (0xF ^ LF_ECHO);
		}
		cp->sc = s;
		return ';';
	}
	else if (w & LF_RIGHT) {
		if (isalpha(c) || (c == '_')) {
			return seek_symbol_only(cp);
		}
	}
	else {
		if ((c == '`') || (c == '/')) {
			return seek_regex(cp);
		}
		else if ((c == '"') || (c == '\'')) {
			return seek_string(cp);
		}
		else if (isdigit(c) || (c == '.')) {
			return seek_number(cp);
		}
		else if (isalpha(c) || (c == '_')) {
			return seek_symbol_or_keyword(cp);
		}
	}
	return is_operand(cp->sc, &cp->nc);
}

static void unlex(compile_t* cp, op_t c)
{
	assert ((c > 0) && (cp->lc == 0)); /* > EOF, ERR */
	cp->lc = c;
}

/************************************************************
 *
 ************************************************************/
static int add_symbol(compile_t* cp, char *s, int n)
{
	int k, i;
	k = strtoh(s, n);
	exerrif(cp, !k, "bad symbol") ;
	switch (k) {
#include "_sym2sym.h"	
	}
	for (i = 0; i < cp->nn; i++) {
		if (cp->np[i].k == k) {
			if ((cp->np[i].n != n)
			|| strncmp(cp->np[i].s, s, n)) {
				fprintf(cp->err, "%s != %s\n", cp->np[i].s, s);
			}
			return k;
		}
	}
	i = cp->nn++;
	cp->np = realloc(cp->np, cp->nn * sizeof(cp_name_t));
	cp->np[i].k = k;
	cp->np[i].s = calloc(n + 1, 1);
	memcpy(cp->np[i].s, s, cp->np[i].n = n);
	return k;
}
static int cp_symbol_to_string(compile_t* cp, int k, char **s, int *n)
{
	int i;
	switch (k) {
#include "_sym2str.h"	
	}
	for (i = 0; i < cp->nn; i++) {
		if (cp->np[i].k == k) {
			*s = cp->np[i].s;
			*n = cp->np[i].n;
			return *n;
		}
	}
	*s = NULL;
	*n = 0;
	return 0;
}
static char* cp_symbol_to_chars(compile_t* cp, int k)
{
	char *s;
	int n;
	if (cp_symbol_to_string(cp, k, &s, &n)) {
		return s;
	}
	return NULL;
}

/************************************************************
 *
 ************************************************************/
static int get_define(compile_t* cp, int k, char **s, int *n)
{
	int i;
	for (i = 0; i < cp->dn; i++) {
		if (cp->dp[i].k == k) {
			if (s) *s = cp->dp[i].s;
			if (n) *n = cp->dp[i].n;
			return 1;
		}
	}
	if (s) *s = NULL;
	if (n) *n = 0;
	return 0;
}
static void add_define(compile_t* cp, int k, char* s, int n)
{
	int i;
	for (i = 0; i < cp->dn; i++) {
		if (cp->dp[i].k == k) {
			break;
		}
	}
	if (i >= cp->dn) {
		i = cp->dn++;
		cp->dp = realloc(cp->dp, cp->dn * sizeof(cp_def_t));
	} else {
		free(cp->dp[i].s);
	}
	cp->dp[i].k = k;
	cp->dp[i].s = calloc(n + 1, 1);
	memcpy(cp->dp[i].s, s, cp->dp[i].n = n);
}
static void add_local(compile_t* cp, op_t c, int k, int j)
{
	int i;
	for (i = cp->fp->vi; i < cp->vn; i++) {
		if (cp->vp[i].k == k) {
			fprintf(cp->err, "multiple definition: %s\n", cp_symbol_to_chars(cp, k));
			return;
		}
	}
	i = cp->vn++;
	cp->vp = realloc(cp->vp, cp->vn * sizeof(cp_var_t));
	cp->vp[i].c = c;
	cp->vp[i].k = k;
	cp->vp[i].j = j;
}

/************************************************************
 *
 ************************************************************/
static int is_assign(op_t c)
{
	switch (c) {
	case C_addA: return C_add; 
	case C_subA: return C_sub; 
	case C_mulA: return C_mul;
	case C_divA: return C_div; 
	case C_modA: return C_mod;
	case C_powA: return C_pow;
	case C_bitandA: return C_bitand;
	case C_bitorA: return C_bitor;
	case C_bitsubA: return C_bitsub;
	case C_bitxorA: return C_bitxor;
	case C_bitshLA: return C_bitshL;
	case C_bitshRA: return C_bitshR;
	default: return 0;
	}
	return 0;
}
static int is_right(op_t c)
{
	switch (c) {
	case C_match:
	case C_notmatch:
		return 7;
	case C_mul:
	case C_div:
	case C_mod:
	case C_pow:
		return 6;
	case C_add:
	case C_sub:
		return 5;
	case C_bitshR:
	case C_bitshL:
		return 4;
	case C_lt:
	case C_gt:
		return 3;
	case C_eq:
	case C_ne:
	case C_cmp:
		return 2;
	case C_bitand:
	case C_bitor:
	case C_bitsub:
	case C_bitxor:
		return 1;
	default:
		return 0;
	}
	return 0;
}
static int is_prefix(op_t c)
{
	switch (c) {
	case C_add: return C_plus;
	case C_sub: return C_minus;
	case C_bitsub:
		return C_bitnot;
	case C_not:
	case C_typeof:
		return c;
	default:
		return 0;
	}
	return 0;
}
static int is_typecast(op_t c)
{
	return (c == C_toBool
	|| c == C_toChar
	|| c == C_toInt
	|| c == C_toFloat
	|| c == C_toNumber
	|| c == C_toString);
}

/************************************************************
 *
 ************************************************************/
static double read_as_float(char *p)
{
	switch (*p++) {
	case C_null:  return 0;
	case C_false: return 0;
	case C_true:  return 1;
	case C_char:  return *(int*)p;
	case C_int:   return *(int*)p;
	case C_float:  return *(double*)p;
	case C_string: return strtod(p + 4, NULL);
	}
	return 0;
}
static int read_as_int(char *p)
{
	switch (*p++) {
	case C_null:  return 0;
	case C_false: return 0;
	case C_true:  return 1;
	case C_char:  return *(int*)p;
	case C_int:   return *(int*)p;
	case C_float:  return (int)*(double*)p;
	case C_string: return strtol(p + 4, NULL, 0);
	}
	return 0;
}
static int read_as_string(char *p, char **s)
{
	switch (*p++) {
	case C_null:  *s = "null" ; return 4;
	case C_false: *s = "false"; return 5;
	case C_true:  *s = "true" ; return 4;
	case C_char:  return sprintf(*s, "%c", *(int*)p);
	case C_int:   return sprintf(*s, "%d", *(int*)p);
	case C_float: return sprintf(*s, "%g", *(double*)p);
	case C_string: *s = p + 4; return *(int*)p;
	}
	*s = "";
	return 0;
}
static void calc_const(compile_t* cp, int i1, op_t c, int i2)
{
	static char pk1[32], pk2[32];
	char *p1, *p2, *s1=pk1, *s2=pk2;
	double f1, f2;
	int n1, n2;
	p1 = cp->bc + i1;
	p2 = cp->bc + i2;
	cp->bn = i1;
	if (c==C_lt || c==C_gt
	 || c==C_eq || c==C_ne || c==C_cmp) {
		if (*p1 == C_string || *p2 == C_string) {
			n1 = read_as_string(p1, &s1);
			n2 = read_as_string(p2, &s2);
			i1 = strnncmp(s1, n1, s2, n2);
		} else if (*p1 == C_float || *p2 == C_float) {
			f1 = read_as_float(p1);
			f2 = read_as_float(p2);
			i1 = (f1 < f2) ? -1 : (f1 > f2);
		} else {
			i1 = read_as_int(p1);
			i2 = read_as_int(p2);
			i1 = i1 - i2;
		}
		switch (c) {
		case C_cmp: put_c_int(cp, C_int, i1); break;
		case C_lt: put_c(cp, (i1 <  0) ? C_true : C_false); break;
		case C_gt: put_c(cp, (i1 >  0) ? C_true : C_false); break;
		case C_eq: put_c(cp, (i1 == 0) ? C_true : C_false); break;
		case C_ne: put_c(cp, (i1 != 0) ? C_true : C_false); break;
		default: break;
		}
	} else 
	if ((c == C_add) && (*p1 == C_string || *p2 == C_string)) {
		n1 = read_as_string(p1, &s1);
		n2 = read_as_string(p2, &s2);
		put_c(cp, C_string);
		put_int(cp, n1 + n2);
		put_bytes(cp, s1, n1);
		put_bytes(cp, s2, n2);
	} else 
	if ((c==C_mul || c==C_div
	  || c==C_add || c==C_sub
	  || c==C_pow) && (*p1 == C_float || *p2 == C_float)) {
		f1 = read_as_float(p1);
		f2 = read_as_float(p2);
		switch (c) {
		case C_mul: f1 *= f2; break;
		case C_div: f1 /= f2; break;
		case C_add: f1 += f2; break;
		case C_sub: f1 -= f2; break;
		case C_pow: f1 = pow(f1, f2); break;
		default: break;
		}
		put_c(cp, C_float);
		put_float(cp, f1);
	} else {
		i1 = read_as_int(p1);
		i2 = read_as_int(p2);
		switch (c) {
		case C_mul: i1 *= i2; break;
		case C_div: i1 /= i2; break;
		case C_mod: i1 %= i2; break;
		case C_add: i1 += i2; break;
		case C_sub: i1 -= i2; break;
		case C_pow: i1 = pow(i1, i2); break;
		case C_bitshR: i1 >>= i2; break;
		case C_bitshL: i1 <<= i2; break;
		case C_bitand: i1 &= i2; break;
		case C_bitor:  i1 |= i2; break;
		case C_bitsub: i1 &= ((unsigned int)-1) ^ i2; break;
		case C_bitxor: i1 ^= i2; break;
		default: break;
		}
		put_c(cp, C_int);
		put_int(cp, i1);
	}
}

/************************************************************
 *
 ************************************************************/
static int expr_post(compile_t* cp);
static int expr_pre(compile_t* cp, int prev);
static int expr_brace(compile_t* cp, int prev);
static int stmt_function(compile_t* cp, int prev);

/************************************************************
 *
 ************************************************************/
#ifndef INFINITY
#define INFINITY HUGE_VAL
#endif

#ifndef NAN
#define NAN (0.0/0.0)
#endif

static int expr_const(compile_t* cp, int k)
{
	int n;
	switch (k) {
	case K_stdin:
		put_c(cp, C_stdin);
		return 1;
	case K_stdout:
		put_c(cp, C_stdout);
		return 1;
	case K_stderr:
		put_c(cp, C_stderr);
		return 1;
	case K___FILE__:
		n = strlen(cp->fn);
		put_c_int(cp, C_string, n);
		put_bytes(cp, cp->fn, n);
		return 1;
	case K___LINE__:
		put_c(cp, C_int);
		put_int(cp, cp->ln);
		return 1;
	case K_INFINITY:
		put_c(cp, C_float);
		put_float(cp, INFINITY);
		return 1;
	case K_NAN:
		put_c(cp, C_float);
		put_float(cp, NAN);
		return 1;
	case K_M_E:
		put_c(cp, C_float);
		put_float(cp, M_E);
		return 1;
	case K_M_LOG2E:
		put_c(cp, C_float);
		put_float(cp, M_LOG2E);
		return 1;
	case K_M_LOG10E:
		put_c(cp, C_float);
		put_float(cp, M_LOG10E);
		return 1;
	case K_M_LN2:
		put_c(cp, C_float);
		put_float(cp, M_LN2);
		return 1;
	case K_M_LN10:
		put_c(cp, C_float);
		put_float(cp, M_LN10);
		return 1;
	case K_M_PI:
		put_c(cp, C_float);
		put_float(cp, M_PI);
		return 1;
	case K_M_PI_2:
		put_c(cp, C_float);
		put_float(cp, M_PI_2);
		return 1;
	case K_M_PI_4:
		put_c(cp, C_float);
		put_float(cp, M_PI_4);
		return 1;
	case K_M_1_PI:
		put_c(cp, C_float);
		put_float(cp, M_1_PI);
		return 1;
	case K_M_2_PI:
		put_c(cp, C_float);
		put_float(cp, M_2_PI);
		return 1;
	case K_M_2_SQRTPI:
		put_c(cp, C_float);
		put_float(cp, M_2_SQRTPI);
		return 1;
	case K_M_SQRT2:
		put_c(cp, C_float);
		put_float(cp, M_SQRT2);
		return 1;
	case K_M_SQRT1_2:
		put_c(cp, C_float);
		put_float(cp, M_SQRT1_2);
		return 1;
	}
	return 0;
}

static int expr_symbol(compile_t* cp)
{
	char *s;
	op_t c;
	int n, k;
	k = strtoh(cp->sc, cp->nc);
	if (get_define(cp, k, &s, &n)) {
		put_bytes(cp, s, n);
		cp->vf = VF_CONST;
		c = lex(cp, LF_RIGHT);
	} else if (expr_const(cp, k)) {
		cp->vf = VF_CONST;
		c = lex(cp, LF_RIGHT);
	} else {
		k = add_symbol(cp, cp->sc, cp->nc);
		put_c_int(cp, C_symbol, k);
		cp->vf = VF_SYMBOL;
		c = expr_post(cp);
	}
	return c;
}

static int expr_post(compile_t* cp)
{
	op_t c;
	int n,vf;
	vf = cp->vf;
	c = lex(cp, LF_RIGHT); /* get right */
	if (c == '.') {
		c = lex(cp, LF_RIGHT);
		exerrif(cp, c != C_symbol, "bad property");
		put_c_int(cp, C_symbol, add_symbol(cp, cp->sc, cp->nc));
		put_c(cp, C_join);
		cp->vf = VF_REFER;
		c = expr_post(cp);
	} else if (c == '[') {
		c = expr_brace(cp, 0);
		exerrif(cp, c != ']', "bad index");
		put_c(cp, C_join);
		cp->vf = VF_REFER;
		c = expr_post(cp);
	} else if (c == '(') {
		exerrif(cp, !(vf & VF_UNRESOLV), "bad arguments");
		n = 0;
		do {
			c = expr_brace(cp, 0);
			if (!cp->vf) break;
			n++;
		} while (c == ',');
		exerrif(cp, c != ')', "bad arguments");
		put_c_int(cp, C_call, n);
		cp->vf = (vf == VF_SYMBOL) ? VF_NEW : VF_OBJECT;
		c = expr_post(cp);
	} else if (c==C_inc || c==C_dec) {
		exerrif(cp, !(vf & VF_UNRESOLV), "bad postfix");
		put_c(cp, C_copy);
		put_c(cp, C_load);
		put_c(cp, C_sand);
		put_c(cp, c);
		put_c(cp, C_store);
		put_c(cp, C_pop);
		cp->vf = VF_OBJECT; /* resolved */
		c = lex(cp, LF_RIGHT);
	}
	return c;
}

static int expr_pre(compile_t* cp, int prev)
{
	op_t c;
	int n;
	
	c = lex(cp, 0);
	if (c == '(') {
		exerrif(cp,  cp->vf != 0, "bad brace");
		c = lex(cp, 0);
		exerrif(cp, c <= 0, "bad brace"); /* <= EOF, ERR */
		if (is_typecast(c)) {
			exerrif(cp, lex(cp, 0) != ')', "bad cast");
			n = expr_pre(cp, 0);
			exerrif(cp,  !cp->vf , "bad cast");
			if (cp->vf & VF_UNRESOLV) {
				put_c(cp, C_load);
			}
			put_c(cp, c);
			cp->vf = VF_OBJECT;
			c = n;
		} else {
			if (c != ')') {
				unlex(cp, c);
				c = expr_brace(cp, -1);
			}
			exerrif(cp, c != ')', "bad sub-expr");
			exerrif(cp, !cp->vf, "bad sub-expr");
			c = expr_post(cp);
		}
	} else if (c==C_inc || c==C_dec) {
		exerrif(cp, cp->vf != 0, "bad prefix");
		n = expr_pre(cp, 0);
		exerrif(cp, !(cp->vf & VF_UNRESOLV), "bad prefix");
		put_c(cp, C_copy);
		put_c(cp, C_load);
		put_c(cp, c);
		put_c(cp, C_store);
		put_c(cp, C_load);
		cp->vf = VF_OBJECT;
		c = n;
	} else if (c == C_bitand) { /* `&` reference */
		exerrif(cp, cp->vf != 0, "bad reference");
		c = expr_pre(cp, 0);
		exerrif(cp, !(cp->vf & VF_UNRESOLV), "bad reference");
		cp->vf = VF_OBJECT;
	} else if ((n = is_prefix(c)) > 0) {
		exerrif(cp, cp->vf != 0, "bad prefix");
		c = expr_pre(cp, 0);
		exerrif(cp, !cp->vf, "bad prefix");
		if (cp->vf & VF_UNRESOLV) {
			put_c(cp, C_load);
		}
		put_c(cp, n);
		cp->vf = VF_OBJECT;
	} else if (c=='[') {
		n = 0;
		do {
			c = expr_brace(cp, 0);
			if (!cp->vf) break;
			n++;
		} while (c == ',');
		exerrif(cp, c != ']', "bad const-array");
		put_c_int(cp, C_array, n);
		cp->vf = VF_OBJECT;
		c = lex(cp, LF_RIGHT); /* get right */
	} else if (c == '{') {
		n = 0;
		do {
			c = lex(cp, 0);
			if (c == C_string) {
				put_string(cp, cp->sc, cp->nc);
			} else if (c == C_symbol) {
				put_c_int(cp, C_symbol, add_symbol(cp, cp->sc, cp->nc));
			} else {
				break;
			}
			c = lex(cp, 0);
			exerrif(cp, c != ':', "bad const-object");
			c = expr_brace(cp, 0);
			n++;
		} while (c == ',');
		exerrif(cp, c != '}', "bad const-object");
		put_c_int(cp, C_object, n);
		cp->vf = VF_OBJECT;
		c = lex(cp, LF_RIGHT); /* get right */
	} else if (c == C_regex) {
		put_regex(cp, cp->sc, cp->nc);
		cp->vf = VF_OBJECT;
		c = lex(cp, LF_RIGHT);
	} else if (c == C_string) {
		put_string(cp, cp->sc, cp->nc);
		cp->vf = VF_CONST;
		c = lex(cp, LF_RIGHT);
	} else if (c == C_int) {
		put_c(cp, c);
		put_int(cp, strtol(cp->sc, NULL, 0));
		cp->vf = VF_CONST;
		c = lex(cp, LF_RIGHT);
	} else if (c == C_float) {
		put_c(cp, c);
		put_float(cp, strtod(cp->sc, NULL));
		cp->vf = VF_CONST;
		c = lex(cp, LF_RIGHT);
	} else if (c == C_doc) {
		put_doc(cp, cp->sc, cp->nc, 1);
		cp->vf = VF_CONST;
		c = ';';
	} else if (c == C_null
	|| c == C_true
	|| c == C_false) {
		put_c(cp, c);
		cp->vf = VF_CONST;
		c = lex(cp, LF_RIGHT);
	} else if (c == C_break) {
		exerrif(cp, cp->vf != 0, "bad break");
		exerrif(cp, !(cp->fp->af & AF_BREAK), "bad break");
		put_c_int(cp, C_goto, K_break);
		c = lex(cp, 0);
		exerrif(cp, c != ';', "bad break");
		cp->vf = 0;
	} else if (c == C_continue) {
		exerrif(cp, cp->vf != 0, "bad continue");
		exerrif(cp, !(cp->fp->af & AF_CONTINUE), "bad continue");
		put_c_int(cp, C_goto, K_continue);
		c = lex(cp, 0);
		exerrif(cp, c != ';', "bad continue");
		cp->vf = 0;
	} else if (c == C_goto) {
		exerrif(cp, cp->vf != 0, "bad goto");
		c = lex(cp, 0);
		exerrif(cp, c != C_symbol, "bad goto");
		n = add_symbol(cp, cp->sc, cp->nc);
		put_c_int(cp, C_goto, n);
		c = lex(cp, 0);
		exerrif(cp, c != ';', "bad goto");
		cp->vf = 0;
	} else if (c == C_return) {
		exerrif(cp, cp->vf != 0, "bad return");
		exerrif(cp, !(cp->fp->af & AF_RETURN), "bad return");
		c = lex(cp, 0);
		exerrif(cp, c <= 0, "bad return"); /* <= EOF, ERR */
		if (c == ';') {
			put_c(cp, C_null);
		} else {
			unlex(cp, c);
			c = expr_brace(cp, 0);
		}
		exerrif(cp, c != ';', "bad return");
		put_c(cp, C_return);
		cp->vf = 0;
	} else if (c == C_throw) {
		exerrif(cp, cp->vf != 0, "bad throw");
		c = lex(cp, 0);
		exerrif(cp, c <= 0, "bad throw"); /* <= EOF, ERR */
		if (c == ';') {
			put_c(cp, C_null);
		} else {
			unlex(cp, c);
			c = expr_brace(cp, 0);
		}
		exerrif(cp, c != ';', "bad throw");
		put_c(cp, C_throw);
		cp->vf = 0;
	} else if (c == C_exit) {
		exerrif(cp, cp->vf != 0, "bad exit");
		c = lex(cp, 0);
		exerrif(cp, c <= 0, "bad exit"); /* <= EOF, ERR */
		if (c == ';') {
			put_c(cp, C_null);
		} else {
			unlex(cp, c);
			c = expr_brace(cp, 0);
		}
		exerrif(cp, c != ';', "bad exit");
		put_c(cp, C_exit);
		cp->vf = 0;
	} else if (c == C_function) {
		exerrif(cp, cp->vf != 0, "bad function");
		c = stmt_function(cp, 1);
		exerrif(cp, c != ';', "bad function");
		cp->vf = VF_OBJECT;
		c = lex(cp, 0);
	} else if (c == C_new) {
		exerrif(cp, cp->vf != 0, "bad new");
		c = expr_pre(cp, 0);
		exerrif(cp, cp->vf != VF_NEW, "bad new");
		put_c(cp, C_new);
		cp->vf = VF_OBJECT;
	} else if (c == C_delete) {
		exerrif(cp, cp->vf != 0, "bad delete");
		c = expr_pre(cp, 0);
		exerrif(cp, !(cp->vf & VF_UNRESOLV), "bad delete");
		put_c(cp, C_delete);
		cp->vf = VF_OBJECT;
	} else if (c == C_var) {
		exerrif(cp, cp->vf != 0, "bad var");
		c = lex(cp, 0);
		exerrif(cp, c != C_symbol, "bad var");
		n = add_symbol(cp, cp->sc, cp->nc);
		add_local(cp, C_var, n, 0);
		c = lex(cp, 0);
		exerrif(cp, c != '=', "bad var");
		put_c_int(cp, C_symbol, n);
		cp->vf = VF_SYMBOL;
	} else if (c == C_this) {
		exerrif(cp, !(cp->fp->af & AF_THIS), "bad this");
		put_c(cp, c);
		cp->vf = VF_OBJECT;
		c = expr_post(cp);
	} else if (c == C_symbol) {
		c = expr_symbol(cp);
	}
	return c;
}

static int expr_brace(compile_t* cp, int prev)
{
	op_t c;
	int n,j,e,vf;
	cp->vf = 0;
	j = cp->bn;
	c = expr_pre(cp, 0);
	vf = cp->vf;
	if (!vf) {
		return c;
	}
	else if (c == '=') {
		exerrif(cp, !(vf & VF_UNRESOLV), "bad assign");
		n = expr_brace(cp, 0);
		exerrif(cp, !cp->vf, "bad assign");
		put_c(cp, C_store);
		c = n;
	}
	else if (c == C_andA || c==C_orA) {
		exerrif(cp, !(vf & VF_UNRESOLV), "bad assign");
		put_c(cp, C_copy);
		put_c(cp, C_load);
		put_c(cp, C_copy);
		c = (c == C_andA) ? C_and : C_or;
		put_c(cp, c);
		j = seek_int(cp, 0);
		put_c(cp, C_pop);
		c = expr_brace(cp, 0);
		put_c(cp, C_store);
		set_offset(cp, j);
	}
	else if ((n = is_assign(c)) != 0) {
		exerrif(cp, !(vf & VF_UNRESOLV), "bad assign");
		put_c(cp, C_copy);
		put_c(cp, C_load);
		c = expr_brace(cp, 0);
		put_c(cp, n);
		put_c(cp, C_store);
	}
	else if (c==C_and || c==C_or) {
		if (vf & VF_UNRESOLV) {
			put_c(cp, C_load);
			vf = VF_OBJECT;
		}
		put_c(cp, C_copy);
		put_c(cp, c);
		j = seek_int(cp, 0);
		put_c(cp, C_pop);
		c = expr_brace(cp, 0);
		set_offset(cp, j);
	}
	else if (c == '?') {
		if (vf & VF_UNRESOLV) {
			put_c(cp, C_load);
			vf = VF_OBJECT;
		}
		put_c(cp, C_and);
		j = seek_int(cp, 0);
		c = expr_brace(cp, 0);
		e = seek_int(cp, 0);
		exerrif(cp, c != ':', "bad `?:` statement");
		set_offset(cp, j);
		c = expr_brace(cp, 0);
		set_offset(cp, e);
	}
	else if ((n = is_right(c)) != 0) {
		if (vf & VF_UNRESOLV) {
			put_c(cp, C_load);
			vf = VF_OBJECT;
		}
		if (prev < 0) prev = 0;
		while (n > prev) {
			e = cp->bn;
			n = expr_brace(cp, n);
			if ((vf == VF_CONST) && (cp->vf == VF_CONST)) {
				calc_const(cp, j, c, e);
			} else {
				put_c(cp, c);
				vf = VF_OBJECT;
			}
			c = n;
			n = is_right(c);
		}
	}
	if ((cp->vf = vf) & VF_UNRESOLV) {
		if (prev >= 0) {
			put_c(cp, C_load);
			cp->vf = VF_OBJECT;
		}
	}
	return c;
}

static int expr_vars(compile_t* cp)
{
	op_t c;
	int k;
	do {
		c = lex(cp, 0);
		exerrif(cp, c != C_symbol, "bad vars");
		k = add_symbol(cp, cp->sc, cp->nc);
		add_local(cp, C_var, k, 0);
		
		c = lex(cp, 0);
		if (c == '=') {
			put_c_int(cp, C_symbol, k);
			
			put_lineno(cp);
			c = expr_brace(cp, 0);
			exerrif(cp, !cp->vf, "bad vars");
			
			put_c(cp, C_store);
			put_c(cp, C_pop);
		}
	} while (c == ',');
	return c;
}

static int expr_line(compile_t* cp)
{
	op_t c;
	do {
		
		put_lineno(cp);
		c = expr_brace(cp, 0);
		
		if (!c || !cp->vf) {
			;
		} else if (c == -1) { /* EOF */
			put_c(cp, C_return);
		} else if (!(cp->lf & LF_ECHO)) {
			put_c(cp, C_pop);
		} else {
			if (c == ',') {
				put_c(cp, C_echo);
				put_c(cp, ' ');
			} else if (c == ';') {
				put_c(cp, C_echo);
				put_c(cp, '\n');
			} else if (c == C_doc) {
				put_c(cp, C_echo);
				put_c(cp, 0);
				unlex(cp, c);
				c = ';';
			} else {
				fprintf(cp->err, "bad result: `%c`\n", c);
				return 0;
			}
		}
	} while (c == ',');
	return c;
}

/************************************************************
 *
 ************************************************************/
static int frame_in(compile_t* cp, cp_frame_t* fp, int af)
{
	memset(fp, 0, sizeof(cp_frame_t));
	fp->p = cp->fp;
	fp->vi = cp->vn;
	fp->af = (cp->fp ? (cp->fp->af & 0xFF) : 0) | af;
	fp->bi = cp->bn;
	cp->fp = fp;
	return 1;
}
static int frame_out(compile_t* cp, cp_frame_t* fp)
{
	size_t sz, ad;
	int i;
	
	cp->fp = fp->p;
	
	ad = 0;
	if ((fp->af & AF_NAMES) && cp->np) {
		ad += 1 + 4;
		for (i = 0; i < cp->nn; i++) {
			ad += cp->np[i].n + 1;
		}
	}
	if ((fp->af & AF_FILE) && cp->fn) {
		ad += 1 + strlen(cp->fn) + 1;
	}
	for (i = fp->vi; i < cp->vn; i++) {
		ad += 1 + 4;
		if (cp->vp[i].j != 0) ad += 4;
	}
	
	if (ad > 0) {
		if (!(fp->af & AF_GLOBAL)) {
			ad++;
			put_c(cp, C_frameOut);
		}
		sz = cp->bn - fp->bi;
		/* alloc ad */
		memmove(cp->bc + fp->bi + ad, cp->bc + fp->bi, sz);
		cp->bn = fp->bi;
		if ((fp->af & AF_NAMES) && cp->np) {
			put_c_int(cp, C_names, cp->nn);
			for (i = 0; i < cp->nn; i++) {
				put_chars(cp, cp->np[i].s, cp->np[i].n);
			}
		}
		if (!(fp->af & AF_GLOBAL)) {
			put_c(cp, C_frameIn);
		}
		if ((fp->af & AF_FILE) && cp->fn) {
			put_c(cp, C_file);
			put_chars(cp, cp->fn, strlen(cp->fn));
		}
		for (i = fp->vi; i < cp->vn; i++) {
			put_c_int(cp, cp->vp[i].c, cp->vp[i].k);
			if (cp->vp[i].j != 0) {
				put_offset(cp, cp->vp[i].j + ad);
			}
		}
		cp->bn += sz;
		cp->bc[cp->bn] = 0;
	}
	
	cp->vn = fp->vi;
	
	if (fp->af & AF_NAMES) {
		for (i = 0; i < cp->nn; i++) free(cp->np[i].s);
		for (i = 0; i < cp->dn; i++) free(cp->dp[i].s);
		if (cp->np) free(cp->np);
		if (cp->dp) free(cp->dp);
		if (cp->vp) free(cp->vp);
	}
	
	return 1;
}

/************************************************************
 *
 ************************************************************/
static int stmt_line(compile_t* cp);
static int stmt_file(compile_t* cp, cp_frame_t* fp);
static int stmt_block(compile_t* cp);

/************************************************************
 *
 ************************************************************/
static int stmt_Minclude(compile_t* cp)
{
	compile_t old;
	FILE* fp;
	int sz;
	char *sc;
	char *fn;
	op_t c;
	int i, n;
	
	i = cp->bn;
	c = expr_brace(cp, 0);
	exerrif(cp, c != C_crlf, "bad include");
	exerrif(cp, cp->vf != VF_CONST, "bad include");
	
	exerrif(cp, cp->bc[i] != C_string, "bad include");
	cp->bc[i++] = C_file;
	fn = cp->bc + i;
	n = *(int*)fn;
	i += n;
	memmove(fn, fn + 4, n);
	cp->bc[i++] = 0;
	cp->bn = i;
	
	if (cp->fn && !strcmp(cp->fn, fn)) {
		cp->bn = i;
		return ';';
	}
	
	if (!(fp = fopen(fn, "r"))) {
		fprintf(cp->err, "fopen error: %s %s\n", fn, strerror(errno));
		return 0;
	}
	fseek(fp, 0, SEEK_END);
	sz = ftell(fp);
	rewind(fp);
	sc = mmap(0, sz, PROT_READ, MAP_SHARED, fileno(fp), 0);
	assert(sc && (sc != MAP_FAILED));
	
	memcpy(&old, cp, sizeof(compile_t));
	memset(cp, 0, sizeof(compile_t));
	cp->np   = old.np;
	cp->nn   = old.nn;
	cp->vp   = old.vp;
	cp->vn   = old.vn;
	cp->fp   = old.fp;
	cp->bc   = old.bc;
	cp->bn   = old.bn;
	cp->out = old.out;
	cp->err = old.err;
	cp->in  = old.in;
	cp->fn   = fn;
	cp->ln   = 1;
	cp->ss   = sc;
	cp->sc   = sc;
	cp->es   = sc + sz;
	
	c = stmt_file(cp, NULL);
	
	// vmclose(sc, sz, fp);
	assert( munmap(sc, sz) == 0 );
	assert( fclose(fp) == 0 );
	
	old.np = cp->np;
	old.nn = cp->nn;
	old.vp = cp->vp;
	old.vn = cp->vn;
	old.fp = cp->fp;
	old.bc = cp->bc;
	old.bn = cp->bn;
	memcpy(cp, &old, sizeof(compile_t));
	
	put_c(cp, C_file);
	put_chars(cp, cp->fn, strlen(cp->fn));
	
	return ';';
}

static int stmt_Mdefine(compile_t* cp)
{
	op_t c;
	int k, i;
	c = lex(cp, 0);
	exerrif(cp, c != C_symbol, "bad define");
	k = strtoh(cp->sc, cp->nc);
	i = cp->bn;
	c = expr_brace(cp, 0);
	exerrif(cp, c != C_crlf, "bad define");
	if (cp->bn == i) {
		put_c(cp, C_null);
	} else {
		exerrif(cp, cp->vf != VF_CONST, "bad define");
	}
	add_define(cp, k, cp->bc + i, cp->bn - i);
	cp->bn = i;
	return ';';
}

/************************************************************
 *
 ************************************************************/
static int stmt_if(compile_t* cp)
{
	op_t c;
	int j, e;
	
	c = lex(cp, 0);
	
	exerrif(cp, c != '(', "bad if expr");
	
	put_lineno(cp);
	c = expr_brace(cp, 0);
	exerrif(cp, !cp->vf, "bad if expr");
	exerrif(cp, c != ')', "bad if expr");
	
	put_c(cp, C_and);
	j = seek_int(cp, 0);
	c = stmt_line(cp);
	exerrif(cp, c != ';', "bad if statement");
	
	c = lex(cp, LF_LEFT);
	if (c == C_else) {
		put_c(cp, C_jump);
		e = seek_int(cp, 0);
		set_offset(cp, j);
		c = stmt_line(cp);
		set_offset(cp, e);
	} else {
		if ((c > 0) && (c != ';')) { /* > EOF, ERR */
			unlex(cp, c);
			c = ';';
		}
		set_offset(cp, j);
	}
	return c;
}

static int stmt_foreach(compile_t* cp)
{
	cp_frame_t fp;
	op_t c;
	int d, k, v, s, e;
	
	frame_in(cp, &fp, AF_BREAK | AF_CONTINUE);
	
	c = lex(cp, 0);
	exerrif(cp, c != '(', "bad foreach expr");
	
	c = lex(cp, 0);
	d = 0; /* define or not */
	if (c == C_var) {
		c = lex(cp, 0);
		d = 1;
	}
	exerrif(cp, c != C_symbol, "bad symbol");
	k = 0;
	v = add_symbol(cp, cp->sc, cp->nc);
	if (d) {
		add_local(cp, C_var, v, 0);
	}
	c = lex(cp, 0);
	if (c == ',') {
		c = lex(cp, 0);
		exerrif(cp, c != C_symbol, "bad symbol");
		k = v;
		v = add_symbol(cp, cp->sc, cp->nc);
		if (d) {
			add_local(cp, C_var, v, 0);
		}
		c = lex(cp, 0);
	}
	exerrif(cp, c != C_in, "not 'IN'");
	
	put_c_int(cp, C_symbol, k);
	put_c_int(cp, C_symbol, v);
	
	put_lineno(cp);
	c = expr_brace(cp, 0);
	exerrif(cp, !cp->vf, "bad foreach expr");
	exerrif(cp, c != ')', "bad foreach expr");
	
	s = cp->bn;
	add_local(cp, C_label, K_continue, cp->bn);
	
	put_c(cp, C_in);
	e = seek_int(cp, 0);
	
	c = stmt_line(cp);
	
	put_c(cp, C_jump);
	put_offset(cp, s);
	
	add_local(cp, C_label, K_break, cp->bn);
	set_offset(cp, e);
	
	put_c(cp, C_pop);
	
	frame_out(cp, &fp);
	
	return c;
}

static int stmt_for(compile_t* cp)
{
	cp_frame_t fp;
	op_t c;
	int s, r, j, e;
	
	frame_in(cp, &fp, AF_BREAK | AF_CONTINUE);
	
	c = lex(cp, 0);
	exerrif(cp, c != '(', "bad for expr");
	
	c = lex(cp, 0);
	exerrif(cp, c <= 0, "bad for expr"); /* <= EOF, ERR */
	if (c == C_var) {
		c = expr_vars(cp);
	} else {
		if (c != ';') {
			unlex(cp, c);
			c = expr_line(cp);
		}
	}
	exerrif(cp, c != ';', "bad for expr");
	
	r = s = cp->bn;
	
	put_lineno(cp);
	c = expr_brace(cp, 0);
	exerrif(cp, !cp->vf, "bad for expr");
	exerrif(cp, c != ';', "bad for expr");
	
	put_c(cp, C_loop);
	e = seek_int(cp, 0);
	
	c = lex(cp, 0);
	exerrif(cp, c <= 0, "bad for expr"); /* <= EOF, ERR */
	if (c != ')') {
		unlex(cp, c);
		
		put_c(cp, C_jump);
		j = seek_int(cp, 0);
		
		r = cp->bn;
		c = expr_line(cp);
		exerrif(cp, c != ')', "bad for expr");
		
		put_c(cp, C_jump);
		put_offset(cp, s);
		set_offset(cp, j);
	}
	add_local(cp, C_label, K_continue, r);
	
	c = stmt_line(cp);
	
	put_c(cp, C_jump);
	put_offset(cp, r);
	
	set_offset(cp, e);
	add_local(cp, C_label, K_break, cp->bn);
	
	frame_out(cp, &fp);
	return c;
}

static int stmt_while(compile_t* cp)
{
	cp_frame_t fp;
	op_t c;
	int s, e;
	
	frame_in(cp, &fp, AF_BREAK | AF_CONTINUE);
	
	s = cp->bn;
	add_local(cp, C_label, K_continue, cp->bn);
	c = lex(cp, 0);
	exerrif(cp, c != '(', "bad while expr");
	
	put_lineno(cp);
	c = expr_brace(cp, 0);
	exerrif(cp, !cp->vf, "bad while expr");
	exerrif(cp, c != ')', "bad while expr");
	
	put_c(cp, C_loop);
	e = seek_int(cp, 0);
	
	c = stmt_line(cp);
	
	put_c(cp, C_jump);
	put_offset(cp, s);
	
	set_offset(cp, e);
	add_local(cp, C_label, K_break, cp->bn);
	
	frame_out(cp, &fp);
	return c;
}

static int stmt_do(compile_t* cp)
{
	cp_frame_t fp;
	op_t c;
	int s, e;
	
	frame_in(cp, &fp, AF_BREAK | AF_CONTINUE);
	
	s = cp->bn;
	c = stmt_line(cp);
	exerrif(cp, c != ';', "bad do");
	
	c = lex(cp, 0);
	exerrif(cp, c != C_while, "bad do");
	
	c = lex(cp, 0);
	exerrif(cp, c != '(', "bad do expr");
	
	add_local(cp, C_label, K_continue, cp->bn);
	
	put_lineno(cp);
	c = expr_brace(cp, 0);
	exerrif(cp, !cp->vf, "bad do expr");
	exerrif(cp, c != ')', "bad do expr");
	
	put_c(cp, C_loop);
	e = seek_int(cp, 0);
	
	c = stmt_line(cp);
	
	put_c(cp, C_jump);
	put_offset(cp, s);
	
	set_offset(cp, e);
	add_local(cp, C_label, K_break, cp->bn);
	
	frame_out(cp, &fp);
	return c;
}

static int stmt_switch(compile_t* cp)
{
	cp_frame_t fp;
	op_t c;
	int j, k;
	
	frame_in(cp, &fp, AF_BREAK | AF_BLOCK);
	
	c = lex(cp, 0);
	exerrif(cp, c != '(', "bad switch expr");
	
	put_lineno(cp);
	c = expr_brace(cp, 0);
	exerrif(cp, !cp->vf, "bad switch expr");
	exerrif(cp, c != ')', "bad switch expr");
	
	exerrif(cp, lex(cp, LF_LEFT) != '{', "bad switch statement");
	
	j = k = 0;
	do {
		c = lex(cp, LF_LEFT);
		exerrif(cp, c <= 0, "bad switch content"); /* <= EOF, ERR */
		if (c == C_case) {
			if (j) {
				put_c(cp, C_jump);
				k = seek_int(cp, k);
				set_offset(cp, j);
			}

			put_lineno(cp);
			c = expr_brace(cp, 0);
			exerrif(cp, !cp->vf, "bad case");
			exerrif(cp, c != ':', "bad case");
			
			put_c(cp, C_case);
			j = seek_int(cp, 0);
			c = ';';
		} else if (c == C_default) {
			c = lex(cp, 0);
			exerrif(cp, c != ':', "bad default");
			if (j) set_offset(cp, j);
			if (k) set_offset(cp, k);
			j = k = 0;
			c = ';';
		} else {
			if (c != ';' && c != '}') {
				while (k) {
					k = set_offset(cp, k);
				}
				unlex(cp, c);
				c = stmt_line(cp);
			}
		}
	} while (c == ';');
	
	exerrif(cp, c != '}', "bad switch statement");
	
	if (j) set_offset(cp, j);
	if (k) set_offset(cp, k);
	
	add_local(cp, C_label, K_break, cp->bn);
	put_c(cp, C_pop);
	
	frame_out(cp, &fp);
	
	return ';';
}

static int stmt_try(compile_t* cp)
{
	cp_frame_t fp;
	op_t c;
	int j;
	
	frame_in(cp, &fp, AF_THROW);
	
	put_c(cp, C_tmperr);
	
	c = stmt_line(cp);
	exerrif(cp, c != ';', "bad try statement");
	
	c = lex(cp, LF_LEFT);
	if (c == C_catch) {
		put_c(cp, C_jump);
		j = seek_int(cp, 0);
		add_local(cp, C_label, K_catch, cp->bn);
		c = lex(cp, 0);
		exerrif(cp, c != '(', "bad catch arg");
		c = lex(cp, 0);
		exerrif(cp, c != C_symbol, "bad symbol");
		put_c_int(cp, C_catch, add_symbol(cp, cp->sc, cp->nc));
		c = lex(cp, 0);
		exerrif(cp, c != ')', "bad catch expr");
		c = stmt_line(cp);
		set_offset(cp, j);
	} else {
		if ((c > 0) && (c != ';')) { /* > EOF, ERR */
			unlex(cp, c);
			c = ';';
		}
		add_local(cp, C_label, K_catch, cp->bn);
		put_c_int(cp, C_catch, 0);
	}
	
	frame_out(cp, &fp);
	
	return c;
}

static int stmt_function(compile_t* cp, int prev)
{
	cp_frame_t fp;
	op_t c;
	int k, n, j;
	
	if (!prev) {
		
		c = lex(cp, 0);
		exerrif(cp, c != C_symbol, "bad function symbol");
		
		k = add_symbol(cp, cp->sc, cp->nc);
		
		put_c(cp, C_jump);
		j = seek_int(cp, 0); /* seek before `C_function` */
		add_local(cp, C_function, k, cp->bn);
		
	} else {
		put_c_int(cp, C_function, 0);
		j = seek_int(cp, 0);
	}
	
	frame_in(cp, &fp, AF_FILE | AF_RETURN);
	
	c = lex(cp, 0);
	exerrif(cp, c != '(', "bad function args");
	
	n = 0;
	do {
		c = lex(cp, 0);
		if (c == C_symbol) {
			k = add_symbol(cp, cp->sc, cp->nc);
			add_local(cp, C_var, k, 0);
			put_c_int(cp, C_symbol, k);
			n++;
			c = lex(cp, 0);
		}
	} while (c == ',');
	exerrif(cp, c != ')', "bad function args");
	
	put_c_int(cp, C_args, n);
	
	c = lex(cp, LF_LEFT);
	exerrif(cp, c != '{', "bad function statement");
	c = stmt_block(cp);
	exerrif(cp, c != ';', "bad function statement");
	
	put_c(cp, C_null);
	put_c(cp, C_return);
	frame_out(cp, &fp);
	
	set_offset(cp, j);
	
	return ';';
}

static int stmt_class(compile_t* cp)
{
	cp_frame_t fp;
	op_t c;
	int k, n, j;
	
	put_c(cp, C_jump);
	j = seek_int(cp, 0);
	
	c = lex(cp, 0);
	exerrif(cp, c != C_symbol, "bad class symbol");
	
	k = add_symbol(cp, cp->sc, cp->nc);
	add_local(cp, C_class, k, cp->bn);
	
	frame_in(cp, &fp,
		AF_FILE | AF_THIS
		| AF_FUNCTION | AF_GLOBAL);
	
	c = lex(cp, 0);
	if (c == '(') {
		n = 0;
		do {
			c = lex(cp, 0);
			if (c == C_symbol) {
				k = add_symbol(cp, cp->sc, cp->nc);
				add_local(cp, C_var, k, 0);
				put_c_int(cp, C_symbol, k);
				n++;
				c = lex(cp, 0);
			}
		} while (c == ',');
		exerrif(cp, c != ')', "bad cass args");
		
		put_c_int(cp, C_args, n);
		
		c = lex(cp, LF_LEFT);
	}
	
	exerrif(cp, c != '{', "bad class statement");
	c = stmt_block(cp);
	exerrif(cp, c != ';', "bad class statement");
	
	put_c(cp, C_this);
	put_c(cp, C_return);
	frame_out(cp, &fp);
	
	set_offset(cp, j);
	
	return ';';
}

static int stmt_block(compile_t* cp)
{
	cp_frame_t fp;
	int af = 0;
	op_t c;
	
	if (!(cp->fp->af & AF_BLOCK)) {
		assert((af = cp->fp->af) != 0);
		cp->fp->af |= AF_BLOCK;
	} else {
		frame_in(cp, &fp, AF_BLOCK);
	}
	
	do {
		c = stmt_line(cp);
	} while (c == ';');
	exerrif(cp, c != '}', "bad block");
	
	if (af) {
		cp->fp->af = af;
	} else {
		frame_out(cp, &fp);
	}
	
	return ';';
}

static int stmt_line(compile_t* cp)
{
	op_t c;
	c = lex(cp, LF_LEFT);
	if (c == '{') {
		c = stmt_block(cp);
	} else if (c == C_doc) {
		put_doc(cp, cp->sc, cp->nc, 0);
		c = ';';
	} else if (c == C_if) {
		c = stmt_if(cp);
	} else if (c == C_for) {
		c = stmt_for(cp);
	} else if (c == C_foreach) {
		c = stmt_foreach(cp);
	} else if (c == C_while) {
		c = stmt_while(cp);
	} else if (c == C_do) {
		c = stmt_do(cp);
	} else if (c == C_switch) {
		c = stmt_switch(cp);
	} else if (c == C_try) {
		c = stmt_try(cp);
	} else if (c == C_class) {
		exerrif(cp, !(cp->fp->af & AF_CLASS), "bad class");
		c = stmt_class(cp);
	} else if (c == C_function) {
		exerrif(cp, !(cp->fp->af & AF_FUNCTION), "bad function");
		c = stmt_function(cp, 0);
	} else if (c == C_Minclude) {
		c = stmt_Minclude(cp);
	} else if (c == C_Mdefine) {
		c = stmt_Mdefine(cp);
	} else if (c == C_label) {
		c = lex(cp, 0);
		exerrif(cp, c != C_symbol, "bad label");
		add_local(cp, C_label, add_symbol(cp, cp->sc, cp->nc), cp->bn);
		c = lex(cp, 0);
		exerrif(cp, c != ':', "bad label");
		c = ';';
	} else if (c == C_var) {
		c = expr_vars(cp);
	} else {
		if ((c > 0) && (c != ';') && (c != '}')) { /* > EOF, ERR */
			unlex(cp, c);
			c = expr_line(cp);
		}
	}
	return c;
}

/************************************************************
 *
 ************************************************************/
static int stmt_file(compile_t* cp, cp_frame_t* fp)
{
	op_t c;
	int sz;
	
	if (*cp->sc=='#' && *(cp->sc + 1)=='!') {
		while (*cp->sc && !iscrlf(*cp->sc)) cp->sc++;
	}
	while (isspace(*cp->sc)) cp->sc++;
	sz = cp->es - cp->sc;
	
	if (*cp->sc == 1) {
		cp->bc = realloc(cp->bc, cp->bn + sz + 1);
		memcpy(cp->bc + cp->bn, cp->sc, sz);
		cp->bn += sz;
		return 1;
	}
	
	cp->bc = realloc(cp->bc, cp->bn
		+ (((sz < 256) ? 256 : sz) * 4) );
	
	cp->bc[cp->bn++] = 1; /* void */
	
	if (fp) {
		frame_in(cp, fp,
			AF_NAMES | AF_GLOBAL
			| AF_CLASS| AF_FUNCTION
			| AF_FILE | AF_BLOCK);
	}
	
	if (*cp->sc == '<') {
		seek_doc(cp);
		put_doc(cp, cp->sc, cp->nc, 0);
	}
	
	do {
		c = stmt_line(cp);
	} while (c == ';');
	
	if (c != -1) { /* != EOF */
		char *s, *e;
		s = e = cp->sc;
		while (((cp->sc - s) < 16) && (s > cp->ss)
			&& (*(s-1) != 0) && !iscrlf(*(s-1))) { s--; }
		while (((e - cp->sc) < 16) && (e < cp->es)
			&& (*(e+1) != 0) && !iscrlf(*(e+1))) { e++; }
		fprintf(cp->err, "%.*s>%.*s [%s:%d]\n",
			(int)(cp->sc - s), s, (int)(e - cp->sc), cp->sc,
			cp->fn, cp->ln);
	}
	
	if (fp) {
		frame_out(cp, fp);
	}
	
	return c;
}

/************************************************************
 *
 ************************************************************/
int compile_code(exec_t* ex, char* fn, char *sc, int sz, char **bc, int* bn)
{
	compile_t cp;
	cp_frame_t fp;
	int r;
	
	memset(&cp, 0, sizeof(compile_t));
	cp.bc = *bc;
	cp.bn = *bn;
	cp.fn = fn;
	cp.ln = 1;
	cp.ss = sc;
	cp.sc = sc;
	cp.es = sc + sz;
	cp.out = ex->out;
	cp.err = ex->err;
	cp.in  = ex->in;
	
	r = 0;
	if (!stmt_file(&cp, &fp)) {
		r = -1;
	}
	
	*bc = cp.bc;
	*bn = cp.bn;
	
	return r;
}

int compile_file(exec_t* ex, char *fn, char **bc, int* bn)
{
	FILE *fp;
	char* sc;
	int sz, r;
	
	if (!(fp = fopen(fn, "r"))) {
		fprintf(ex->err, "fopen error: %s %s\n", fn, strerror(errno));
		return 0;
	}
	fseek(fp, 0, SEEK_END);
	sz = ftell(fp);
	rewind(fp);
	sc = mmap(0, sz, PROT_READ, MAP_SHARED, fileno(fp), 0);
	assert(sc && (sc != MAP_FAILED));
	
	r = compile_code(ex, fn, sc, sz, bc, bn);
	
	// vmclose(sc, sz, fp);
	assert( munmap(sc, sz) == 0 );
	assert( fclose(fp) == 0 );
	
	return r;
}

int compile_file_to_file(exec_t* ex, char *infn, char *outfn)
{
	FILE* fp;
	char *bc;
	int bn;
	int r;
	
	bc = NULL;
	bn = 0;
	
	r = compile_file(ex, infn, &bc, &bn);
	if (r != 0) {
		if (bc) free(bc);
		return r;
	}
	
	fp = fopen(outfn, "w");
	if (!fp) {
		fprintf(ex->err, "fopen error: %s %s\n", outfn, strerror(errno));
	} else {
		fputs("#!/usr/bin/js\n", fp);
		fwrite(bc, 1, bn, fp);
		fclose(fp);
		r = chmod(outfn, 0755);
	}
	
	free(bc);
	
	return r;
}
