/*
 * Copyright (c) 2007, to-do. All rights reserved.
 */
#include "util.h"
#include "var.h"
#include "_sym.h"
#include "_ext.h"

/*
  c/C ... char 
  b/B ... bit string
  h/H ... hex string
  i/I ... int machine
  s/S ... short machine
  l/L ... long machine
  f/F ... float machine
  d/D ... double machine
  n/N ... network byte order
  v/V ... 
  x   ... 0
*/

int pack_to_bytes(exec_t* ex, char **dst, char* fmt, var_t* va, int vn)
{
	int num, len;
	char *s;
	unsigned long u;
	int vi, c, j, n, r;
	double d;
	
	*dst = NULL;
	len = 0;
	vi = 0;
	while ((vi < vn) && ((c = *fmt++) != 0)) {
		if (!isalpha(c)) {
			continue;
		}
		num = 1;
		if (isdigit(*fmt)) {
			do {
				num = (num * 10) + (*fmt++ - '0');
			} while (isdigit(*fmt));
		} else if (*fmt == '*') {
			num = -1;
			fmt++;
		}
		switch (c) {
		case 'c':
		case 'C':
			if (num < 0) num = vn - vi;
			*dst = realloc(*dst, len + num + 1);
			while (num-- > 0) {
				c = (vi < vn) ? var_toChar(ex, va + vi++) : 0;
				*(*dst + len++) = c;
			}
			break;
		/* as machine */
		case 's':
		case 'S':
			if (num < 0) num = vn - vi;
			*dst = realloc(*dst, len + (num * sizeof(short)) + 1);
			while (num-- > 0) {
				j = (vi < vn) ? var_toInt(ex, va + vi++) : 0;
				*(short*)(*dst + len) = (short)j;
				len += sizeof(short);
			}
			break;
		case 'i':
		case 'I':
			if (num < 0) num = vn - vi;
			*dst = realloc(*dst, len + (num * sizeof(int)) + 1);
			while (num-- > 0) {
				j = (vi < vn) ? var_toInt(ex, va + vi++) : 0;
				*(int*)(*dst + len) = (int)j;
				len += sizeof(int);
			}
			break;
		case 'n':
		case 'N':
			if (num < 0) num = vn - vi;
			n = islower(c) ? sizeof(short) : sizeof(int);
			*dst = realloc(*dst, len + (num * n) + 1);
			while (num-- > 0) {
				u = (vi < vn) ? var_toInt(ex, va + vi++) : 0;
				for (j = n - 1; j >= 0; j--) {
					*(*dst + len++) = (u >> (j * 8)) & 0xFF;
				}
			}
			break;
		case 'v':
		case 'V':
			if (num < 0) num = vn - vi;
			n = islower(c) ? sizeof(short) : sizeof(int);
			*dst = realloc(*dst, len + (num * n) + 1);
			while (num-- > 0) {
				u = (vi < vn) ? var_toInt(ex, va + vi++) : 0;
				for (j = 0; j < n; j++) {
					*(*dst + len++) = (u >> (j * 8)) & 0xFF;
				}
			}
			break;
		case 'f':
		case 'F':
			if (num < 0) num = vn - vi;
			*dst = realloc(*dst, len + (num * sizeof(float)) + 1);
			while (num-- > 0) {
				d = (vi < vn) ? var_toFloat(ex, va + vi++) : 0;
				*(float*)(*dst + len) = (float)d;
				len += sizeof(float);
			}
			break;
		case 'd':
		case 'D':
			if (num < 0) num = vn - vi;
			*dst = realloc(*dst, len + (num * sizeof(double)) + 1);
			while (num-- > 0) {
				d = (vi < vn) ? var_toFloat(ex, va + vi++) : 0;
				*(double*)(*dst + len) = (double)d;
				len += sizeof(double);
			}
			break;
		case 'b':
		case 'B':
			if ((vi >= vn) || !var_toString(ex, va + vi++, &s, &n)) {
				n = 0;
			}
			if (num < 0) num = n;
			*dst = realloc(*dst, len + 1 + (num / 8) + 1);
			r = islower(c);
			u = 0;
			j = 0;
			while (num-- > 0) {
				c = (j < n) ? (s[j] == '1') : 0;
				if (c) {
					if (r) {
						u |= 1 << (j % 8);
					} else {
						u |= 1 << (7 - (j % 8));
					}
				}
				if (!(++j % 8)) {
					*(*dst + len++) = u & 0xFF;
					u = 0;
				}
			}
			if (u) {
				*(*dst + len++) = u & 0xFF;
			}
			break;
		case 'h':
		case 'H':
			if ((vi >= vn) || !var_toString(ex, va + vi++, &s, &n)) {
				n = 0;
			}
			if (num < 0) num = n;
			*dst = realloc(*dst, len + 1 + (num / 2) + 1);
			r = islower(c);
			u = 0;
			j = 0;
			while (num-- > 0) {
				c = (j < n) ? xtoi(s[j]) : 0;
				if (c) {
					if (r) {
						u |= (j % 2) ? (c << 4) : c;
					} else {
						u |= (j % 2) ? c : (c << 4);
					}
				}
				if (!(++j % 2)) {
					*(*dst + len++) = u & 0xFF;
					u = 0;
				}
			}
			if (u) {
				*(*dst + len++) = u;
			}
			break;
		case 'x':
			if (num < 0) num = vn - vi;
			*dst = realloc(*dst, len + num + 1);
			while (num-- > 0) {
				*(*dst + len++) = 0;
			}
			break;
		default:
			if (num < 0) num = vn - vi;
			*dst = realloc(*dst, len + num + 1);
			while (num-- > 0) {
				*(*dst + len++) = c;
			}
			break;
		}
	}
	if (!len) {
		if (*dst) free(*dst);
		*dst = NULL;
		return 0;
	}
	return len;
}

int unpack_from_bytes(exec_t* ex, char *fmt, char *src, int len, var_t* res)
{
	array_t* arr;
	int num;
	char *s;
	unsigned long u;
	int si, c, r, j, n;
	double d;
	
	arr = array_alloc(ex);
	si = 0;
	while ((si < len) && ((c = *fmt++) != 0)) {
		if (!isalpha(c)) {
			continue;
		}
		num = 1;
		if (isdigit(*fmt)) {
			num = 0;
			do {
				num = (num * 10) + (*fmt++ - '0');
			} while (isdigit(*fmt));
		} else if (*fmt == '*') {
			num = -1;
			fmt++;
		}
		switch (c) {
		case 'c':
		case 'C':
			if (num < 0) num = len - si;
			while (num-- > 0) {
				c = src[si++];
				new_char(res, c);
				array_push(ex, arr, res);
			}
			break;
		case 's':
		case 'S':
			if (num < 0) num = (len - si) / sizeof(short);
			while (num-- > 0) {
				j = *(short*)(src + si);
				si += sizeof(short);
				new_int(res, j);
				array_push(ex, arr, res);
			}
			break;
		/* as machine */
		case 'i':
		case 'I':
			if (num < 0) num = (len - si) / sizeof(int);
			while (num-- > 0) {
				j = *(int*)(src + si);
				si += sizeof(int);
				new_int(res, j);
				array_push(ex, arr, res);
			}
			break;
		case 'n':
		case 'N':
			n = islower(c) ? sizeof(short) : sizeof(int);
			if (num < 0) num = (len - si) / n;
			while (num-- > 0) {
				u = 0;
				for (j = (n - 1); j >= 0; j--) {
					u |= src[si++] << (j * 8);
				}
				new_int(res, u);
				array_push(ex, arr, res);
			}
			break;
		case 'v':
		case 'V':
			n = islower(c) ? sizeof(short) : sizeof(int);
			if (num < 0) num = (len - si) / n;
			while (num-- > 0) {
				u = 0;
				for (j = 0; j < n; j++) {
					u |= src[si++] << (j * 8);
				}
				new_int(res, u);
				array_push(ex, arr, res);
			}
			break;
		case 'f':
		case 'F':
			if (num < 0) num = (len - si) / sizeof(float);
			while (num-- > 0) {
				d = *(float*)(src + si);
				si += sizeof(float);
				new_float(res, d);
				array_push(ex, arr, res);
			}
			break;
		case 'd':
		case 'D':
			if (num < 0) num = (len - si) / sizeof(double);
			while (num-- > 0) {
				d = *(double*)src;
				src += sizeof(double);
				new_float(res, d);
				array_push(ex, arr, res);
			}
			break;
		case 'b':
		case 'B':
			if (num < 0) num = (len - si) * 8;
			r = islower(c);
			u = 0;
			j = 0;
			s = malloc(num + 1);
			n = 0;
			while (num-- > 0) {
				if (!(j % 8)) {
					u = src[si++];
				}
				if (r) {
					c = u >> (j % 8);
				} else {
					c = u >> (7 - (j % 8));
				}
				s[n++] = (c & 1) ? '1' : '0';
				j++;
			}
			s[n] = 0;
			new_string_with(ex, res, s, n);
			array_push(ex, arr, res);
			break;
		case 'h':
		case 'H':
			if (num < 0) num = (len - si) * 2;
			r = islower(c);
			u = 0;
			j = 0;
			s = malloc(num + 1);
			n = 0;
			while (num-- > 0) {
				if (!(j % 2)) {
					u = src[si++];
				}
				if (r) {
					c = !(j % 2) ? (u & 0xf) : (u >> 4);
				} else {
					c = !(j % 2) ? (u >> 4) : (u & 0xf);
				}
				s[n++] = itoX(c & 0xf);
				j++;
			}
			s[n] = 0;
			new_string_with(ex, res, s, n);
			array_push(ex, arr, res);
			break;
		case 'x':
			if (num < 0) num = len - si;
			while (num-- > 0) {
				if (src[si++] != 0) {
					;
				}
			}
			break;
		default:
			if (num < 0) num = len - si;
			s = malloc(num + 1);
			n = 0;
			while (num-- > 0) {
				if (src[si++] != 0) {
					;
				}
				s[n++] = c;
			}
			s[n] = 0;
			new_string_with(ex, res, s, n);
			array_push(ex, arr, res);
			break;
		}
	}
	if (!arr->n) {
		array_free(ex, arr);
		return 0;
	}
	if (arr->n == 1) {
		array_pop(ex, arr, res);
		array_free(ex, arr);
		return 1;
	}
	new_array(ex, res, arr);
	return arr->n;
}
