/*
 * Java Temlate Project
 * 
 * Copyright (C) 2006 Satoshi Nagashiba, All Rights Reserved.
 */
package org.jtp.common.util;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;

import org.apache.commons.lang.ArrayUtils;

/**
 * 日付関連のユーティリティクラスです。
 * 
 * @author <a href="mailto:nagashiba.satoshi@cosmos-ss.co.jp@cosmos-ss.co.jp">Satoshi Nagashiba</a>
 */
public class DateUtils {

    private static final int DATE_PREFIX = -1;

    private static final int MONTH_PREFIX = 0;

    private static final int YEAR_PREFIX = 0;

    private static final String DEFAULT_DATETIME_FORMAT = "yyyy-MM-dd HH:mm:ss";

    private static final String DEFAULT_DATE_FORMAT = "yyyy-MM-dd";
    
    
    // --------------------------------------------------------- Constructor

    /**
     * コンストラクタ（外部からのインスタンス生成を許可しない）
     */
    private DateUtils() {}

    /**
     * 文字列をDateオブジェクトに変換します。
     * @param date 日付文字列
     * @return 日付
     */
    public static Date stringToDate(String date) throws ParseException {
        return stringToDate(date, DEFAULT_DATE_FORMAT);
    }

    /**
     * 文字列をDateオブジェクトに変換します。
     * @param date 日付文字列
     * @return 日付
     */
    public static Date stringToDate(String date, String format)
            throws ParseException {
        if (date == null) {
            return null;
        }
        Date result = new SimpleDateFormat(format).parse(date);
        return new Date(result.getTime());
    }

    /**
     * 日付を文字列に変換します。
     * @param date 日付
     * @param format 書式
     * @return 日付文字列
     */
    public static String dateToString(Date date, String format) {
        SimpleDateFormat formater = new SimpleDateFormat(format);
        if (date == null) {
            return null;
        }
        return formater.format(date);
    }

    /**
     * 日付を文字列（yyyy-MM-dd HH:mm:ss形式）に変換します。
     * @param date 日付
     * @return 日付文字列（yyyy-MM-dd HH:mm:ss形式）
     */
    public static String dateToString(Date date) {
        return dateToString(date, DEFAULT_DATETIME_FORMAT);
    }

    /**
     * 日付配列を文字列配列に変換します。
     * @param date 日付配列
     * @param format 書式
     * @return 日付文字列配列
     */
    public static String[] dateToString(Date date[], String format) {
    	String[] result = new String[date.length];
    	for (int i = 0; i < date.length; i++) {
    		result[i] = dateToString(date[i], format);
    	}
    	return result;
    }
    
    //---------------------------------------------------------------------------------期間求めるメソッド
    /**
     * 日付の計算を行います。
     * @param calendares
     * @param format
     * @return
     * @throws ParseException
     */
    public static Calendar caculateDate(int[] calendares) {
        if (ArrayUtils.isEmpty(calendares) || (calendares.length % 2 != 0)) {
            throw new IllegalArgumentException();
        }
        Date day = new Date();
        Calendar current = Calendar.getInstance();
        current.setTime(day);
        for (int i = 0; i < calendares.length; i = i + 2) {
            current.add(calendares[i], calendares[i + 1]);
        }
        return current;
    }

    /**
     * 年数を計算したClendarオブジェクトを返します。
     * @param add
     * @return
     */
    private static Calendar caculateYear(int add) {
        Calendar current = caculateMonth(0);
        current.add(Calendar.YEAR, YEAR_PREFIX + add);
        return current;
    }

    /**
     * 年数を計算したClendarオブジェクトを返します。
     * @param add
     * @return
     */
    private static Calendar caculateYear(Date date, int add) {
        Calendar current = caculateMonth(date, 0);
        current.add(Calendar.YEAR, YEAR_PREFIX + add);
        return current;
    }

    /**
     * 月数を計算したCalendarオブジェクトを返します。
     * @param add
     * @return
     */
    private static Calendar caculateMonth(int add) {
        Calendar current = caculateDate(0);
        current.add(Calendar.MONTH, MONTH_PREFIX + add);
        return current;
    }

    /**
     * 月数を計算したCalendarオブジェクトを返します。
     * @param add
     * @return
     */
    private static Calendar caculateMonth(Date date, int add) {
        Calendar current = caculateDate(date, 0);
        current.add(Calendar.MONTH, MONTH_PREFIX + add);
        return current;
    }

    /**
     * 日数を計算したCalendarオブジェクトを返します。
     * @param add
     * @return
     */
    private static Calendar caculateDate(int add) {
        Date day = new Date();
        Calendar current = Calendar.getInstance();
        current.setTime(day);
        current.add(Calendar.DATE, DATE_PREFIX + add);
        return current;
    }

    /**
     * 日数を計算したCalendarオブジェクトを返します。
     * @param add
     * @return
     */
    private static Calendar caculateDate(Date date, int add) {
        Calendar current = Calendar.getInstance();
        current.setTime(date);
        current.add(Calendar.DATE, DATE_PREFIX + add);
        return current;
    }

    /**
     * 1日間隔期間を保持したDateオブジェクト配列を返します。
     * 前日～前日
     * @param format
     * @return
     */
    public static Date[] getDailyTerm() {
        Date day = caculateDate(0).getTime();
        return new Date[] { day, day };
    }

    /**
     * 1週間間隔期間を保持したDateオブジェクト配列を返します。
     * 前日～1週間前
     * @param format
     * @return
     */
    public static Date[] getWeeklyTerm() {
        return new Date[] { caculateDate(-6).getTime(), caculateDate(0).getTime() };
    }

    /**
     * 1ヶ月間隔期間を保持したDateオブジェクト配列を返します。
     * 前日から1ヶ月前
     * @param format
     * @return
     */
    public static Date[] getMonthlyTerm() {
        return new Date[] { caculateMonth(-1).getTime(), caculateMonth(0).getTime() };
    }

    /**
     * 1ヶ月間隔期間を保持したDateオブジェクト配列を返します。
     * 前日から1年前
     * @param format
     * @return
     */
    public static Date[] getYearlyTerm() {
        return new Date[] { caculateYear(-1).getTime(), caculateYear(0).getTime() };
    }

    /**
     * 1日間隔期間を保持したDateオブジェクト配列を返します。
     * 前日～前日
     * @param format
     * @return
     */
    public static Date[] getDailyTerm(Date date) {
        Date day = caculateDate(date, 0).getTime();
        return new Date[] { day, day };
    }

    /**
     * 1週間間隔期間を保持したDateオブジェクト配列を返します。
     * 前日～1週間前
     * @param format
     * @return
     */
    public static Date[] getWeeklyTerm(Date date) {
        return new Date[] { caculateDate(date, -6).getTime(), caculateDate(0).getTime() };
    }

    /**
     * 1ヶ月間隔期間を保持したDateオブジェクト配列を返します。
     * 前日から1ヶ月前
     * @param format
     * @return
     */
    public static Date[] getMonthlyTerm(Date date) {
        return new Date[] { caculateMonth(date, -1).getTime(), caculateMonth(0).getTime() };
    }

    /**
     * 1ヶ月間隔期間を保持したDateオブジェクト配列を返します。
     * 前日から1年前
     * @param format
     * @return
     */
    public static Date[] getYearlyTerm(Date date) {
        return new Date[] { caculateYear(date, -1).getTime(), caculateYear(0).getTime() };
    }
}
