/*
 * Java Template Project
 * 
 * Copyright (C) 2006 Satoshi Nagashiba, All Rights Reserved.
 */
package org.jtp.web.tag;

import java.io.IOException;

import javax.servlet.http.HttpServletResponse;
import javax.servlet.jsp.JspException;
import javax.servlet.jsp.JspWriter;
import javax.servlet.jsp.tagext.TagSupport;

import org.apache.struts.action.ActionMapping;
import org.apache.struts.config.FormBeanConfig;
import org.apache.struts.config.ModuleConfig;
import org.apache.struts.taglib.TagUtils;
import org.apache.struts.taglib.html.Constants;

/**
 * 基本カスタムタグクラス
 *
 * @author <a href="mailto:nagashiba.satoshi@opcybe.co.jp">Satoshi Nagashiba</a>
  * @version $Revision: 1.1 $ $Date: 2007/01/25 16:35:37 $
 * @since 2.0 (Oct,26,2006)
*/
public abstract class BaseTag extends TagSupport {

	/** Bean名 */
	private String name = Constants.BEAN_KEY;
	/** Beanフィールド名 */
	private String property = null;
	/** スタイル */
	private String styleClass = null;
	/** スコープ */
	private String scope = null;
	/** アクション */
	private String action = null;
	/** 画面遷移イベント */
	private String event = null;
	
	/**
	 * 開始タグを処理するメソッド
	 * @return ボディをJSPとして評価しない
	 * @throws JspException JSP例外
	 */
	public int doStartTag() throws JspException {
		//タグ生成設定を行う
		createTagInit();
		
		//JspWriterオブジェクトを取得する
		JspWriter jsp = super.pageContext.getOut();

		try {
			//生成したタグを出力する
			jsp.write(createTag());
		}
		catch (IOException e) {
			System.out.println(e.getMessage());
			throw new JspException(e);
		}
		return super.doStartTag();
	}
	
	/**
	 * 終了タグを処理するメソッド
	 * @return 終了タグ以降のJSPを評価する
	 * @throws JspException JSP例外
	 */
	public int doEndTag() throws JspException {
		return super.doEndTag();
	}

	/**
	 * Beanの取得
	 * @return Bean
	 * @throws JspException JSP例外
	 */
	protected Object getBean() throws JspException {
		return TagUtils.getInstance().lookup(pageContext, name, scope);
	}
	
	/**
	 * Beanフィールド値の取得
	 * @return Beanフィールド値
	 * @throws JspException JSP例外
	 */
	protected Object getBeanValue() throws JspException {
		return TagUtils.getInstance().lookup(pageContext, name, property, scope);
	}

	/**
	 * プロパティ指定でBeanフィールド値の取得
	 * @param propertyName プロパティ名
	 * @return Beanフィールド値
	 * @throws JspException JSP例外
	 */
	protected Object getBeanValue(String propertyName) throws JspException {
		return TagUtils.getInstance().lookup(pageContext, name, propertyName, scope);
	}
	
	/**
	 * アクションフォーム名取得
	 * @return アクションフォーム名
	 * @throws JspException JSP例外
	 */
    protected String getActionFormName() throws JspException {
    	String result = null;
    	ModuleConfig moduleConfig = TagUtils.getInstance().getModuleConfig(pageContext);
    	if (moduleConfig == null) {
            throw new JspException();
        }
        String mappingName = TagUtils.getInstance().getActionMappingName(action);
        ActionMapping mapping = (ActionMapping) moduleConfig.findActionConfig(mappingName);
        if (mapping == null) {
            throw new JspException();
        }
        FormBeanConfig formBeanConfig = moduleConfig.findFormBeanConfig(mapping.getName());
        if (formBeanConfig == null) {
            throw new JspException();
        }
        else {
        	result = mapping.getAttribute();
        }
        return result;
    }
    
    /**
     * アクションURLの取得
     * @return アクションURL
     */
    protected String getActionMappingURL() {
        HttpServletResponse response = (HttpServletResponse) pageContext.getResponse();
        return response.encodeURL(TagUtils.getInstance().getActionMappingURL(action, pageContext));
    }
    
	/**
	 * タグ生成の設定
	 * @throws JspException JSP例外
	 */
	protected void createTagInit() throws JspException {}
	
	/**
	 * タグ生成
	 * @return 生成したタグ
	 * @throws JspException JSP例外
	 */
	protected abstract String createTag() throws JspException;

	/**
	 * Bean名の取得
	 * @return Bean名
	 */
	public String getName() {
		return name;
	}
	
	/**
	 * Bean名の設定
	 * @param name Bean名
	 */
	public void setName(String name) {
		this.name = name;
	}
	
	/**
	 * Beanフィールド名の取得
	 * @return Beanフィールド名
	 */
	public String getProperty() {
		return property;
	}
	
	/**
	 * Beanフィールド名の設定
	 * @param property Beanフィールド名
	 */
	public void setProperty(String property) {
		this.property = property;
	}
	
	/**
	 * スタイルの取得
	 * @return スタイル
	 */
	public String getStyleClass() {
		return styleClass;
	}
	
	/**
	 * スタイルの設定
	 * @param styleClass スタイル
	 */
	public void setStyleClass(String styleClass) {
		this.styleClass = styleClass;
	}
	
	/**
	 * スコープの取得
	 * @return スコープ
	 */
	public String getScope() {
		return scope;
	}
	
	/**
	 * スコープの設定
	 * @param scope スコープ
	 */
	public void setScope(String scope) {
		this.scope = scope;
	}
	
	/**
	 * アクションの取得
	 * @return アクション
	 */
	public String getAction() {
		return action;
	}
	
	/**
	 * アクションの設定
	 * @param action アクション
	 */
	public void setAction(String action) {
		this.action = action;
	}
	
	/**
	 * 画面遷移イベントの取得
	 * @return 画面遷移イベント
	 */
	public String getEvent() {
		return event;
	}
	
	/**
	 * 画面遷移イベントの設定
	 * @param event 画面遷移イベント
	 */
	public void setEvent(String event) {
		this.event = event;
	}
}
