package junkutil.android;

import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.List;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;
import junkutil.common.DataDumper;
import junkutil.common.FileUtil;
import junkutil.set.LzList;

/**
 * FileUtileXgB
 * @author Hiroyuki Shiota
 */
public class AndroidFileUtilTest extends TestCase {
	public static String currentDir = "testdata/";
	//public static String currentDir = Environment.getExternalStorageDirectory().getPath() + "/JunkUtil/";
	static {
		File current = new File(currentDir);
		if (!current.exists()) {
			current.mkdir();
		}
	}
	/**
	 * RXgN^
	 * @param name
	 */
	public AndroidFileUtilTest(String name) {
		super(name);
	}

	/**
	 * @return
	 */
	public static Test suite() {
		TestSuite test = new TestSuite("FileUtilTest");
		test.addTest(new TestSuite(AndroidFileUtilTest.class));
		return test;
	}

	/**
	 * @see junit.framework.TestCase#setUp()
	 */
	protected void setUp() throws Exception {
		FileUtil.writeFile(currentDir+"readFile.txt", "item1\r\nitem2\r\nitem3\r\nACe1\r\nACe2\r\nACe3");
		FileUtil.writeByteArray(currentDir+"byteArray.txt", "abcdABCDeXgeXgeXgeXg".getBytes("UTF-8"));
	}

	/**
	 * @see junit.framework.TestCase#tearDown()
	 */
	protected void tearDown() throws Exception {
	}

	/**
	 * @throws Excepiton
	 */
	public void testExists() throws Exception {
		assertTrue(FileUtil.exists(currentDir+"readFile.txt"));
		assertTrue(!FileUtil.exists(currentDir+"readFile2.txt"));
	}
	
	/**
	 * @throws Exception
	 */
	public void testReadFile() throws Exception {
		String read = FileUtil.readFile(currentDir+"readFile.txt", "Shift_JIS");
		read.equals(("item1\r\nitem2\r\nitem3\r\nACe1\r\nACe2\r\nACe3"));
	}
	
	/**
	 * @throws Exception
	 */
	public void testWriteFile() throws Exception {
		FileUtil.writeFile(currentDir+"writeFile.txt", "item1\r\nitem2\r\nitem3\r\nACe1\r\nACe2\r\nACe3", "Shift_JIS");
		String read = FileUtil.readFile(currentDir+"readFile.txt", "Shift_JIS");
		assertTrue(read.equals("item1\r\nitem2\r\nitem3\r\nACe1\r\nACe2\r\nACe3"));
	}
	
	/**
	 * @throws Exception
	 */
	public void testAppendLines() throws Exception {
		new File(currentDir+"appendLines.txt").delete();
		List strs = Arrays.asList(new String[]{"item1","item2","item3","ACe1","ACe2","ACe3"});
		FileUtil.appendLines(currentDir+"appendLines.txt", strs, "Shift_JIS");
		FileUtil.appendLines(currentDir+"appendLines.txt", strs, "Shift_JIS");
		String read = FileUtil.readFile(currentDir+"appendLines.txt", "Shift_JIS");
		assertTrue(read.equals(
				 "item1\r\nitem2\r\nitem3\r\nACe1\r\nACe2\r\nACe3\r\n"
				+"item1\r\nitem2\r\nitem3\r\nACe1\r\nACe2\r\nACe3"));
	}

	/**
	 * @throws Exception
	 */
	public void testAppendFile() throws Exception {
		FileUtil.writeFile(currentDir+"appendFile.txt", "item1\r\nitem2\r\nitem3\r\nACe1\r\nACe2\r\nACe3\r\n", "Shift_JIS");
		FileUtil.appendFile(currentDir+"appendFile.txt", "item1\r\nitem2\r\nitem3\r\nACe1\r\nACe2\r\nACe3", "Shift_JIS");
		String read = FileUtil.readFile(currentDir+"appendFile.txt", "Shift_JIS");
		assertTrue(read.equals(
				 "item1\r\nitem2\r\nitem3\r\nACe1\r\nACe2\r\nACe3\r\n"
				+"item1\r\nitem2\r\nitem3\r\nACe1\r\nACe2\r\nACe3"));
	}

	/**
	 * @throws Exception
	 */
	public void testReadLines() throws Exception {
		List strs = Arrays.asList(new String[]{"item1","item2","item3","ACe1","ACe2","ACe3"});
		List lines2 = FileUtil.readLines(currentDir+"readFile.txt", "Shift_JIS");
		assertTrue(new LzList(strs).equals(lines2));
	}
	
	/**
	 * @throws Exception
	 */
	public void testWriteLines() throws Exception {
		List strs = Arrays.asList(new String[]{"item1","item2","item3","ACe1","ACe2","ACe3"});
		FileUtil.writeLines(currentDir+"writeLines.txt", strs);
		assertTrue(new LzList(strs).equals(FileUtil.readLines(currentDir+"writeFile.txt")));
		FileUtil.writeLines(currentDir+"writeUTF-8.txt", strs, "UTF-8");
		assertTrue(new LzList(strs).equals(FileUtil.readLines(currentDir+"writeUTF-8.txt", "UTF-8")));
	}
	
	/**
	 * @throws Exception
	 */
	public void testCopy() throws Exception {
		FileUtil.copy(currentDir+"readFile.txt", currentDir+"copyFile.txt");
		assertTrue(true);
		try {
			FileUtil.copy(currentDir+"nothingFile.txt", currentDir+"copyFile.txt");
			assertTrue(false);
		} catch (IOException e) {
			assertTrue(true);
		}
	}
	
	/**
	 * @throws Exception
	 */
	public void testReadByteArray() throws Exception {
		byte[] byteArray = FileUtil.readByteArray(currentDir+"byteArray.txt", 4096);
		DataDumper.dumpByteArray(System.out, byteArray);
		assertTrue(true);
	}
	
	/**
	 * @throws Exception
	 */
	public void testWriteByteArray() throws Exception {
		String str =
			"abcdABCDeXgeXgeXgeXg"+FileUtil.ln+
			"abcdABCDeXgeXgeXgeXg"+FileUtil.ln+
			"abcdABCDeXgeXgeXgeXg"+FileUtil.ln+
			"abcdABCDeXgeXgeXgeXg"+FileUtil.ln+
			"abcdABCDeXgeXgeXgeXg"+FileUtil.ln;
		FileUtil.writeByteArray(currentDir+"byteArray.txt", str.getBytes("UTF-8"));
		String str2 = new String(FileUtil.readByteArray(currentDir+"byteArray.txt", 1024), "UTF-8");
		assertTrue(str.equals(str2));
	}
}
