package junkutil.common;

import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.List;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;
import junkutil.common.FileUtil;
import junkutil.set.LzList;

/**
 * FileUtileXgB
 * @author Hiroyuki Shiota
 */
public class FileUtilTest extends TestCase {
	/**
	 * RXgN^
	 * @param name
	 */
	public FileUtilTest(String name) {
		super(name);
	}

	/**
	 * @return
	 */
	public static Test suite() {
		TestSuite test = new TestSuite("FileUtilTest");
		test.addTest(new TestSuite(FileUtilTest.class));
		return test;
	}

	/**
	 * @see junit.framework.TestCase#setUp()
	 */
	protected void setUp() throws Exception {
		FileUtil.writeFile("testdata/readFile.txt", "item1\r\nitem2\r\nitem3\r\nACe1\r\nACe2\r\nACe3");
		FileUtil.writeByteArray("testdata/byteArray.txt", "abcdABCDeXgeXgeXgeXg".getBytes("UTF-8"));
	}

	/**
	 * @see junit.framework.TestCase#tearDown()
	 */
	protected void tearDown() throws Exception {
	}

	/**
	 * @throws Excepiton
	 */
	public void testExists() throws Exception {
		assertTrue(FileUtil.exists("testdata/readFile.txt"));
		assertTrue(!FileUtil.exists("testdata/readFile2.txt"));
	}

	/**
	 * @throws Exception
	 */
	public void testReadFile() throws Exception {
		String read = FileUtil.readFile("testdata/readFile.txt", "Shift_JIS");
		read.equals(("item1\r\nitem2\r\nitem3\r\nACe1\r\nACe2\r\nACe3"));
	}

	/**
	 * @throws Exception
	 */
	public void testWriteFile() throws Exception {
		FileUtil.writeFile("testdata/writeFile.txt", "item1\r\nitem2\r\nitem3\r\nACe1\r\nACe2\r\nACe3", "Shift_JIS");
		String read = FileUtil.readFile("testdata/readFile.txt", "Shift_JIS");
		assertTrue(read.equals("item1\r\nitem2\r\nitem3\r\nACe1\r\nACe2\r\nACe3"));
	}

	/**
	 * @throws Exception
	 */
	public void testAppendLines() throws Exception {
		new File("testdata/appendLines.txt").delete();
		List strs = Arrays.asList(new String[]{"item1","item2","item3","ACe1","ACe2","ACe3"});
		FileUtil.appendLines("testdata/appendLines.txt", strs, "Shift_JIS");
		FileUtil.appendLines("testdata/appendLines.txt", strs, "Shift_JIS");
		String read = FileUtil.readFile("testdata/appendLines.txt", "Shift_JIS");
		assertTrue(read.equals(
				 "item1\r\nitem2\r\nitem3\r\nACe1\r\nACe2\r\nACe3\r\n"
				+"item1\r\nitem2\r\nitem3\r\nACe1\r\nACe2\r\nACe3"));
	}

	/**
	 * @throws Exception
	 */
	public void testAppendFile() throws Exception {
		FileUtil.writeFile("testdata/appendFile.txt", "item1\r\nitem2\r\nitem3\r\nACe1\r\nACe2\r\nACe3\r\n", "Shift_JIS");
		FileUtil.appendFile("testdata/appendFile.txt", "item1\r\nitem2\r\nitem3\r\nACe1\r\nACe2\r\nACe3", "Shift_JIS");
		String read = FileUtil.readFile("testdata/appendFile.txt", "Shift_JIS");
		assertTrue(read.equals(
				 "item1\r\nitem2\r\nitem3\r\nACe1\r\nACe2\r\nACe3\r\n"
				+"item1\r\nitem2\r\nitem3\r\nACe1\r\nACe2\r\nACe3"));
	}

	/**
	 * @throws Exception
	 */
	public void testReadLines() throws Exception {
		List strs = Arrays.asList(new String[]{"item1","item2","item3","ACe1","ACe2","ACe3"});
		List lines2 = FileUtil.readLines("testdata/readFile.txt", "Shift_JIS");
		assertTrue(new LzList(strs).equals(lines2));
	}

	/**
	 * @throws Exception
	 */
	public void testWriteLines() throws Exception {
		List strs = Arrays.asList(new String[]{"item1","item2","item3","ACe1","ACe2","ACe3"});
		FileUtil.writeLines("testdata/writeLines.txt", strs);
		assertTrue(new LzList(strs).equals(FileUtil.readLines("testdata/writeFile.txt")));
		FileUtil.writeLines("testdata/writeUTF-8.txt", strs, "UTF-8");
		assertTrue(new LzList(strs).equals(FileUtil.readLines("testdata/writeUTF-8.txt", "UTF-8")));
	}

	/**
	 * @throws Exception
	 */
	public void testCopy() throws Exception {
		FileUtil.copy("testdata/readFile.txt", "testdata/copyFile.txt");
		assertTrue(true);
		try {
			FileUtil.copy("testdata/nothingFile.txt", "testdata/copyFile.txt");
			assertTrue(false);
		} catch (IOException e) {
			assertTrue(true);
		}
	}

	/**
	 * @throws Exception
	 */
	public void testReadByteArray() throws Exception {
		byte[] byteArray = FileUtil.readByteArray("testdata/byteArray.txt", 4096);
		DataDumper.dumpByteArray(System.out, byteArray);
		assertTrue(true);
	}

	/**
	 * @throws Exception
	 */
	public void testWriteByteArray() throws Exception {
		String str =
			"abcdABCDeXgeXgeXgeXg";
		FileUtil.writeByteArray("testdata/byteArray.txt", str.getBytes("UTF-8"));
		String str2 = new String(FileUtil.readByteArray("testdata/byteArray.txt", 1024), "UTF-8");
		assertTrue(str.equals(str2));
	}
}
