// JVMStarter.cpp : R\[ AvP[ṼGg |Cg`܂B
//

#include "stdafx.h"

#include <assert.h>
#include <process.h>

#include <exception>
#include <fstream>

#include "message.h"

#include "ConfigParser.hpp"
#include "JavaVMMainThread.hpp"
#include "ExceptionFilter.hpp"
#include "EventLog.hpp"

namespace
{
	EventLog* pEventLog_ = NULL;

	FILE* pJNIOut_ = NULL;

	jint __stdcall onVfprintfJVM(FILE *fp, const char *format, va_list args)
	{
		return vfprintf( ( pJNIOut_ == NULL ) ? stdout : pJNIOut_ , format, args );
	}

	void __stdcall onStartedJVM()
	{
		assert( pEventLog_ != NULL );
		pEventLog_->reportEvent(
			MSG_JAVAVM_STARTED,
			0,
			NULL,
			EVENTLOG_INFORMATION_TYPE
			);
	}

	void __stdcall onNormalExitJVM()
	{
		assert( pEventLog_ != NULL );
		const char* mes[] = { "unspecified" };
		pEventLog_->reportEvent(
			MSG_JAVAVM_EXITED,
			1,
			mes,
			EVENTLOG_INFORMATION_TYPE
			);
		exit( 0 );
	}

	void __stdcall onExitJVM( jint v_exitCode )
	{
		assert( pEventLog_ != NULL );
		CHAR buf[ 64 ];
		wsprintf( buf, "%d", v_exitCode );
		const char* mes[] = { buf };
		pEventLog_->reportEvent(
			MSG_JAVAVM_EXITED,
			1,
			mes,
			EVENTLOG_INFORMATION_TYPE
			);
		exit( v_exitCode );
	}

	void __stdcall onAbortJVM()
	{
		assert( pEventLog_ != NULL );
		pEventLog_->reportEvent(
			MSG_JAVAVM_ABORTED,
			0,
			NULL,
			EVENTLOG_ERROR_TYPE
			);
		abort();
	}

	void __stdcall onNotifyExceptionJVM( const char* v_message )
	{
		assert( pEventLog_ != NULL );
		const char* mes[] = { v_message };
		pEventLog_->reportEvent(
			MSG_JAVAVM_EXCEPTION,
			( v_message != NULL ) ? 1 : 0,
			mes,
			EVENTLOG_ERROR_TYPE
			);
	}

	void __stdcall onNotifyPreloadLibFailure( const char* v_dllPath )
	{
		assert( pEventLog_ != NULL );
		const char* mes[] = { v_dllPath };
		pEventLog_->reportEvent(
			MSG_PRELOADLIB_FAILURE,
			( v_dllPath != NULL ) ? 1 : 0,
			mes,
			EVENTLOG_ERROR_TYPE
			);
	}
}

unsigned __stdcall terminaterThreadProc( void* v_pParam )
{
	UnhandledExceptionHandler::setUnhandledExceptionHandler();

	JavaVMMainThread* pJVMThread = (JavaVMMainThread*) v_pParam;
	assert( pJVMThread != NULL );

	Sleep( 10 * 1000 );
	pJVMThread->stopInterrupt();

	return 0;
}

void preloadLibrary()
{
	CHAR confPath[ MAX_PATH + 9 ];
	confPath[ GetModuleFileName( NULL, confPath, MAX_PATH ) ] = 0;
	LPSTR p = confPath;
	while(*p)p++;
	while( p > confPath ) {
		if( *p == '.' ) {
			strcpy( p, ".preload" );
		}
		p = CharPrev( confPath, p );
	}
	
	std::ifstream fin;
	fin.open( confPath );
	if( ! fin.is_open() ) {
		return;
	}
	
	std::string line;
	while( getline( fin, line ) ) {
		CHAR dllPath[ MAX_PATH ];
		ExpandEnvironmentStrings( line.c_str(), dllPath, MAX_PATH );
		LPCSTR p = dllPath;
		while(*p) {
			if( *p == '#' || *p == ';' ) {
				break;
			}
			if( ! (*p > 0 && *p <= ' ' ) ) {
				if( LoadLibrary( p ) == NULL ) {
					throw std::runtime_error( dllPath );
				}
				break;
			}
			p = CharNext( p );
		}
	}
	fin.close();
}

int main(int argc, char* argv[])
{
	UnhandledExceptionHandler::setUnhandledExceptionHandler();

	EventLog eventLog( "JVMStarter" );
	eventLog.registerEventSource();
	eventLog.open();
	pEventLog_ = &eventLog;

	UnhandledExceptionHandler::setEventLog( &eventLog );

	try{
		preloadLibrary();
	}
	catch( const std::exception& v_exception ) {
		onNotifyPreloadLibFailure( v_exception.what() );
		return 0;
	}

	JavaVMCallback callback;
	callback.onJVMAbort = onAbortJVM;
	callback.onJVMExit = onExitJVM;
	callback.onJVMNormalExit = onNormalExitJVM;
	callback.onJVMVfprintf = onVfprintfJVM;
	callback.onJVMStarted = onStartedJVM;
	callback.onJVMNotifyException = onNotifyExceptionJVM;

	try{
		ConfigParser parser;
		parser.parse( "jvmstarter.ini" );

		JavaVMMainThread jvm( callback, parser.getJavaVMStartupInfo() );
		jvm.start();

		unsigned tid = 0;
		HANDLE hThread = (HANDLE) _beginthreadex( NULL, 0, terminaterThreadProc, &jvm, 0, &tid );
		CloseHandle( hThread );

		const ConfigParser::size_type mx = parser.size();
		ConfigParser::size_type idx = 1;
		for( ConfigParser::const_iterator p=parser.begin(), last=parser.end();
			p != last;
			++p, ++idx)
		{
			const StartupInfo& startupInfo = *p;
			jvm.invokeClassMethod( startupInfo, ( idx == mx ) || ! startupInfo.isFork() );
		}
		jvm.join();
	}
	catch( const std::exception& v_exception ) {
		onNotifyExceptionJVM( v_exception.what() );
	}

	return 0;
}
