#include "stdafx.h"

#include "ServiceControlManager.hpp"
#include "EventLog.hpp"

#include <algorithm>
#include <iterator>

#include <assert.h>
#include <string.h>

////

ServiceIdentifier::ServiceIdentifier( const std::string& v_serviceName ) throw( std::exception )
	: serviceName_( v_serviceName )
{
	if( v_serviceName.empty() ) {
		throw std::invalid_argument( "T[rXɋ󕶎͎wł܂B" );
	}
	for( std::string::const_iterator p = v_serviceName.begin(), last = v_serviceName.end();
		p != last;
		++p )
	{
		const char c = *p;
		if( ( c > 0 && c <= ' ' ) || c == '/' || c == '\\' || c == '\"' ) {
			throw std::invalid_argument( "T[rXɎgȂ܂܂Ă܂B(󔒁A/A\\A\")" );
		}
	}
}

ServiceIdentifier::ServiceIdentifier( const ServiceIdentifier& v_other ) throw()
	: serviceName_( v_other.serviceName_ )
{
}

ServiceIdentifier& ServiceIdentifier::operator=( const ServiceIdentifier& v_other ) throw()
{
	if( this != &v_other ) {
		serviceName_ = v_other.serviceName_;
	}
	return *this;
}

const std::string& ServiceIdentifier::getServiceName() const throw()
{
	return serviceName_;
}

////

ServiceRegistParam::ServiceRegistParam( const std::string& v_serviceName, const std::string& v_displayName, const std::string& v_command ) throw ( std::exception )
	: ServiceIdentifier( v_serviceName )
	, displayName_( v_displayName.empty() ? v_serviceName : v_displayName )
	, command_( v_command )
{
}

ServiceRegistParam::ServiceRegistParam( const ServiceRegistParam& v_other ) throw()
	: ServiceIdentifier( v_other )
	, dependencies_( v_other.dependencies_ )
	, displayName_( v_other.displayName_ )
	, command_( v_other.command_ )
{
}

ServiceRegistParam& ServiceRegistParam::operator=( const ServiceRegistParam& v_other ) throw()
{
	if( this != &v_other ) {
		ServiceIdentifier::operator=( v_other );
		dependencies_ = v_other.dependencies_;
		displayName_ = v_other.displayName_;
		command_ = v_other.command_;
	}
	return *this;
}

const std::string& ServiceRegistParam::getDisplayName() const throw()
{
	return displayName_;
}

const std::string& ServiceRegistParam::getCommand() const throw()
{
	return command_;
}

const ServiceRegistParam::StringVector& ServiceRegistParam::dependencies() const throw()
{
	return dependencies_;
}

ServiceRegistParam::StringVector& ServiceRegistParam::dependencies() throw()
{
	return dependencies_;
}

////

ServiceControlManager::ServiceHandleHolder::ServiceHandleHolderData_::~ServiceHandleHolderData_()
{
	if( hService_ != NULL ) {
		CloseServiceHandle( hService_ );
	}
}


ServiceControlManager::ServiceHandleHolder::ServiceHandleHolder( SC_HANDLE v_hService ) throw()
	: pData_( NULL )
{
	pData_ = new ServiceHandleHolderData_();
	pData_->hService_ = v_hService;
	pData_->ref_ = 1;
}

ServiceControlManager::ServiceHandleHolder::ServiceHandleHolder( const ServiceHandleHolder& v_other ) throw()
{
	pData_ = v_other.pData_;
	pData_->ref_++;
}

ServiceControlManager::ServiceHandleHolder::~ServiceHandleHolder() throw()
{
	if( --pData_->ref_ == 0 ) {
		delete pData_;
	}
}

ServiceControlManager::ServiceHandleHolder& ServiceControlManager::ServiceHandleHolder::operator=( const ServiceHandleHolder& v_other ) throw()
{
	if( this != &v_other ) {
		if( --pData_->ref_ == 0 ) {
			delete pData_;
		}
		pData_ = v_other.pData_;
		pData_->ref_++;
	}
	return *this;
}

ServiceControlManager::ServiceHandleHolder::operator SC_HANDLE() const throw()
{
	return pData_->hService_;
}

SC_HANDLE ServiceControlManager::ServiceHandleHolder::get() const throw()
{
	return pData_->hService_;
}

////

ServiceControlManager::ServiceControlManager() throw( std::exception )
	: hServiceControlManager_( NULL )
{
	hServiceControlManager_ = OpenSCManager( NULL, SERVICES_ACTIVE_DATABASE, SC_MANAGER_ALL_ACCESS );
	Win32Exception::checkAndThrow( hServiceControlManager_ != NULL );
}

ServiceControlManager::~ServiceControlManager() throw()
{
	CloseServiceHandle( hServiceControlManager_ );
}

void ServiceControlManager::allocDependenciesString( const std::vector<std::string> v_dependencies, std::vector<char>& v_buffer )
{
	std::back_insert_iterator<std::vector<char> > d( std::back_inserter( v_buffer ) );

	for( std::vector<std::string>::const_iterator p = v_dependencies.begin(), last = v_dependencies.end();
		p != last;
		++p )
	{
		const std::string dependency = *p;
		std::copy( dependency.begin(), dependency.end(), d );
		*d++ = 0;
	}
	*d++ = 0;

	if( v_dependencies.empty() ) {
		*d++ = 0;
	}
}

void ServiceControlManager::createOrReplaceService( const ServiceRegistParam& v_serviceParam ) throw ( std::exception )
{
	std::vector<char> dependencies;
	allocDependenciesString( v_serviceParam.dependencies(), dependencies );

	const std::string exeModulePath( getOwnModuleFileName() );
	const std::string serviceLaunchCommand( "\"" + exeModulePath + "\" " + v_serviceParam.getCommand() );

	ServiceHandleHolder service_handle( CreateService(
		hServiceControlManager_,
		v_serviceParam.getServiceName().c_str(),
		v_serviceParam.getDisplayName().c_str(),
		SERVICE_ALL_ACCESS,
		SERVICE_WIN32_OWN_PROCESS,
		SERVICE_DEMAND_START, // 蓮Jn
		SERVICE_ERROR_NORMAL,
		serviceLaunchCommand.c_str(),
		NULL,
		NULL,
		&dependencies[ 0 ],
		NULL, // local system account
		NULL
		) );
	if( service_handle.get() == NULL ) {
		const DWORD err = GetLastError();
		if( err != ERROR_SERVICE_EXISTS ) {
			Win32Exception::throwWin32Exception( err );
		}
		updateService( v_serviceParam );
	}
	else {
		EventLog eventLog( v_serviceParam.getServiceName() );
		eventLog.registerEventSource( exeModulePath );
	}
}

void ServiceControlManager::updateService( const ServiceRegistParam& v_serviceParam ) throw ( std::exception )
{
	ServiceHandleHolder service_handle( OpenService(
		hServiceControlManager_,
		v_serviceParam.getServiceName().c_str(),
		SERVICE_ALL_ACCESS
		) );
	Win32Exception::checkAndThrow( service_handle.get() != NULL );

	checkOwnService( service_handle );

	std::vector<char> dependencies;
	allocDependenciesString( v_serviceParam.dependencies(), dependencies );

	const std::string exeModulePath( getOwnModuleFileName() );
	const std::string serviceLaunchCommand( "\"" + exeModulePath + "\" " + v_serviceParam.getCommand() );

	if( ! ChangeServiceConfig(
			service_handle.get(),
			SERVICE_NO_CHANGE,
			SERVICE_NO_CHANGE,
			SERVICE_NO_CHANGE,
			serviceLaunchCommand.c_str(),
			NULL,
			NULL,
			&dependencies[ 0 ],
			NULL,
			NULL,
			v_serviceParam.getDisplayName().c_str()
			) )
	{
		Win32Exception::throwWin32Exception( GetLastError() );
	}

	EventLog eventLog( v_serviceParam.getServiceName() );
	eventLog.registerEventSource( exeModulePath );
}

void ServiceControlManager::deleteService( const ServiceIdentifier& v_serviceIdentifier ) throw ( std::exception )
{
	ServiceHandleHolder service_handle( OpenService(
		hServiceControlManager_,
		v_serviceIdentifier.getServiceName().c_str(),
		SERVICE_ALL_ACCESS
		) );
	Win32Exception::checkAndThrow( service_handle.get() != NULL );

	checkOwnService( service_handle );

	if( ! DeleteService( service_handle.get() ) ) {
		const DWORD errorCode = GetLastError();
		if( errorCode != ERROR_SERVICE_MARKED_FOR_DELETE ) {
			Win32Exception::throwWin32Exception( errorCode );
		}
	}
	EventLog eventLog( v_serviceIdentifier.getServiceName() );
	eventLog.unregisterEventSource( getOwnModuleFileName() );
}


void ServiceControlManager::checkOwnService( const ServiceHandleHolder& v_service_handle ) throw ( std::exception )
{
	//required:
	assert( v_service_handle.get() != NULL );

	//do:
	size_t size = sizeof( QUERY_SERVICE_CONFIG ) + ( MAX_PATH * 2 );
	std::vector<char> queryBuf( size );
	for(;;) {
		DWORD needlen = 0;
		if( QueryServiceConfig(
			v_service_handle.get(),
			reinterpret_cast<LPQUERY_SERVICE_CONFIG>( &queryBuf[ 0 ] ),
			(DWORD) size,
			&needlen
			) == 0 )
		{
			const DWORD errorCode = GetLastError();
			if( errorCode == ERROR_INSUFFICIENT_BUFFER ) {
				size = needlen + 1;
				queryBuf.resize( size );
				continue;
			}
			Win32Exception::throwWin32Exception( errorCode );
		}
		break;
	}
	const QUERY_SERVICE_CONFIG& currentServiceConf = *reinterpret_cast<LPQUERY_SERVICE_CONFIG>( &queryBuf[ 0 ] );
	
	// ŏ̈؂o
	LPCSTR st = currentServiceConf.lpBinaryPathName;
	LPCSTR en = NULL;
	while( *st > 0 && *st <= ' ' ) {
		st++;
	}
	if( *st == '\"' ) {
		st++;
		en = st;
		while( *en ) {
			if( *en == '\"' ) {
				break;
			}
			en = CharNext( en );
		}
	}
	else {
		en = st;
		while( *en ) {
			if( *en > 0 && *en <= ' ' ) {
				break;
			}
			en = CharNext( en );
		}
	}
	if( en < st ) {
		en = st;
	}
	std::string exeModuleName( st, en );
	std::string ownModuleName( getOwnModuleFileName() );

	// v邩?
	if( stricmp( exeModuleName.c_str(), ownModuleName.c_str() ) != 0 ) {
		throw std::runtime_error( "w肳ꂽT[rX́A̎svOɂēo^ꂽ̂ł͂܂B" );
	}

}

std::string ServiceControlManager::getOwnModuleFileName() throw ( std::exception )
{
	CHAR szModuleFileName[ MAX_PATH ];
	szModuleFileName[ GetModuleFileName( NULL, szModuleFileName, MAX_PATH ) ] = 0;
	Win32Exception::checkAndThrow( szModuleFileName[ 0 ] != 0 );
	return std::string( szModuleFileName );
}
