// $Id: JGDevObject.cpp,v 1.5 2003/03/16 14:51:00 fukasawa Exp $

//=============================================================================
/**
 *  @file    JGDevObject.cpp
 *
 *  @author  Fukasawa Mitsuo
 *
 *
 *    Copyright (C) 2001-2003 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================

#define BEE_BUILD_DLL

#include "JGDevObject.h"
#include "JGDevType.h"
#include "JGDevAttribute.h"
#include "JGDevAttrData.h"

//-----------------------------------------------------------------------------
// Constructor/Destructor
//-----------------------------------------------------------------------------
JGDevObject::JGDevObject(JGDevType * objtype, ULONG id)
        : b_object(objtype, NULL)
{
    TRACE_FUNCTION(TRL_CONSTRUCT, "JGDevObject::JGDevObject");
    m_seqid = id;
    ACE_Time_Value systv = ACE_OS::gettimeofday();  // Get system time
    m_logtime = systv;
}

//-----------------------------------------------------------------------------
JGDevObject::JGDevObject(const JGDevObject& rhs) : b_object(rhs)
{
    TRACE_FUNCTION(TRL_CONSTRUCT, "JGDevObject::JGDevObject");
    this->copy(rhs);
    return ;
}

//-----------------------------------------------------------------------------
// Copy
//-----------------------------------------------------------------------------
JGDevObject& JGDevObject::operator=(const JGDevObject& rhs)
{
    TRACE_FUNCTION(TRL_CONSTRUCT, "JGDevObject::operator=");
    if (this == &rhs)
        return *this;
    this->copy(rhs);
    return *this;
}

//-----------------------------------------------------------------------------
void JGDevObject::copy(const JGDevObject& rhs)
{
    TRACE_FUNCTION(TRL_CONSTRUCT, "JGDevObject::copy");
    this->b_object::copy(reinterpret_cast<const b_object&>(rhs));
    m_seqid = rhs.m_seqid;
    m_logtime = rhs.m_logtime;
}

//-----------------------------------------------------------------------------
// Read log data from plc register
//-----------------------------------------------------------------------------
int JGDevObject::read(JGLogTime * timestamp)
{
    // read
    for (size_t i = 0; i < m_values.size(); i++)
    {
        JGvalue value;
        JGDevAttrData * attrdata;
        const JGDevAttribute * attr;

        attrdata = reinterpret_cast<JGDevAttrData *>(m_values[i]);
        attr = reinterpret_cast<const JGDevAttribute *>(attrdata->attribute());
        attr->getValue(value);
        attrdata->set(value);
    }

    if (timestamp != NULL)
    {
        m_logtime = *timestamp;
    }
    return 0;
}

//-----------------------------------------------------------------------------
// Compare attribute data
//-----------------------------------------------------------------------------
bool JGDevObject::equal(const JGDevObject& rhs)
{
    TRACE_FUNCTION(TRL_CONSTRUCT, "JGDevObject::equal");
    BEEAssert(m_objtype == rhs.m_objtype);

    bool result = true;
    for (size_t i = 0; i < rhs.m_values.size(); i++)
    {
        b_attrdata * srcdata = rhs.m_values[i];
        b_attrdata * destdata = m_values[i];
        const JGDevAttribute * attr =
                    reinterpret_cast<JGDevAttrData *>(srcdata)->getAttribute();
        if (attr->isCompare())
        {
            if (! (*srcdata == *destdata))
            {
                result = false;
            }
        }
    }
    return result;
}


//-----------------------------------------------------------------------------
// Dump to xml string
//-----------------------------------------------------------------------------
string JGDevObject::toXml()
{
    string dumpstr = _TX("<JGDevObject>\n");

    // Setting log header
    BCHAR buf[512];
    _stprintf(buf, _TX("%s.%u"), this->getObjType()->typeName(),
                                 (u_int)m_seqid);
    dumpstr += _TX("  <seqid>") + string(buf) + _TX("</seqid>\n");
    dumpstr += _TX("  <timestamp>") + m_logtime.toString();
    dumpstr += _TX("</timestamp>\n");

    dumpstr += _TX("  <members>\n");
    // Setting log data
    JGDevAttrData * attrdata;
    for (size_t at = 0; at < m_values.size(); at++)
    {
        attrdata = reinterpret_cast<JGDevAttrData *>(m_values[at]);
        dumpstr += _TX("    <member name=\"") + attrdata->name() + _TX("\"/>");
        JGvalue& value = attrdata->get();
        dumpstr += value.toString();
        dumpstr += _TX("</member>\n");

    }
    dumpstr += _TX("  </members>\n");
    dumpstr += _TX("</JGDevObject>\n");

    return dumpstr;
}

//-----------------------------------------------------------------------------
// Dump to string
//-----------------------------------------------------------------------------
string JGDevObject::toString()
{
    string dumpstr = m_logtime.toString() + _TX(" ");

    // Setting log header
    BCHAR buf[512];
    _stprintf(buf, _TX("%s.%u"), this->getObjType()->typeName(),
                                 (u_int)m_seqid);
    dumpstr += string(buf) + _TX(" { ");

    // Setting log data
    JGDevAttrData * attrdata;
    for (size_t at = 0; at < m_values.size(); at++)
    {
        if (at > 0)
        {
            dumpstr += _TX(", ");
        }
        attrdata = reinterpret_cast<JGDevAttrData *>(m_values[at]);
        JGvalue& value = attrdata->get();
        if (value.m_t == ATOM_ASCII)
        {
            dumpstr += _TX("\"") + value.toString() + _TX("\"");
        }
        else
        {
            dumpstr += value.toString();
        }
    }
    dumpstr += _TX(" }\n");

    return dumpstr;
}

//-----------------------------------------------------------------------------
// Dump
//-----------------------------------------------------------------------------
void JGDevObject::dump() const
{
    ACE_DEBUG((LM_DEBUG, ACE_TEXT("*** JGDevObject(%u): %s [%s] ***\n"),
                         m_seqid, this->getObjType()->typeName(),
                         const_cast<JGLogTime&>(m_logtime).toString().c_str()));
    this->b_object::dump();
}

