// $Id: reportCmd.cpp,v 1.3 2003/03/16 14:51:02 fukasawa Exp $

//=============================================================================
/**
 *  @file    reportCmd.cpp
 *
 *  @author  Fukasawa Mitsuo
 *
 *
 *    Copyright (C) 2002 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================

#define BEE_BUILD_DLL

#ifdef _MSC_VER
#pragma warning(disable: 4786)  // too long identifier
#endif

#include "JGEquipment.h"
#include "JGInfoManager.h"
#include "JGEventManager.h"
#include "tclbee.h"

#ifdef _MSC_VER
#include <windows.h>
#endif

// function prototype
static int dump_reports(Tcl_Interp * interp);
static JGReport * find_report(Tcl_Interp * interp, char * rptidStr);
static int parse_vids(Tcl_Interp * interp, char * idsStr, vector<BS2id>& ids);
static int set_variables(Tcl_Interp * interp, JGReport * report,
                         char * idsStr);
static int add_variables(Tcl_Interp * interp, JGReport * report,
                         char * idsStr);
static int del_variables(Tcl_Interp * interp, JGReport * report,
                         char * idsStr);

/*
 *----------------------------------------------------------------------
 *
 * Gem_reportCmd --
 *
 *      This procedure is invoked to process the "gem::report"
 *      command. See the user documentation for details on what it does.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *----------------------------------------------------------------------
 */

    /* ARGSUSED */
int
Gem_reportCmd(ClientData , Tcl_Interp * interp, int objc,
              Tcl_Obj *CONST objv[])
{
    static char *gemcmds[] = {
        "init",
        "set",
        "add",
        "del",
        "dump",
        NULL
    };
    /*
     * All commands enums below ending in X are compatibility
     */
    enum gemcmds {
        GEMCMD_INIT,
        GEMCMD_SET,
        GEMCMD_ADD,
        GEMCMD_DEL,
        GEMCMD_DUMP,
    };

    int      cmdindex, result;
    Tcl_Obj  * res;
    JGReport * report;
    string   buf;

    Tcl_ResetResult(interp);
    result = TCL_OK;

    // JGEquipment * jyugem = (JGEquipment *)arg;

    /*
     * Get the command name index from the object based on the cmds
     * defined above.
     */
    if (Tcl_GetIndexFromObj(interp,
        objv[1], (const char **)gemcmds, "command", TCL_EXACT, &cmdindex) != TCL_OK)
        return TCL_ERROR;

    if (objc >= 3)
    {
        // Find report by id
        report = find_report(interp, Tcl_GetStringFromObj(objv[2], NULL));
        if (report == NULL)
        {
            return TCL_ERROR;
        }
    }

    res = NULL;
    switch ((enum gemcmds)cmdindex) {
    case GEMCMD_INIT:
        break;
    case GEMCMD_SET:
        if (objc != 4)
        {
            Tcl_WrongNumArgs(interp, 2, objv, "rptid ?args ?");
            return TCL_ERROR;
        }
        set_variables(interp, report, Tcl_GetStringFromObj(objv[3], NULL));
        break;

    case GEMCMD_ADD:
        if (objc != 4)
        {
            Tcl_WrongNumArgs(interp, 2, objv, "rptid vids");
            return TCL_ERROR;
        }
        result = add_variables(interp, report,
                               Tcl_GetStringFromObj(objv[3], NULL));
        break;

    case GEMCMD_DEL:
        if (objc == 3)
        {
            report->clearVariables();
        }
        else if (objc == 4)
        {
            result = del_variables(interp, report,
                                   Tcl_GetStringFromObj(objv[3], NULL));
        }
        else
        {
            Tcl_WrongNumArgs(interp, 2, objv, "rptid ?vids ?");
            return TCL_ERROR;
        }
        break;

    case GEMCMD_DUMP:
        if (objc == 2)
        {
            dump_reports(interp);
        }
        else if (objc == 3)
        {
            report->printOn(buf);
            Tcl_AppendResult(interp, "{", NULL);            // print header
            Tcl_AppendResult(interp, buf.c_str(), NULL);
            Tcl_AppendResult(interp, "} ", NULL);
        }
        else
        {
            Tcl_WrongNumArgs(interp, 2, objv, "?rptid ?");
            return TCL_ERROR;
        }
        break;

    }

    /*
     * For each different arg call different function to create
     * new commands (or if version, get/return it).
     */
    if (result == TCL_OK && res != NULL)
        Tcl_SetObjResult(interp, res);

    return TCL_OK;
}

/*----------------------------------------------------------------------
 * Dump all variables
 *----------------------------------------------------------------------
 */
int
dump_reports(Tcl_Interp * interp)
{
    string buf;

    // Variable tables in the equipment
    JGReportTable * rptbl = JGEventManager::instance()->reportTable();
    JGReportTable::iterator rpt_it = rptbl->begin();

    for ( ; rpt_it != rptbl->end(); ++rpt_it)
    {
        JGReport * report = &((*rpt_it).second);
        report->printOn(buf);

        Tcl_AppendResult(interp, "{", NULL);            // print header
        Tcl_AppendResult(interp, buf.c_str(), NULL);
        Tcl_AppendResult(interp, "} ", NULL);
    }
    return TCL_OK;
}

/*----------------------------------------------------------------------
 * Set value in the variable
 *----------------------------------------------------------------------
 */
static int set_variables(Tcl_Interp * interp, JGReport * report,
                         char * idsStr)
{
    int  result;
    vector<BS2id> ids;

    result = parse_vids(interp, idsStr, ids);
    if (result != TCL_OK)
    {
        return result;
    }

    report->clearVariables();

    // Set linked variables in the report
    for (size_t i = 0; i < ids.size(); i++)
    {
        JGVariable * variable = JGInfoManager::instance()->variable(ids[i]);
        report->insert(variable);
    }

    return TCL_OK;
}

//---------------------------------------------------------------------------
// Parse rptid in command parameter
//---------------------------------------------------------------------------
static JGReport * find_report(Tcl_Interp * interp, char * rptidStr)
{
    BS2id  rptid;

    const string& typeName = JGEquipment::instance()->getConf(_TX("IdentificationType"));
    if (typeName == _TX("UINT4"))
    {
        unsigned int num = strtoul(rptidStr, NULL, 0);
        if (num == 0)
        {
            Tcl_AppendResult(interp, _TX("Illegal RPTID : "), rptidStr, NULL);
            return NULL;
        }
        rptid = num;
    }
    else
    {
        rptid = rptidStr;
    }

    JGReport * report = JGEventManager::instance()->findReport(rptid);
    return report;
}

//---------------------------------------------------------------------------
// Parse vid in command parameter
//---------------------------------------------------------------------------
static int parse_vids(Tcl_Interp * interp, char * idsStr, vector<BS2id>& ids)
{
    bool  isAscii = true;
    BS2id id;
    int   result;
    int   count, i;
    const char  **lists;

    const string& typeName = JGEquipment::instance()->getConf(_TX("IdentificationType"));
    if (typeName == _TX("UINT4"))
    {
        isAscii = false;
    }

    result = Tcl_SplitList(interp, idsStr, &count, &lists);
    if (result != TCL_OK)
    {
        return result;
    }

    // parse id
    for (i = 0; i < count; i++)
    {
        if (isAscii)
        {
            id = lists[i];
        }
        else
        {
            unsigned int num = strtoul(lists[i], NULL, 0);
            if (num == 0)
            {
                Tcl_AppendResult(interp, "Illegal ID : ", lists[i], NULL);
                Tcl_Free((char *)lists);   // release list area
                return TCL_ERROR;
            }
            id = num;
        }
        JGVariable * variable = JGEquipment::instance()->variable(id);
        if (variable == NULL)
        {
            Tcl_AppendResult(interp, "Illegal VID : ", lists[i], NULL);
            return TCL_ERROR;
        }
        ids.push_back(id);
    }

    Tcl_Free((char *)lists);               // release list area
    return TCL_OK;
}

//---------------------------------------------------------------------------
// Add variables in the report
//---------------------------------------------------------------------------
static int add_variables(Tcl_Interp * interp, JGReport * report,
                         char * idsStr)
{
    int result;
    vector<BS2id> ids;

    result = parse_vids(interp, idsStr, ids);
    if (result != TCL_OK)
    {
        return result;
    }

    // Not check duplicate.

    for (size_t i = 0; i < ids.size(); i++)
    {
        JGVariable * variable = JGInfoManager::instance()->variable(ids[i]);
        report->insert(variable);
    }

    return TCL_OK;
}

//---------------------------------------------------------------------------
// Add variables in the report
//---------------------------------------------------------------------------
static int del_variables(Tcl_Interp * interp, JGReport * report,
                         char * idsStr)
{
    int result;
    size_t i;
    vector<BS2id> ids;

    result = parse_vids(interp, idsStr, ids);
    if (result != TCL_OK)
    {
        return result;
    }

    // Check vid.
    for (i = 0; i < ids.size(); i++)
    {
        if (! report->exist(ids[i]))
        {
            string badid;
            ids[i].get(badid);
            Tcl_AppendResult(interp, "Variable Not found: ", badid.c_str(), NULL);
            return TCL_ERROR;
        }
    }

    // Erase
    for (i = 0; i < ids.size(); i++)
    {
        JGVariable * variable = JGInfoManager::instance()->variable(ids[i]);
        report->erase(variable);
    }

    return TCL_OK;
}

