<?php
/**
* KakikoImage.class.php 処理概要：画像切り出し処理クラス
* PHP versions：4.4.2
* @パッケージ：-
* @作者：(IIS)H.Keta
* @作成日/作成者：2006/09/01 (IIS)H.Keta
* @更新日/更新者：
* @バージョン：1.0.0
*
* @メソッド一覧
* 	KakikoImage()
* 	setParam(array $param)
* 	void setFilename(string $img_filename)
* 	void setSavepath(string $img_savepath)
* 	void setSliceW(integer $slice_w)
* 	void setSliceH(integer $slice_h)
*   void setRim(integer $rim)
* 	bool start()
*   integer getCountAll()
*   integer getCountX()
*   integer getCountY()
*
* @使用方法
*
*
*
*
*
*
*
*
*
*
* Copyright (c) 2006 Iwate Information System Co.,Ltd. All Rights Reserved.       
* You may see the License at LICENSE_EN.txt(English) or LICENSE_JA.txt(Japanese). 
*/
class KakikoImage {

    var $_img_filename;    // 切り出し元ファイル名(フルパス)
    var $_img_savepath;    // 切り出し先フォルダパス
    var $_slice_w;         // 切り出し幅
    var $_slice_h;         // 切り出し高
    var $_rim;             // 画像に縁を付ける場合の幅(ピクセル)
    var $_format;          // 切り出しファイル命名規則
    var $_img_cnt;         // 総ファイル数
    var $_img_x_cnt;       // 横方向ファイル数
    var $_img_y_cnt;       // 縦方向ファイル数

    /**
    * 関数概要：コンストラクタ
    * 処理詳細：画像切り出し処理クラスのコンストラクタ処理
    *
    * @引数： なし
    * @戻値： なし
    * @作成日/作成者：2006/09/01 (IIS)H.Keta
    * @更新日/更新者：
    * @更新内容：
    */
    function KakikoImage() {
        $this->_img_filename = "";
        $this->_img_savepath = "";
        $this->_slice_w = 0;
        $this->_slice_h = 0;
        $this->_rim = 0;
        $this->_format = "slice_%d_%d.xxx";
    }

    /**
    * 関数概要：パラメータセット
    * 処理詳細：必要パラメータを連想配列でセット
    *
    * @引数： (I) param - array パラメータ連想配列
    * @戻値： なし
    * @作成日/作成者：2006/09/01 (IIS)H.Keta
    * @更新日/更新者：
    * @更新内容：
    */
    function setParam($param) {
        if (is_array($param)) {
            if (!empty($param['filename'])) {
                $this->setFilename($param['filename']);
            }
            if (!empty($param['savepath'])) {
                $this->setSavepath($param['savepath']);
            }
            if (!empty($param['sliceW'])) {
                $this->setSliceW($param['sliceW']);
            }
            if (!empty($param['sliceH'])) {
                $this->setSliceH($param['sliceH']);
            }
            if (!empty($param['rim'])) {
                $this->setRim($param['rim']);
            }
        }
    }

    /**
    * 関数概要：切り出し元ファイル名セット
    * 処理詳細：切り出し元ファイル名のパラメータを個別セット
    *
    * @引数： (I) img_filename - string ファイル名
    * @戻値： なし
    * @作成日/作成者：2006/09/01 (IIS)H.Keta
    * @更新日/更新者：
    * @更新内容：
    */
    function setFilename($img_filename) {
        $this->_img_filename = $img_filename;
    }

    /**
    * 関数概要：切り出し先フォルダパスセット
    * 処理詳細：切り出し先フォルダパスのパラメータを個別セット
    *
    * @引数： (I) img_savepath - string フォルダパス
    * @戻値： なし
    * @作成日/作成者：2006/09/01 (IIS)H.Keta
    * @更新日/更新者：
    * @更新内容：
    */
    function setSavepath($img_savepath) {
        $this->_img_savepath = $img_savepath;
    }

    /**
    * 関数概要：切り出し幅セット
    * 処理詳細：切り出し幅のパラメータを個別セット
    *
    * @引数： (I) slice_w - integer 切り出し幅
    * @戻値： なし
    * @作成日/作成者：2006/09/01 (IIS)H.Keta
    * @更新日/更新者：
    * @更新内容：
    */
    function setSliceW($slice_w) {
        $this->_slice_w = $slice_w;
    }

    /**
    * 関数概要：切り出し高セット
    * 処理詳細：切り出し高のパラメータを個別セット
    *
    * @引数： (I) slice_h - integer 切り出し高
    * @戻値： なし
    * @作成日/作成者：2006/09/01 (IIS)H.Keta
    * @更新日/更新者：
    * @更新内容：
    */
    function setSliceH($slice_h) {
        $this->_slice_h = $slice_h;
    }

    /**
    * 関数概要：縁の幅セット
    * 処理詳細：縁の幅のパラメータを個別セット
    *
    * @引数： (I) rim - integer 幅
    * @戻値： なし
    * @作成日/作成者：2006/09/01 (IIS)H.Keta
    * @更新日/更新者：
    * @更新内容：
    */
    function setRim($rim) {
        $this->_rim = $rim;
    }

    /**
    * 関数概要：総ファイル数の取得
    * 処理詳細：総ファイル数を返す
    *
    * @引数： なし
    * @戻値： integer : 総ファイル数
    * @作成日/作成者：2006/09/01 (IIS)H.Keta
    * @更新日/更新者：
    * @更新内容：
    */
    function getCountAll() {
        return $this->_img_cnt;
    }

    /**
    * 関数概要：横方向のファイル数の取得
    * 処理詳細：横方向のファイル数を返す
    *
    * @引数： なし
    * @戻値： integer : 横方向のファイル数
    * @作成日/作成者：2006/09/01 (IIS)H.Keta
    * @更新日/更新者：
    * @更新内容：
    */
    function getCountX() {
        return $this->_img_x_cnt;
    }

    /**
    * 関数概要：縦方向のファイル数の取得
    * 処理詳細：縦方向のファイル数を返す
    *
    * @引数： なし
    * @戻値： integer : 縦方向のファイル数
    * @作成日/作成者：2006/09/01 (IIS)H.Keta
    * @更新日/更新者：
    * @更新内容：
    */
    function getCountY() {
        return $this->_img_y_cnt;
    }

    /**
    * 関数概要：切り出し処理の実行
    * 処理詳細：切り出し処理を行う
    *
    * @引数： なし
    * @戻値： boolean : true  -> 正常
    *                   false -> 異常
    * @作成日/作成者：2006/09/01 (IIS)H.Keta
    * @更新日/更新者：
    * @更新内容：
    */
    function start() {

        // 切り出し元ファイルが存在するか
        if (!file_exists($this->_img_filename)) {
            return false;
        }
        // 切り出し先フォルダが存在するか
        if (!file_exists($this->_img_savepath)) {
            return false;
        }
        // 切り出し幅がセットされているか
        if ($this->_slice_w == 0) {
            return false;
        }
        // 切り出し高がセットされているか
        if ($this->_slice_h == 0) {
            return false;
        }

        // 切り出し元ファイル情報の取得
        list($img_org_w, $img_org_h, $img_type, $attr) = getimagesize($this->_img_filename);
        // ファイルタイプの判定(gif/jpg/png)
        switch ($img_type) {
            case IMAGETYPE_GIF:
            case IMAGETYPE_JPEG:
            case IMAGETYPE_PNG:
                break;
            default:
                return false;
        }

        $img_w = $this->_slice_w;    // 切り出し幅
        $img_h = $this->_slice_h;    // 切り出し高
        $outfilecnt = 0;             //カウンタ(総ファイル数)
        $x = 0;                      //カウンタ(横数)
        $y = 0;                      //カウンタ(縦数)

        // 切り出し元イメージ生成(メモリ)
        $src_img = $this->imagecreate($this->_img_filename, $img_type);
        if ($src_img == false) {
            return false;
        }
        // 縁の幅の設定があるか
        if ($this->_rim > 0) {
            $src_img = $this->imageaddrim($src_img, $img_org_w, $img_org_h);
        }
        //指定した切り出し高さが元画像の高さを超えるまでループ
        do {

            // 横数カウンタをリセット
            $x = 0;

            //指定した切り出し幅が元画像の幅を超えるまでループ
            do {

                // 切り出し先イメージ生成(メモリ)
                $dst_img = imagecreatetruecolor($img_w, $img_h);
                $bgcolor = imagecolorallocate($dst_img, 255, 255, 255);
                imagefill($dst_img, 0, 0, $bgcolor);

                // 切り出し作業
                $dst_w = (($img_org_w - ($img_w * $x)) >= $img_w) ? $img_w : ($img_org_w - ($img_w * $x));
                $dst_h = (($img_org_h - ($img_h * $y)) >= $img_h) ? $img_h : ($img_org_h - ($img_h * $y));
                $result  = imagecopyresampled(
                                              $dst_img,    //resource dst_image (切り出し先イメージID)
                                              $src_img,    //resource src_image (切り出し元イメージID)
                                              0,           //int dstX (貼り付けを開始するX座標)
                                              0,           //int dstY (貼り付けを開始するY座標)
                                              $img_w * $x, //int srcX (コピーを開始するX座標)
                                              $img_h * $y, //int srcY (コピーを開始するY座標)
                                              $dst_w,      //int dstW (貼り付けする幅)
                                              $dst_h,      //int dstH (貼り付けする高さ)
                                              $dst_w,      //int srcW (コピーする幅)
                                              $dst_h       //int srcH (コピーする高さ)
                                             );

                // イメージをファイルに保存
                $outfilename = "";
                $outfilename = $this->_img_savepath.sprintf($this->_format, $x, $y);
                $this->imagefile($dst_img, $outfilename, $img_type);
                // 切り出し先イメージ破棄(メモリ)
                imagedestroy($dst_img);

                // 横数カウンタをアップ
                $x++;
                // 総ファイル数カウンタをアップ
                $outfilecnt++;

            } while($img_org_w >= ($img_w * $x));

            // 縦数カウンタをアップ
            $y++;

        } while($img_org_h >= ($img_h * $y));
        // 切り出し元イメージ破棄(メモリ)
        imagedestroy($src_img);
        
        $this->_img_cnt = $outfilecnt;    // 総ファイル数
        $this->_img_x_cnt = $x;           // 横数
        $this->_img_y_cnt = $y;           // 縦数
        
        return true;

    }

    /**
    * 関数概要：イメージを作成
    * 処理詳細：指定されたサイズで縁を付加したイメージを作成
    *
    * @引数： (I) resimage  - string  出力イメージID
    *         (I) img_org_w - integer 元画像の幅
    *         (I) img_org_h - integer 元画像の高さ
    * @戻値： resource : イメージID
    *         boolean  : false -> 異常
    * @作成日/作成者：2006/09/01 (IIS)H.Keta
    * @更新日/更新者：
    * @更新内容：
    */
    function imageaddrim(&$resimage, &$img_org_w, &$img_org_h) {
        // 縁の幅がセットされていない場合は処理しない
        if ($this->_rim == 0) {
            return false;
        }
        // 切り出し先イメージ生成(メモリ)
        $dst_img = imagecreatetruecolor($img_org_w + ($this->_rim * 2), $img_org_h + ($this->_rim * 2));
        $bgcolor = imagecolorallocate($dst_img, 255, 255, 255);
        imagefill($dst_img, 0, 0, $bgcolor);
        // 切り出し作業
        $result  = imagecopyresampled(
                                      $dst_img,     //resource dst_image (切り出し先イメージID)
                                      $resimage,    //resource src_image (切り出し元イメージID)
                                      $this->_rim,  //int dstX (貼り付けを開始するX座標)
                                      $this->_rim,  //int dstY (貼り付けを開始するY座標)
                                      0,            //int srcX (コピーを開始するX座標)
                                      0,            //int srcY (コピーを開始するY座標)
                                      $img_org_w,   //int dstW (貼り付けする幅)
                                      $img_org_h,   //int dstH (貼り付けする高さ)
                                      $img_org_w,   //int srcW (コピーする幅)
                                      $img_org_h    //int srcH (コピーする高さ)
                                     );
        // 旧イメージは破棄
        imagedestroy($resimage);
        // 切り出し元のサイズを再セット
        $img_org_w += $this->_rim * 2;
        $img_org_h += $this->_rim * 2;
        return $dst_img;
    }

    /**
    * 関数概要：イメージを作成
    * 処理詳細：指定されたパスのイメージを作成する(ファイルタイプ別)
    *
    * @引数： (I) filename - string  切り出し元ファイル
    *         (I) img_type - integer ファイルタイプ
    * @戻値： resource : イメージID
    *         boolean  : false -> 異常
    * @作成日/作成者：2006/09/01 (IIS)H.Keta
    * @更新日/更新者：
    * @更新内容：
    */
    function imagecreate($filename, $img_type) {
        switch ($img_type) {
            case IMAGETYPE_GIF:
                $ret = @imagecreatefromgif($filename);
                break;
            case IMAGETYPE_JPEG:
                $ret = @imagecreatefromjpeg($filename);
                break;
            case IMAGETYPE_PNG:
                $ret = @imagecreatefrompng($filename);
                break;
            default:
                $ret = false;
                break;
        }
        return $ret;
    }

    /**
    * 関数概要：画像のファイル出力
    * 処理詳細：指定されたイメージをファイルへ出力する(ファイルタイプ別)
    *
    * @引数： (I) resimage    - string  出力イメージID
    *         (I) outfilename - integer 出力ファイル名
    *         (I) img_type    - integer ファイルタイプ
    * @戻値： boolean : true  -> 正常
    *                   false -> 異常
    * @作成日/作成者：2006/09/01 (IIS)H.Keta
    * @更新日/更新者：
    * @更新内容：
    */
    function imagefile($resimage, $outfilename, $img_type) {
        switch ($img_type) {
            case IMAGETYPE_GIF:
                $ret = imagegif($resimage, $outfilename);
                break;
            case IMAGETYPE_JPEG:
                $ret = imagejpeg($resimage, $outfilename);
                break;
            case IMAGETYPE_PNG:
                $ret = imagepng($resimage, $outfilename);
                break;
            default:
                $ret = false;
                break;
        }
        return $ret;
    }
}

?>

<?php
/*
//****************************************************************************************************
// 以下はクラスのテスト用です。
//****************************************************************************************************
$o_kakikoimage = new KakikoImage();
$param = array(
                "filename"=>"c:/work/xdfs03.jpg",    // 切り出し元ファイルパス
                "savepath"=>"c:/work/slice/",        // 切り出し後ファイル格納フォルダ
                "sliceW"=>"400",                     // 切り出しサイズ(幅)
                "sliceH"=>"400",                     // 切り出しサイズ(高さ)
                "rim"=>"100"                         // のりしろサイズ
               );
$o_kakikoimage->setParam($param);
if ($o_kakikoimage->start() === true) {
    echo "OK!";
    echo "<br>all file count : ".$o_kakikoimage->getCountAll();
    echo "<br>x file count : ".$o_kakikoimage->getCountX();
    echo "<br>y file count : ".$o_kakikoimage->getCountY();
} else {
    echo "NG!";
}
echo "<body bgcolor='#FFCCCC'>";
echo "<br>";
echo "<br>";
$cnt_x=0;
$cnt_y=0;
print("<table>");
while (file_exists($param['savepath'].sprintf("slice_%d_%d.xxx", $cnt_x, $cnt_y))==true) {
	print("<tr>");
	while (file_exists($param['savepath'].sprintf("slice_%d_%d.xxx", $cnt_x, $cnt_y))==true) {
		print("<td style='width:1em'>");
		print(sprintf("slice_%d_%d.xxx", $cnt_x, $cnt_y));
		print("<img src=".$param['savepath'].sprintf("slice_%d_%d.xxx", $cnt_x, $cnt_y)." alt='".sprintf("slice_%d_%d.xxx", $cnt_x, $cnt_y)."' width=100 height=100>\n");
		print("</td>");
		$cnt_x++;
	}
	print("</tr>");
	$cnt_x=0;
	$cnt_y++;
}
print("</table>");
print("</body>");
//****************************************************************************************************
*/
?>
