<?php
/**
* CommentTableAccess.class.php commenttableDAO
* PHP versions：4.4.2
* @パッケージ：-
* @作者：K.Shimomura
* @作成日/作成者：2006/09/01 (IIS)K.Shimomura
* @更新日/更新者：
* @バージョン：1.0.0
*
* Copyright (c) 2006 Iwate Information System Co.,Ltd. All Rights Reserved.       
* You may see the License at LICENSE_EN.txt(English) or LICENSE_JA.txt(Japanese). 
*/ 
//データクラス
require_once("class/db/tabledata/CommentData.class.php");
//データオブジェクト定義
require_once("class/db/dbaccess/DataObject.php");
//データアクセスクラス
require_once("class/db/dbaccess/KakikoDbAccess.class.php");
//継承テーブルアクセスクラス
require_once("class/db/dbaccess/TableAccess.class.php");


class CommentTableAccess extends TableAccess {

	/**
	 *	クラス変数
	 */
	var $prefix;				// プレフィックス


	/**
	 * コンストラクタ
	 * @引数： (I) directryname  - string ディレクトリ名称
	 */
    function CommentTableAccess($directryname) {

		// 基底クラスのコンストラクタ呼び出し
    	$this->TableAccess();

    	$this->setTableName($directryname."_commenttable");
    	$this->setRecordSet(null);

		// プレフィックス設定
		$this->prefix = $directryname;
    }

    /**
     * 関数概要：コメントテーブルデータ全件取得クエリ発行
     * 処理詳細：コメントテーブルデータの全件取得クエリを生成する
     *
     * @引数： (I) db       - object DBオブジェクト
     * @戻値： boolean true :正常
     *      ：         false:異常
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
	function selectAllData($db) {

		$dba = new KakikoDBAccess($db, $this->getTablename());

		if ($dba -> dba_select() != true) {
			trigger_error($this->o_resource->getValue("D4_001_01"), E_USER_ERROR);
			return false;
		}
		$this->setRecordSet($dba->dba_getRecordSet());

		// 解放
		$dba->dba_free();

		return true;
	}

    /**
     * 関数概要：コメントテーブルキー指定データ取得クエリ発行
     * 処理詳細：コメントテーブルキー指定データの取得クエリを生成する
     *
     * @引数： (I) db       - object  DBオブジェクト
     *      ： (I) key      - integer キー内容(コメントID)
     * @戻値： boolean true :正常
     *      ：         false:異常
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
	function selectFindByKeyData($db, $key) {

		$dba = new KakikoDBAccess($db, $this->getTablename());

		// 条件指定
		$dba->dba_whereAdd("", "commentid = ".$key);

		if ($dba -> dba_select() != true) {
			trigger_error($this->o_resource->getValue("D4_002_01"), E_USER_ERROR);
			return false;
		}

		$this->setRecordSet($dba->dba_getRecordSet());

		// 解放
		$dba->dba_free();

		return true;
	}

    /**
     * 関数概要：コメントテーブルデータ挿入
     * 処理詳細：コメントテーブルデータの挿入クエリを生成する
     *
     * @引数： (I) db       - object DBオブジェクト
     *      ： (I) dataobj  - object コメントデータオブジェクト
     * @戻値： boolean true :正常
     *      ：         false:異常
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
	function insertData($db, $dataobj) {

		$dba = new KakikoDBAccess($db, $this->getTablename());

		if (is_object($dataobj) && get_class($dataobj) == COMMENT_DATAOBJ) {

			// コメントID
			$dataobj->setCommentID('default');
			// メモ種類ID
			$dataobj->setMemoTypeID($dataobj->getMemoTypeID());
			// メモID
			$dataobj->setMemoID($dataobj->getMemoID());
			// 上位コメントID
			$dataobj->setHigherCommentID($dataobj->getHigherCommentID());
			// ユーザ名称
			$dataobj->setUserName(
				$dba->query_str_convert($dataobj->getUserName()));
			// コメントタイトル
			$dataobj->setCommentTitle(
				$dba->query_str_convert($dataobj->getCommentTitle()));
			// 本文
			$dataobj->setCommentText(
				$dba->query_str_convert($dataobj->getCommentText()));
			// メールアドレス
			if (is_null($dataobj->getMailAddress()) != true) {
				$dataobj->setMailAddress(
					$dba->query_str_convert($dataobj->getMailAddress()));
			}
			// 関連URL
			if (is_null($dataobj->getReleteURL()) != true) {
				$dataobj->setReleteURL(
					$dba->query_str_convert($dataobj->getReleteURL()));
			}
			// 画像ファイル
			if (is_null($dataobj->getImageFile()) != true) {
				$dataobj->setImageFile(
					$dba->query_str_convert($dataobj->getImageFile()));
			}
			// 修正・削除用パスワード
			if (is_null($dataobj->getUpdateDeletePassword()) != true) {
				$dataobj->setUpdateDeletePassword(
					$dba->query_str_convert($dataobj->getUpdateDeletePassword()));
			}
			// アクセス回数
			$dataobj->setAccessCount($dataobj->getAccessCount());
			// 削除フラグ
			if (is_null($dataobj->getDeleteFlg()) != true) {
				$dataobj->setDeleteFlg(
					$dba->query_str_convert($dataobj->getDeleteFlg()));
			}
			// 登録日時
			$dataobj->setAddDateTime(
				$dba->query_str_convert($dataobj->getAddDateTime()));
			// 更新日時
			$dataobj->setUpdateDateTime(
				$dba->query_str_convert($dataobj->getUpdateDateTime()));

			if ($dba->dba_insert($dataobj) != true) {
				trigger_error($this->o_resource->getValue("D4_003_01"), E_USER_ERROR);
				return false;
			}
		} else {
			trigger_error($this->o_resource->getValue("D4_003_02"), E_USER_ERROR);
			return false;
		}

		// 解放
		$dba->dba_free();

		return true;
	}

	/**
	 *	関数名：コメントテーブルキー指定データ更新
     * @引数：(I) $db               - DBオブジェクト
     *        (I) $targetDataobj    - 対象列(条件列/更新列)指定用データオブジェクト
     *        (I) $valDataobj       - 設定値(条件値/更新値)指定用データオブジェクト
     * @戻値：$bool			- true/false
     * @作成日/作成者：2006/09/01 (IIS) K.Shimomura
     * @更新日/更新者：
     * @使用方法：whereキーとして使用したい列は$targetDataobjの該当変数をWHERE_KEYに指定し、条件値を$valDataobjの該当変数に使用する。
	 *            (プライマリキー以外にWHERE_KEYを指定しても無視されます)
     *            更新したい列は$targetDataobjの該当変数をVAL_TARGETに指定し、更新値を$valDataobjの該当変数に使用する。
     *            NULLで更新したい時は$valDataobjの該当変数にNULL_CHARを指定する。
     */
	function updateByKeyData($db, $targetDataobj, $valDataobj) {

		$whereflg = false;
		$dba = new KakikoDBAccess($db, $this->getTablename());

		if ((is_object($targetDataobj) && get_class($targetDataobj) == COMMENT_DATAOBJ) && 
			(is_object($valDataobj) && get_class($valDataobj) == COMMENT_DATAOBJ)) {

			// 条件および更新内容指定

			// コメントID
			if ($targetDataobj->getCommentID() === WHERE_KEY) {
				$dba->dba_whereAdd("", "commentid = ".$valDataobj->getCommentID());
				$whereflg = true;
			} elseif ($targetDataobj->getCommentID() === VAL_TARGET) {
				// コメントIDはDBでのシーケンスナンバーを使用しているので
				// 変えられると困る
//				$dba->dba_updateAdd(
//						"commentid = "
//							.$valDataobj->getCommentID());
			}
			// メモ種類ID
			if ($targetDataobj->getMemoTypeID() === VAL_TARGET) {
				$dba->dba_updateAdd(
						"memotypeid = "
							.$valDataobj->getMemoTypeID());
			}
			// メモID
			if ($targetDataobj->getMemoID() === VAL_TARGET) {
				$dba->dba_updateAdd(
						"memoid = "
							.$valDataobj->getMemoID());
			}
			// 上位コメントID
			if ($targetDataobj->getHigherCommentID() === VAL_TARGET) {
				$dba->dba_updateAdd(
						"highercommentid = "
							.$valDataobj->getHigherCommentID());
			}
			// ユーザ名称
			if ($targetDataobj->getUserName() === VAL_TARGET) {
				$dba->dba_updateAdd(
						"username = "
							.$dba->query_str_convert($valDataobj->getUserName()));
			}
			// コメントタイトル
			if ($targetDataobj->getCommentTitle() === VAL_TARGET) {
				$dba->dba_updateAdd(
						"commenttitle = "
							.$dba->query_str_convert($valDataobj->getCommentTitle()));
			}
			// 本文
			if ($targetDataobj->getCommentText() === VAL_TARGET) {
				$dba->dba_updateAdd(
						"commenttext = "
							.$dba->query_str_convert($valDataobj->getCommentText()));
			}
			// メールアドレス
			if ($targetDataobj->getMailAddress() === VAL_TARGET) {
				if ($valDataobj->getMailAddress() === NULL_CHAR) {
					$dba->dba_updateAdd("mailaddress = null");
				} else {
					$dba->dba_updateAdd(
							"mailaddress = "
								.$dba->query_str_convert($valDataobj->getMailAddress()));
				}
			}
			// 関連URL
			if ($targetDataobj->getReleteURL() === VAL_TARGET) {
				if ($valDataobj->getReleteURL() === NULL_CHAR) {
					$dba->dba_updateAdd("releteurl = null");
				} else {
					$dba->dba_updateAdd(
							"releteurl = "
								.$dba->query_str_convert($valDataobj->getReleteURL()));
				}
			}
			// 画像ファイル
			if ($targetDataobj->getImageFile() === VAL_TARGET) {
				if ($valDataobj->getImageFile() === NULL_CHAR) {
					$dba->dba_updateAdd("imagefile = null");
				} else {
					$dba->dba_updateAdd(
							"imagefile = "
								.$dba->query_str_convert($valDataobj->getImageFile()));
				}
			}
			// 修正・削除用パスワード
			if ($targetDataobj->getUpdateDeletePassword() === VAL_TARGET) {
				if ($valDataobj->getUpdateDeletePassword() === NULL_CHAR) {
					$dba->dba_updateAdd("updatedeletepassword = null");
				} else {
					$dba->dba_updateAdd(
							"updatedeletepassword = "
								.$dba->query_str_convert($valDataobj->getUpdateDeletePassword()));
				}
			}
			// アクセス回数
			if ($targetDataobj->getAccessCount() === VAL_TARGET) {
				$dba->dba_updateAdd(
						"accesscount = "
							.$valDataobj->getAccessCount());
			}
			// 削除フラグ
			if ($targetDataobj->getDeleteFlg() === VAL_TARGET) {
					$dba->dba_updateAdd(
							"deleteflg = "
								.$dba->query_str_convert($valDataobj->getDeleteFlg()));
			}
			// 登録日時
			if ($targetDataobj->getAddDateTime() === VAL_TARGET) {
				$dba->dba_updateAdd(
						"adddatetime = "
							.$dba->query_str_convert($valDataobj->getAddDateTime()));
			}
			// 更新日時
			if ($targetDataobj->getUpdateDateTime() === VAL_TARGET) {
				$dba->dba_updateAdd(
						"updatedatetime = "
							.$dba->query_str_convert($valDataobj->getUpdateDateTime()));
			}

			// 条件指定確認
			if ($whereflg == false) {
				trigger_error($this->o_resource->getValue("D4_004_01"), E_USER_ERROR);
				return false;
			}
			//SQL実行
			if ($dba -> dba_update() != true) {
				trigger_error($this->o_resource->getValue("D4_004_02"), E_USER_ERROR);
				return false;
			}
		} else {
			trigger_error($this->o_resource->getValue("D4_004_03"), E_USER_ERROR);
			return false;
		}

		// 解放
		$dba->dba_free();

		return true;

	}

	/**
	 *	関数名：コメントテーブルINDEXキー指定データ更新
     * @引数：(I) $db               - DBオブジェクト
     *        (I) $targetDataobj    - 対象列(条件列/更新列)指定用データオブジェクト
     *        (I) $valDataobj       - 設定値(条件値/更新値)指定用データオブジェクト
     * @戻値：$bool			- true/false
     * @作成日/作成者：2006/09/01 (IIS) J.Hashimoto
     * @更新日/更新者：
     * @使用方法：whereキーとして使用したい列は$targetDataobjの該当変数をWHERE_KEYに指定し、条件値を$valDataobjの該当変数に使用する。
	 *            (INDEXキー以外にWHERE_KEYを指定しても無視されます)
     *            更新したい列は$targetDataobjの該当変数をVAL_TARGETに指定し、更新値を$valDataobjの該当変数に使用する。
     *            NULLで更新したい時は$valDataobjの該当変数にNULL_CHARを指定する。
     */
	function updateByMemoIdData($db, $targetDataobj, $valDataobj) {

		$whereflg = false;
		$dba = new KakikoDBAccess($db, $this->getTablename());

		if ((is_object($targetDataobj) && get_class($targetDataobj) == COMMENT_DATAOBJ) && 
			(is_object($valDataobj) && get_class($valDataobj) == COMMENT_DATAOBJ)) {

			// 条件および更新内容指定

			// メモID
			if ($targetDataobj->getMemoID() === WHERE_KEY) {
				$dba->dba_whereAdd("", "memoid = ".$valDataobj->getMemoID());
				$whereflg = true;

			}
			// メモ種類ID
			if ($targetDataobj->getMemoTypeID() === VAL_TARGET) {
				$dba->dba_updateAdd(
						"memotypeid = "
							.$valDataobj->getMemoTypeID());
			}
			// 更新日時
			if ($targetDataobj->getUpdateDateTime() === VAL_TARGET) {
				$dba->dba_updateAdd(
						"updatedatetime = "
							.$dba->query_str_convert($valDataobj->getUpdateDateTime()));
			}

			// 条件指定確認
			if ($whereflg == false) {
				trigger_error($this->o_resource->getValue("D4_004_01"), E_USER_ERROR);
				return false;
			}
			//SQL実行
			if ($dba -> dba_update() != true) {
				trigger_error($this->o_resource->getValue("D4_004_02"), E_USER_ERROR);
				return false;
			}
		} else {
			trigger_error($this->o_resource->getValue("D4_004_03"), E_USER_ERROR);
			return false;
		}

		// 解放
		$dba->dba_free();

		return true;

	}

    /**
     * 関数概要：コメントテーブルキー指定データ削除
     * 処理詳細：コメントテーブルキー指定データの削除クエリを生成する
     *
     * @引数： (I) db       - object  DBオブジェクト
     *      ： (I) key      - integer キー内容(コメントID)
     * @戻値： boolean true :正常
     *      ：         false:異常
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
	function deleteByKeyData($db, $key) {

		$dba = new KakikoDBAccess($db, $this->getTablename());

		// 条件指定
		$dba->dba_whereAdd("", "commentid = ".$key);

		if ($dba -> dba_delete() != true) {
			trigger_error($this->o_resource->getValue("D4_005_01"), E_USER_ERROR);
			return false;
		}

		// 解放
		$dba->dba_free();

		return true;
	}

    /**
     * 関数概要：コメントテーブルキー指定（メモ）データ一括削除
     * 処理詳細：コメントテーブルキー指定（メモ）データの一括削除クエリを生成する
     *
     * @引数： (I) db       - object  DBオブジェクト
     *      ： (I) key      - integer キー内容(メモID)
     * @戻値： boolean true :正常
     *      ：         false:異常
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
	function deleteByMemoIdData($db, $key) {

		$dba = new KakikoDBAccess($db, $this->getTablename());

		// 条件指定
		$dba->dba_whereAdd("", "memoid = ".$key);

		if ($dba -> dba_delete() != true) {
			trigger_error($this->o_resource->getValue("D4_006_01"), E_USER_ERROR);
			return false;
		}

		// 解放
		$dba->dba_free();

		return true;
	}

    /**
     * 関数概要：一覧データ取得
     * 処理詳細：レコードセットより、コメントリストを生成する
     *
     * @引数： 
     * @戻値： array   コメントリスト
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
    function getDataList() {
        
		$comment_array = array();

		$rs = $this->getRecordSet();

		while($row = $rs->fetchrow(DB_FETCHMODE_ASSOC)) {

			$commentdata = new CommentData();

			$commentdata->setCommentID(isset($row['commentid'])?$row['commentid']:null);
			$commentdata->setMemoTypeID(isset($row['memotypeid'])?$row['memotypeid']:null);
			$commentdata->setMemoID(isset($row['memoid'])?$row['memoid']:null);
			$commentdata->setHigherCommentID(isset($row['highercommentid'])?$row['highercommentid']:null);
			$commentdata->setUserName(isset($row['username'])?$row['username']:null);
			$commentdata->setCommentTitle(isset($row['commenttitle'])?$row['commenttitle']:null);
			$commentdata->setCommentText(isset($row['commenttext'])?$row['commenttext']:null);
			$commentdata->setMailAddress(isset($row['mailaddress'])?$row['mailaddress']:null);
			$commentdata->setReleteURL(isset($row['releteurl'])?$row['releteurl']:null);
			$commentdata->setImageFile(isset($row['imagefile'])?$row['imagefile']:null);
			$commentdata->setUpdateDeletePassword(isset($row['updatedeletepassword'])?$row['updatedeletepassword']:null);
			$commentdata->setAccessCount(isset($row['accesscount'])?$row['accesscount']:null);
			$commentdata->setDeleteFlg(isset($row['deleteflg'])?$row['deleteflg']:null);
			$commentdata->setAddDateTime(isset($row['adddatetime'])?$row['adddatetime']:null);
			$commentdata->setUpdateDateTime(isset($row['updatedatetime'])?$row['updatedatetime']:null);

			array_push($comment_array, $commentdata);
		}
 		return $comment_array;
    }

    /**
     * 関数概要：単体データ取得
     * 処理詳細：レコードセットより、一件のコメントデータを生成する
     *
     * @引数： 
     * @戻値： array   コメントデータ型オブジェクト
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
    function getOneData() {

		$rs = $this->getRecordSet();

		$row = $rs->fetchrow(DB_FETCHMODE_ASSOC);

		$commentdata = new CommentData();

		$commentdata->setCommentID(isset($row['commentid'])?$row['commentid']:null);
		$commentdata->setMemoTypeID(isset($row['memotypeid'])?$row['memotypeid']:null);
		$commentdata->setMemoID(isset($row['memoid'])?$row['memoid']:null);
		$commentdata->setHigherCommentID(isset($row['highercommentid'])?$row['highercommentid']:null);
		$commentdata->setUserName(isset($row['username'])?$row['username']:null);
		$commentdata->setCommentTitle(isset($row['commenttitle'])?$row['commenttitle']:null);
		$commentdata->setCommentText(isset($row['commenttext'])?$row['commenttext']:null);
		$commentdata->setMailAddress(isset($row['mailaddress'])?$row['mailaddress']:null);
		$commentdata->setReleteURL(isset($row['releteurl'])?$row['releteurl']:null);
		$commentdata->setImageFile(isset($row['imagefile'])?$row['imagefile']:null);
		$commentdata->setUpdateDeletePassword(isset($row['updatedeletepassword'])?$row['updatedeletepassword']:null);
		$commentdata->setAccessCount(isset($row['accesscount'])?$row['accesscount']:null);
		$commentdata->setDeleteFlg(isset($row['deleteflg'])?$row['deleteflg']:null);
		$commentdata->setAddDateTime(isset($row['adddatetime'])?$row['adddatetime']:null);
		$commentdata->setUpdateDateTime(isset($row['updatedatetime'])?$row['updatedatetime']:null);

 		return $commentdata;
    }

    /**
     * 関数概要：コメントテーブルキー指定重複チェック
     * 処理詳細：コメントテーブルデータをキー指定で重複しているかをチェック
     *
     * @引数： (I) db       - object  DBオブジェクト
     *      ： (I) key      - integer キー内容(コメントID)
     * @戻値： boolean true :データなし
     *      ：         false:データあり
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
	function checkFindByKeyData($db, $key) {

		$rtn = true;

		// キー指定データの取得
		$this->selectFindByKeyData($db, $key);

		if ($this->getRecordCount() > 0) {
			$rtn = false;	
		}

		return $rtn;
	}

	/**
	 * 関数名：メモ配下コメント存在チェック
     * @引数：(I) $db    	- DBオブジェクト
     *        (I) $memoId   - メモID     
     * @戻値：$bool			- true  配下にコメントが存在する
     *                      - false 配下にコメントが存在しない
     * @作成日/作成者：2006/09/01 (IIS)N.Soma
     * @更新日/更新者：
	 * @注意事項：削除フラグが立っているコメントは無視される
     */
	function checkFindByMemoidData($db, $memoId) {

		$rtn = false;

		$dba = new KakikoDBAccess($db, $this->getTablename());

		// 条件指定
		$dba->dba_whereAdd("", "memoid = ".$memoId);
		$dba->dba_whereAdd("AND", "deleteflg = '0'");	//削除フラグがあるコメントは対象外
		
		if ($dba -> dba_select() != true) {
			trigger_error($this->o_resource->getValue("D4_007_01"), E_USER_ERROR);
			return false;
		}

		$this->setRecordSet($dba->dba_getRecordSet());
		if ($this->getRecordCount() > 0) {
			$rtn = true;	
		}

		// 解放
		$dba->dba_free();

		return $rtn;

	}
	/**
	 * 関数名：コメント配下コメント存在チェック
     * @引数：(I) $db    		- DBオブジェクト
     *        (I) $commentId   	- 湖面とID     
     * @戻値：$bool			- true  配下にコメントが存在する
     *                      - false 配下にコメントが存在しない
     * @作成日/作成者：2006/09/01 (IIS)N.Soma
     * @更新日/更新者：
	 * @注意事項：削除フラグが立っているコメントは無視される
	 *            
     */
	function checkFindByCommentidData($db, $commentId) {

		$rtn = false;

		$dba = new KakikoDBAccess($db, $this->getTablename());

		// 条件指定
		$dba->dba_whereAdd("", "highercommentid = ".$commentId);
		$dba->dba_whereAdd("AND", "deleteflg = '0'");	//削除フラグがあるコメントは対象外
		if ($dba -> dba_select() != true) {
			trigger_error($this->o_resource->getValue("D4_008_01"), E_USER_ERROR);
			return false;
		}

		$this->setRecordSet($dba->dba_getRecordSet());
		if ($this->getRecordCount() > 0) {
			$rtn = true;	
		}

		// 解放
		$dba->dba_free();

		return $rtn;

	}	
}
?>