<?php
/**
* LayerPriorityTableAccess.class.php layerprioritytableDAO
* PHP versions：4.4.2
* @パッケージ：-
* @作者：K.Shimomura
* @作成日/作成者：2006/09/01 (IIS)K.Shimomura
* @更新日/更新者：
* @バージョン：1.0.0
*
* Copyright (c) 2006 Iwate Information System Co.,Ltd. All Rights Reserved.       
* You may see the License at LICENSE_EN.txt(English) or LICENSE_JA.txt(Japanese). 
*/ 
//データクラス
require_once("class/db/tabledata/LayerPriorityData.class.php");
//データオブジェクト定義
require_once("class/db/dbaccess/DataObject.php");
//データアクセスクラス
require_once("class/db/dbaccess/KakikoDbAccess.class.php");
//継承テーブルアクセスクラス
require_once("class/db/dbaccess/TableAccess.class.php");


class LayerPriorityTableAccess extends TableAccess {

	/**
	 *	クラス変数
	 */
	var $prefix;				// プレフィックス


	/**
	 * コンストラクタ
	 * @引数： (I) directryname  - string ディレクトリ名称
	 */
    function LayerPriorityTableAccess($directryname) {

		// 基底クラスのコンストラクタ呼び出し
    	$this->TableAccess();

    	$this->setTableName($directryname."_layerprioritytable");
    	$this->setRecordSet(null);

		// プレフィックス設定
		$this->prefix = $directryname;
    }

    /**
     * 関数概要：レイヤー優先順位テーブルデータ全件取得クエリ発行
     * 処理詳細：レイヤー優先順位テーブルデータの全件取得クエリを生成する
     *
     * @引数： (I) db       - object DBオブジェクト
     * @戻値： boolean true :正常
     *      ：         false:異常
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
	function selectAllData($db) {

		$dba = new KakikoDBAccess($db, $this->getTablename());

		if ($dba -> dba_select() != true) {
			trigger_error($this->o_resource->getValue("D8_001_01"), E_USER_ERROR);
			return false;
		}
		$this->setRecordSet($dba->dba_getRecordSet());

		// 解放
		$dba->dba_free();

		return true;
	}

    /**
     * 関数概要：レイヤー優先順位テーブルキー指定データ取得クエリ発行
     * 処理詳細：レイヤー優先順位テーブルキー指定データの取得クエリを生成する
     *
     * @引数： (I) db       - object  DBオブジェクト
     *      ： (I) key      - integer キー内容(優先順位)
     * @戻値： boolean true :正常
     *      ：         false:異常
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
	function selectFindByKeyData($db, $key) {

		$dba = new KakikoDBAccess($db, $this->getTablename());

		// 条件指定
		$dba->dba_whereAdd("", "priorityorder = ".$dba->query_str_convert($key));

		if ($dba -> dba_select() != true) {
			trigger_error($this->o_resource->getValue("D8_002_01"), E_USER_ERROR);
			return false;
		}

		$this->setRecordSet($dba->dba_getRecordSet());

		// 解放
		$dba->dba_free();

		return true;
	}

    /**
     * 関数概要：レイヤー優先順位テーブルデータ挿入
     * 処理詳細：レイヤー優先順位テーブルデータの挿入クエリを生成する
     *
     * @引数： (I) db       - object DBオブジェクト
     *      ： (I) dataobj  - object レイヤー優先順位データオブジェクト
     * @戻値： boolean true :正常
     *      ：         false:異常
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
	function insertData($db, $dataobj) {

		$dba = new KakikoDBAccess($db, $this->getTablename());

		if (is_object($dataobj) && get_class($dataobj) == LAYERPRIORITY_DATAOBJ) {

			// 優先順位
			$dataobj->setPriorityOrder($dataobj->getPriorityOrder());
			// レイヤーID
			$dataobj->setLayerID(
				$dba->query_str_convert($dataobj->getLayerID()));
			// 登録日時
			$dataobj->setAddDateTime(
				$dba->query_str_convert($dataobj->getAddDateTime()));
			// 更新日時
			$dataobj->setUpdateDateTime(
				$dba->query_str_convert($dataobj->getUpdateDateTime()));

			if ($dba->dba_insert($dataobj) != true) {
				trigger_error($this->o_resource->getValue("D8_003_01"), E_USER_ERROR);
				return false;
			}
		} else {
			trigger_error($this->o_resource->getValue("D8_003_02"), E_USER_ERROR);
			return false;
		}

		// 解放
		$dba->dba_free();

		return true;
	}

	/**
	 *	関数名：レイヤー優先順位テーブルキー指定データ更新
     * @引数：(I) $db               - DBオブジェクト
     *        (I) $targetDataobj    - 対象列(条件列/更新列)指定用データオブジェクト
     *        (I) $valDataobj       - 設定値(条件値/更新値)指定用データオブジェクト
     * @戻値：$bool			- true/false
     * @作成日/作成者：2006/09/01 (IIS) K.Shimomura
     * @更新日/更新者：
     * @使用方法：whereキーとして使用したい列は$targetDataobjの該当変数をWHERE_KEYに指定し、条件値を$valDataobjの該当変数に使用する。
	 *            (プライマリキー以外にWHERE_KEYを指定しても無視されます)
     *            更新したい列は$targetDataobjの該当変数をVAL_TARGETに指定し、更新値を$valDataobjの該当変数に使用する。
     *            NULLで更新したい時は$valDataobjの該当変数にNULL_CHARを指定する。
     */
	function updateByKeyData($db, $targetDataobj, $valDataobj) {

		$whereflg = false;
		$dba = new KakikoDBAccess($db, $this->getTablename());

		if ((is_object($targetDataobj) && get_class($targetDataobj) == LAYERPRIORITY_DATAOBJ) && 
			(is_object($valDataobj) && get_class($valDataobj) == LAYERPRIORITY_DATAOBJ)) {

			// 条件および更新内容指定

			// 優先順位
			if ($targetDataobj->getPriorityOrder() === WHERE_KEY) {
				$dba->dba_whereAdd("", "priorityorder = ".$valDataobj->getPriorityOrder());
				$whereflg = true;
			} elseif ($targetDataobj->getPriorityOrder() === VAL_TARGET) {
				$dba->dba_updateAdd(
						"priorityorder = "
							.$valDataobj->getPriorityOrder());
			}
			// レイヤーID
			if ($targetDataobj->getLayerID() === VAL_TARGET) {
				$dba->dba_updateAdd(
						"layerid = "
							.$dba->query_str_convert($valDataobj->getLayerID()));
			}
			// 登録日時
			if ($targetDataobj->getAddDateTime() === VAL_TARGET) {
				$dba->dba_updateAdd(
						"adddatetime = "
							.$dba->query_str_convert($valDataobj->getAddDateTime()));
			}
			// 更新日時
			if ($targetDataobj->getUpdateDateTime() === VAL_TARGET) {
				$dba->dba_updateAdd(
						"updatedatetime = "
							.$dba->query_str_convert($valDataobj->getUpdateDateTime()));
			}

			// 条件指定確認
			if ($whereflg == false) {
				trigger_error($this->o_resource->getValue("D8_004_01"), E_USER_ERROR);
				return false;
			}
			//SQL実行
			if ($dba -> dba_update() != true) {
				trigger_error($this->o_resource->getValue("D8_004_02"), E_USER_ERROR);
				return false;
			}
		} else {
			trigger_error($this->o_resource->getValue("D8_004_03"), E_USER_ERROR);
			return false;
		}

		// 解放
		$dba->dba_free();

		return true;

	}

    /**
     * 関数概要：レイヤー優先順位テーブル全データ削除
     * 処理詳細：レイヤー優先順位テーブル全データの削除クエリを生成する
     *
     * @引数： (I) db       - object DBオブジェクト
     * @戻値： boolean true :正常
     *      ：         false:異常
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
	function deleteAllData($db) {

		$dba = new KakikoDBAccess($db, $this->getTablename());

		if ($dba -> dba_delete() != true) {
			trigger_error($this->o_resource->getValue("D8_005_01"), E_USER_ERROR);
			return false;
		}

		// 解放
		$dba->dba_free();

		return true;
	}

    /**
     * 関数概要：レイヤー優先順位テーブルキー指定データ削除
     * 処理詳細：レイヤー優先順位テーブルキー指定データの削除クエリを生成する
     *
     * @引数： (I) db       - object  DBオブジェクト
     *      ： (I) key      - integer キー内容(優先順位)
     * @戻値： boolean true :正常
     *      ：         false:異常
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
	function deleteByKeyData($db, $key) {

		$dba = new KakikoDBAccess($db, $this->getTablename());

		// 条件指定
		$dba->dba_whereAdd("", "priorityorder = ".$dba->query_str_convert($key));

		if ($dba -> dba_delete() != true) {
			trigger_error($this->o_resource->getValue("D8_006_01"), E_USER_ERROR);
			return false;
		}

		// 解放
		$dba->dba_free();

		return true;
	}

    /**
     * 関数概要：一覧データ取得
     * 処理詳細：レコードセットより、レイヤー優先順位リストを生成する
     *
     * @引数： 
     * @戻値： array   レイヤー優先順位リスト
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
    function getDataList() {
        
		$layerpriority_array = array();

		$rs = $this->getRecordSet();

		while($row = $rs->fetchrow(DB_FETCHMODE_ASSOC)) {

			$layerprioritydata = new LayerPriorityData();

			$layerprioritydata->setPriorityOrder(isset($row['priorityorder'])?$row['priorityorder']:null);
			$layerprioritydata->setLayerID(isset($row['layerid'])?$row['layerid']:null);
			$layerprioritydata->setAddDateTime(isset($row['adddatetime'])?$row['adddatetime']:null);
			$layerprioritydata->setUpdateDateTime(isset($row['updatedatetime'])?$row['updatedatetime']:null);

			array_push($layerpriority_array, $layerprioritydata);
		}
 		return $layerpriority_array;
    }

    /**
     * 関数概要：単体データ取得
     * 処理詳細：レコードセットより、一件のレイヤー優先順位データを生成する
     *
     * @引数： 
     * @戻値： array   レイヤー優先順位データ型オブジェクト
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
    function getOneData() {

		$rs = $this->getRecordSet();

		$row = $rs->fetchrow(DB_FETCHMODE_ASSOC);

		$layerprioritydata = new LayerPriorityData();

		$layerprioritydata->setPriorityOrder(isset($row['priorityorder'])?$row['priorityorder']:null);
		$layerprioritydata->setLayerID(isset($row['layerid'])?$row['layerid']:null);
		$layerprioritydata->setAddDateTime(isset($row['adddatetime'])?$row['adddatetime']:null);
		$layerprioritydata->setUpdateDateTime(isset($row['updatedatetime'])?$row['updatedatetime']:null);

 		return $layerprioritydata;
    }

    /**
     * 関数概要：レイヤー優先順位テーブルキー指定重複チェック
     * 処理詳細：ベースマップテーブルデータをキー指定で重複しているかをチェック
     *
     * @引数： (I) db       - object  DBオブジェクト
     *      ： (I) key      - integer キー内容(優先順位)
     * @戻値： boolean true :データなし
     *      ：         false:データあり
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
	function checkFindByKeyData($db, $key) {

		$rtn = true;

		// キー指定データの取得
		$this->selectFindByKeyData($db, $key);

		if ($this->getRecordCount() > 0) {
			$rtn = false;	
		}

		return $rtn;
	}
}
?>