<?php
/**
* ContentsAdminService.class.php コンテンツ管理系用DBアクセスサービス
* PHP versions：4.4.2
* @パッケージ：-
* @作者：K.Shimomura
* @作成日/作成者：2006/09/01 (IIS)K.Shimomura
* @更新日/更新者：
* @バージョン：1.0.0
*
* Copyright (c) 2006 Iwate Information System Co.,Ltd. All Rights Reserved.       
* You may see the License at LICENSE_EN.txt(English) or LICENSE_JA.txt(Japanese). 
*/

require_once("class/db/dbaccess/BaseMapJoinMapDetailTableAccess.class.php");
require_once("class/db/dbaccess/LayerTableAccess.class.php");
require_once("class/db/dbaccess/LayerUserTableAccess.class.php");
require_once("class/db/dbaccess/MapTableAccess.class.php");
require_once("class/db/dbaccess/MemoDetailTableAccess.class.php");
require_once("class/db/dbaccess/MemoTypeTableAccess.class.php");
require_once("class/db/dbaccess/UserTableAccess.class.php");
require_once("class/db/service/KakikoService.class.php");
require_once("class/db/dbaccess/ImiBaseInfoTableAccess.class.php");
require_once("class/db/dbaccess/ImiLayerTableAccess.class.php");
require_once("class/db/dbaccess/ImiMemoPositionTableAccess.class.php");
require_once("class/db/tabledata/LayerUserData.class.php");
require_once("class/xml/RSSController.class.php");
require_once("class/db/dbaccess/UserJoinLayerTableAccess.class.php");
require_once("class/db/dbaccess/AdminUserJoinMapTableAccess.class.php");


class ContentsAdminService extends KakikoService {

    /**
     * コンストラクタ
     * @引数： (I) directryname  - string ディレクトリ名称
     */
    function ContentsAdminService($directryname = "") {

		// 基底クラスのコンストラクタ呼び出し
    	$this->KakikoService($directryname);

		// DB接続情報の設定
		$this->setDBConnectInfo();
    }

    /**
     * 関数概要：レイヤーテーブルキー指定重複チェック
     * 処理詳細：レイヤーテーブルに対してキー指定で重複データチェックを
     *         ：行うクエリ作成を呼び出す
     *
     * @引数： (I) key       - string レイヤーテーブルキー(レイヤーID)
     * @戻値： boolean true :データなし
     *      ：         false:データあり
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
	function checkLayerDataByKey($key) {

		// DBオブジェクトの生成
		$this->createDBObject();

		$table_access = new LayerTableAccess($this->directryname);

		// レイヤーデータのキー指定チェック
		$rtn = $table_access->checkFindByKeyData($this->o_db, $key);

		// 解放
		$table_access->free();

		// DB接続の切断
		$this->disconnectDB();

		return $rtn;
	}

    /**
     * 関数概要：ユーザテーブルキー指定重複チェック
     * 処理詳細：ユーザテーブルに対してキー指定で重複データチェックを
     *         ：行うクエリ作成を呼び出す
     *
     * @引数： (I) key       - string ユーザテーブルキー(ユーザID)
     * @戻値： boolean true :データなし
     *      ：         false:データあり
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
	function checkUserDataByKey($key) {

		// DBオブジェクトの生成
		$this->createDBObject();

		$table_access = new UserTableAccess($this->directryname);

		// ユーザデータのキー指定チェック
		$rtn = $table_access->checkFindByKeyData($this->o_db, $key);

		// 解放
		$table_access->free();

		// DB接続の切断
		$this->disconnectDB();

		return $rtn;
	}

    /**
     * 関数概要：レイヤーユーザテーブルキー指定重複チェック
     * 処理詳細：レイヤーユーザテーブルに対してキー指定で重複データチェックを
     *         ：行うクエリ作成を呼び出す
     *
     * @引数： (I) key1      - string レイヤーユーザテーブルキー(ユーザID)
     *      ： (I) key2      - string レイヤーユーザテーブルキー(レイヤーID)
     * @戻値： boolean true :データなし
     *      ：         false:データあり
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
	function checkLayerUserDataByKey($key1, $key2) {

		// DBオブジェクトの生成
		$this->createDBObject();

		$table_access = new LayerUserTableAccess($this->directryname);

		// レイヤーユーザデータのキー指定チェック
		$rtn = $table_access->checkFindByKeyData($this->o_db, $key1, $key2);

		// 解放
		$table_access->free();

		// DB接続の切断
		$this->disconnectDB();

		return $rtn;
	}

    /**
     * 関数概要：メモ詳細テーブルメモ種類ID指定重複チェック
     * 処理詳細：レイヤーユーザテーブルに対してメモ種類ID指定で重複データチェックを
     *         ：行うクエリ作成を呼び出す
     *
     * @引数： (I) memotypeid    - integer メモ種類ID
     * @戻値： boolean true :データなし
     *      ：         false:データあり
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
	function checkMemoDetailDataByMemoTypeID($memotypeid) {

		// DBオブジェクトの生成
		$this->createDBObject();

		$table_access = new MemoDetailTableAccess($this->directryname);

		// ユーザデータのキー指定チェック
		$rtn = $table_access->checkFindByMemoTypeIDData($this->o_db, $memotypeid);

		// 解放
		$table_access->free();

		// DB接続の切断
		$this->disconnectDB();

		return $rtn;
	}

    /**
     * 関数概要：レイヤーデータリスト取得
     * 処理詳細：レイヤーテーブルとユーザテーブルをリンクしたデータの取得を
     *         ：行うクエリ作成を呼び出す
     *
     * @引数： (I) userid    - string ユーザID
     * @戻値： array   レイヤーリンクユーザデータクラス
     *      ：         DBエラーの場合はnull
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
	function getLayerDataList($userid) {

		// DBオブジェクトの生成
		$this->createDBObject();

		// SQL:	SELECT directry_layertable.layerid, 
		//				directry_layertable.layername,
		//				directry_layertable.layernote,
		//				directry_layertable.adddatetime,
		//				directry_layertable.statusflg
		//		FROM   directry_layertable, directry_usertable
		//		WHERE  directry_layertable.userid = directry_usertable.userid
		//			AND  directry_usertable.userid = ?
		//		ORDER BY directry_layertable.layerid

		// クエリー条件(ユーザ種別)
		$usertype = array(KAKIKOMAP_USERTYPE_NORMAL);

		// レイヤーテーブルDBアクセス
		$table_access = new LayerTableAccess($this->directryname);

		// クエリー発行
		$rtn = $table_access->selectLayerDataInUserData($this->o_db, $userid, $usertype);
		if ($rtn != true) {
			$table_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S3_001_01"), E_USER_ERROR);
			return null;
		}

		// データオブジェクトの取得
		$array_data = $table_access->getDataList();

		// 解放
		$table_access->free();

		// DB接続の切断
		$this->disconnectDB();

		return $array_data;

	}

    /**
     * 関数概要：ユーザデータリスト取得
     * 処理詳細：ユーザテーブルとレイヤーユーザテーブルをリンクしたデータの取得を
     *         ：行うクエリ作成を呼び出す
     *
     * @引数： (I) userid    - string ユーザID
     *      ： (I) layerid   - string レイヤーID
     *      ： (I) type      - string 取得状態フラグ 0:認証済み
     *      ：                                       1:許可待ち
     *      ： (I) userName  - string 検索ユーザ名称
     * @戻値： array   ユーザリンクレイヤーユーザデータクラス
     *      ：         DBエラーの場合はnull
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
	function getUserDataList($userid, $layerid, $type, $userName) {

		// DBオブジェクトの生成
		$this->createDBObject();

		// SQL:	SELECT directry_usertable.layerid, 
		//				directry_usertable.layername,
		//				directry_usertable.layernote,
		//				directry_layerusertable.adddatetime,
		//				directry_layerusertable.statusflg
		//		FROM   directry_usertable, directry_layerusertable 
		//		WHERE  directry_usertable.userid = directry_layerusertable.userid
		//			AND  directry_usertable.usertype IN('3')
		//			AND	 directry_layerusertable.statusflg IN ('0', '1')
		//			AND	 directry_layerusertable.layerid = ?
		//			AND	 directry_usertable.userid != ?
		//			AND  directry_usertable.username LIKE '%?%' 
		//		ORDER BY directry_usertable.userid

		// クエリー条件(ユーザ種別)
		$usertype = array(KAKIKOMAP_USERTYPE_NORMAL);

		// クエリー条件(状態フラグ)
		$statusflg = $type == 0 
						? array(KAKIKOMAP_STATUSFLG_ON, KAKIKOMAP_STATUSFLG_OFF) 
								: array(KAKIKOMAP_STATUSFLG_WAIT);

		// ユーザ管理テーブルDBアクセス
		$table_access = new UserTableAccess($this->directryname);

		// クエリー発行
		$rtn = $table_access->selectUserDataInLayerUserData(
								$this->o_db, $userid, $layerid, $usertype, $statusflg, $userName);
		if ($rtn != true) {
			$table_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S3_002_01"), E_USER_ERROR);
			return null;
		}

		// データオブジェクトの取得
		$array_data = $table_access->getDataList();

		// 解放
		$table_access->free();

		// DB接続の切断
		$this->disconnectDB();

		return $array_data;

	}

    /**
     * 関数概要：レイヤー情報データ取得
     * 処理詳細：レイヤーテーブルのデータの取得を
     *         ：行うクエリ作成を呼び出す
     *
     * @引数： (I) userid    - string ユーザID
     *      ： (I) layerid   - string レイヤーID
     * @戻値： array   レイヤーデータクラス
     *      ：         DBエラーの場合はnull
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
	function getLayerInfoData($userid, $layerid) {

		// DBオブジェクトの生成
		$this->createDBObject();

		// SQL:	SELECT directry_layertable.layerid,
		//				directry_layertable.layername,
		//				 directry_layertable.layernote,
		//				 directry_layertable.publiclayer,
		//				 directry_layertable.receiptlayer,
		//				 directry_layertable.addattestuser,
		//				 directry_layertable.directlink
		//		FROM directry_layertable 
		//		WHERE directry_layertable.userid = ?
		//			AND directry_layertable.layerid = ?

		// レイヤー管理テーブルDBアクセス
		$table_access = new LayerTableAccess($this->directryname);

		// クエリー発行
		$rtn = $table_access->selectFindByKeyData($this->o_db, $userid, $layerid);
		if ($rtn != true) {
			$table_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S3_003_01"), E_USER_ERROR);
			return null;
		}

		if ($table_access->getRecordCount() == 0){
			return null;
		}

		// データオブジェクトの取得
		$array_data = $table_access->getOneData();

		// 解放
		$table_access->free();

		// DB接続の切断
		$this->disconnectDB();

		return $array_data;

	}

    /**
     * 関数概要：メモ種類データリスト取得
     * 処理詳細：メモ種類テーブルのデータの取得を
     *         ：行うクエリ作成を呼び出す
     *
     * @引数： (I) layerid   - string レイヤーID
     * @戻値： array   メモ種類データクラス
     *      ：         DBエラーの場合はnull
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
	function getMemoTypeDataList($layerid) {

		// DBオブジェクトの生成
		$this->createDBObject();

		// SQL:	SELECT directry_memotypetable.memotypeid, 
		//				directry_memotypetable.memotypename,
		//				directry_memotypetable.publicmemotype,
		//				directry_memotypetable.authoritymemotype,
		//				directry_memotypetable.icontype,
		//				directry_memotypetable.icon,
		//				directry_memotypetable.ribboncolor,
		//				directry_memotypetable.listtype,
		//				directry_memotypetable.effectiveterm
		//		FROM   directry_memotypetable 
		//		WHERE  directry_memotypetable.layerid = ?
		//		ORDER BY directry_memotypetable.memotypeid

		// メモ種類テーブルDBアクセス
		$table_access = new MemoTypeTableAccess($this->directryname);

		// クエリー発行
		$rtn = $table_access->selectMemoTypeListData($this->o_db, $layerid);
		if ($rtn != true) {
			$table_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S3_004_01"), E_USER_ERROR);
			return null;
		}

		// データオブジェクトの取得
		$array_data = $table_access->getDataList();

		// 解放
		$table_access->free();

		// DB接続の切断
		$this->disconnectDB();

		return $array_data;

	}

    /**
     * 関数概要：メモ種類情報データ取得
     * 処理詳細：メモ種類テーブルのデータの取得を
     *         ：行うクエリ作成を呼び出す
     *
     * @引数： (I) memotypeid  - integer メモ種類ID
     * @戻値： array   メモ種類データクラス
     *      ：         DBエラーの場合はnull
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
	function getMemoTypeInfoData($memotypeid) {

		// DBオブジェクトの生成
		$this->createDBObject();

		// SQL:	SELECT directry_memotypetable.memotypeid,
		//				directry_memotypetable.memotypename,
		//				 directry_memotypetable.publicmemotype,
		//				 directry_memotypetable.authoritymemotype,
		//				 directry_memotypetable.icontype,
		//				 directry_memotypetable.icon,
		//				 directry_memotypetable.ribboncolor,
		//				 directry_memotypetable.listtype,
		//				 directry_memotypetable.effectiveterm
		//		FROM directry_memotypetable 
		//		WHERE directry_memotypetable.memotypeid = ?

		// メモ種類管理テーブルDBアクセス
		$table_access = new MemoTypeTableAccess($this->directryname);

		// クエリー発行
		$rtn = $table_access->selectFindByKeyData($this->o_db, $memotypeid);
		if ($rtn != true) {
			$table_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S3_005_01"), E_USER_ERROR);
			return null;
		}

		if ($table_access->getRecordCount() == 0){
			return null;
		}

		// データオブジェクトの取得
		$array_data = $table_access->getOneData();

		// 解放
		$table_access->free();

		// DB接続の切断
		$this->disconnectDB();

		return $array_data;

	}

	/**
	 *	関数名：レイヤー登録
     * @引数：(I/O) layerDataobj       - レイヤーテーブル用データオブジェクト
     *        (I)   layeruserDataobj   - レイヤーユーザマップテーブル用データオブジェクト
     * @戻値：integer		- 1  正常登録
     *						- 0  チェックエラー
     *                      - -1 異常  
     * @作成日/作成者：2006/09/01 (IIS) K.Shimomura
     * @更新日/更新者：
     * @使用方法：レイヤー登録を行います。
	 *            
     */
	function insertLayer(&$layerDataobj, $layeruserDataobj) {

		$NoError = 1;
		$Error = -1;
		$Notice = 0;

		// DBオブジェクトの生成
		$this->createDBObject();

		// トランザクション開始
		$this->o_db->autoCommit(false);

		// -----------------------------------------------
		// マップ管理テーブルの「レイヤー登録認証区分」を参照
		// -----------------------------------------------

		$maptable_access = new MapTableAccess();

		// SQL:	SELECT *
		//		FROM   maptable
		//		WHERE  directryname = ?

		// クエリ発行
		$rtn = $maptable_access->selectDirectryNameKeyData($this->o_db, $this->directryname);
		if ($rtn != true) {
			$this->o_db->rollback();
			$maptable_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S3_006_01"), E_USER_ERROR);

			return $Error;
		}

		// 存在チェック(整合性チェック)
		if ($maptable_access->getRecordCount() != 1) {
			$this->o_db->rollback();
			$maptable_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S3_006_02"), E_USER_ERROR);

			return $Error;
		}

		// マップデータの取得
		$data_array = array();
		$data_array = $maptable_access->getOneData();

	    $addattestlayer = $data_array->getAddAttestLayer();
	    $statusflg = $addattestlayer == KAKIKOMAP_ADDATTESTLAYER_OFF
	    					? KAKIKOMAP_STATUSFLG_ON 
	    						: KAKIKOMAP_STATUSFLG_WAIT;

		// -----------------------------------------------
		// レイヤーテーブルの件数をチェック
		// -----------------------------------------------

		$layertable_access = new LayerTableAccess($this->directryname);

		// クエリ発行
		$rtn = $layertable_access->selectAllData($this->o_db);
		if ($rtn != true) {
			$this->o_db->rollback();
			$maptable_access->free();
			$layertable_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S3_006_03"), E_USER_ERROR);

			return $Error;
		}

		// マップデータの登録可能レイヤー数と現レイヤー数の比較を行い
		// 登録可能数を上回る場合は、登録できない
		if ($layertable_access->getRecordCount() >= $data_array->getLayerCount()) {
			$this->o_db->rollback();
			$maptable_access->free();
			$layertable_access->free();
			$this->disconnectDB();

			return $Notice;
		}

		// -----------------------------------------------
		// レイヤーテーブルへ新規にレコードを追加
		// -----------------------------------------------

		// 状態フラグの設定
		$layerDataobj->setStatusFlg($statusflg);

		// データ追加
		$rtn = $layertable_access->insertData($this->o_db, $layerDataobj);
		if ($rtn != true) {
			$this->o_db->rollback();
			$maptable_access->free();
			$layertable_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S3_006_04"), E_USER_ERROR);

			return $Error;
		}

        // -----------------------------------------------
        // RSSファイルの作成
        // -----------------------------------------------
        $rss_control_obj = new RSSController($layerDataobj->getLayerID(), $this->directryname);
        $rss_control_obj->writeRssFile();

		// -----------------------------------------------
		// レイヤーユーザテーブルへ新規にレコードを追加
		// -----------------------------------------------

		$layerusertable_access = new LayerUserTableAccess($this->directryname);

		// データ追加
		$rtn = $layerusertable_access->insertData($this->o_db, $layeruserDataobj);
		if ($rtn != true) {
			$this->o_db->rollback();
			$maptable_access->free();
			$layertable_access->free();
			$layerusertable_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S3_006_05"), E_USER_ERROR);

			return $Error;
		}

        // マップ管理者のユーザIDの取得

        // ユーザ管理テーブルDBアクセス
        $usertable_access = new UserTableAccess($this->directryname);

        // データ取得
        $rtn = $usertable_access->selectMapAdminData($this->o_db);
        if ($rtn != true) {
            $this->o_db->rollback();
            $maptable_access->free();
            $layertable_access->free();
            $layerusertable_access->free();
            $usertable_access->free();
            $this->disconnectDB();
            trigger_error($this->o_resource->getValue("S3_006_06"), E_USER_ERROR);
            return $Error;
        }

        if ($usertable_access->getRecordCount() == 0) {
            // マップ管理者データが存在しない(データ不整合)
            trigger_error($this->o_resource->getValue("S3_006_07"), E_USER_ERROR);
        }

        // データオブジェクトの取得
        $user_array_data = $usertable_access->getOneData();

        if ($layeruserDataobj->getUserID() != $user_array_data->getUserID()) {
            // レイヤーユーザデータ型オブジェクト
            $m_layerUserDataObj = new LayerUserData();

            $m_layerUserDataObj->setUserID($user_array_data->getUserID());
            $m_layerUserDataObj->setLayerID($layeruserDataobj->getLayerID());
            $m_layerUserDataObj->setStatusFlg($layeruserDataobj->getStatusFlg());
            $m_layerUserDataObj->setAddDateTime($layeruserDataobj->getAddDateTime());

		    // データ追加
		    $rtn = $layerusertable_access->insertData($this->o_db, $m_layerUserDataObj);
		    if ($rtn != true) {
			    $this->o_db->rollback();
			    $maptable_access->free();
			    $layertable_access->free();
			    $layerusertable_access->free();
			    $usertable_access->free();
			    $this->disconnectDB();
			    trigger_error($this->o_resource->getValue("S3_006_08"), E_USER_ERROR);

			    return $Error;
		    }
		}

		// 解放
		$maptable_access->free();
		$layertable_access->free();
		$layerusertable_access->free();
		$usertable_access->free();

		// トランザクションコミット
		$this->o_db->commit();

		// DB接続の切断
		$this->disconnectDB();

		return $NoError;

	}

	/**
	 *	関数名：ユーザ登録
     * @引数：(I) userDataobj        - ユーザテーブル用データオブジェクト
     *        (I) layeruserDataobj   - レイヤーユーザマップテーブル用データオブジェクト
     * @戻値：bool			- true  正常登録
     *                      - false 異常
     * @作成日/作成者：2006/09/01 (IIS) K.Shimomura
     * @更新日/更新者：
     * @使用方法：ユーザ登録を行います。
	 *            
     */
	function insertUser($userDataobj, $layeruserDataobj) {

		// DBオブジェクトの生成
		$this->createDBObject();

		// トランザクション開始
		$this->o_db->autoCommit(false);

		// -----------------------------------------------
		// ユーザテーブルへ新規にレコードを追加
		// -----------------------------------------------

		// ユーザDBアクセスインスタンス生成
		$usertable_access = new UserTableAccess($this->directryname);

		if (!is_null($userDataobj)) {
			// データ追加
			$rtn = $usertable_access->insertData($this->o_db, $userDataobj);
			if ($rtn != true) {
				$this->o_db->rollback();
				$usertable_access->free();
				$this->disconnectDB();
				trigger_error($this->o_resource->getValue("S3_007_01"), E_USER_ERROR);
	
				return false;
			}
		}

		// -----------------------------------------------
		// レイヤーユーザテーブルへ新規にレコードを追加
		// -----------------------------------------------

		$layerusertable_access = new LayerUserTableAccess($this->directryname);

		if (!is_null($layeruserDataobj)) {
			// データ追加
			$rtn = $layerusertable_access->insertData($this->o_db, $layeruserDataobj);
			if ($rtn != true) {
				$this->o_db->rollback();
				$usertable_access->free();
				$layerusertable_access->free();
				$this->disconnectDB();
				trigger_error($this->o_resource->getValue("S3_007_02"), E_USER_ERROR);
	
				return false;
			}
		}

		// 解放
		$usertable_access->free();
		$layerusertable_access->free();

		// トランザクションコミット
		$this->o_db->commit();

		// DB接続の切断
		$this->disconnectDB();

		return true;

	}

	/**
	 *	関数名：メモ種類登録
     * @引数：(I) memotypeDataobj   - メモ種類テーブル用データオブジェクト
     * @戻値：integer		- 1  正常登録
     *						- 0  チェックエラー
     *                      - -1 異常
     * @作成日/作成者：2006/09/01 (IIS) K.Shimomura
     * @更新日/更新者：
     * @使用方法：メモ種類登録を行います。
	 *            
     */
	function insertMemoType($memotypeDataobj) {

		$NoError = 1;
		$Error = -1;
		$Notice = 0;

		// DBオブジェクトの生成
		$this->createDBObject();

		// トランザクション開始
		$this->o_db->autoCommit(false);

		// -----------------------------------------------
		// マップ管理テーブルを参照
		// -----------------------------------------------

		$maptable_access = new MapTableAccess();

		// SQL:	SELECT *
		//		FROM   maptable
		//		WHERE  directryname = ?

		// クエリ発行
		$rtn = $maptable_access->selectDirectryNameKeyData($this->o_db, $this->directryname);
		if ($rtn != true) {
			$this->o_db->rollback();
			$maptable_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S3_008_01"), E_USER_ERROR);

			return $Error;
		}

		// 存在チェック(整合性チェック)
		if ($maptable_access->getRecordCount() != 1) {
			$this->o_db->rollback();
			$maptable_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S3_008_02"), E_USER_ERROR);

			return $Error;
		}

		// マップデータの取得
		$data_array = array();
		$data_array = $maptable_access->getOneData();

		// -----------------------------------------------
		// メモ種類テーブルの件数をチェック
		// -----------------------------------------------

		$memotypetable_access = new MemoTypeTableAccess($this->directryname);

		// クエリ発行
		$rtn = $memotypetable_access->selectMemoTypeListData(
									$this->o_db, $memotypeDataobj->getLayerID());
		if ($rtn != true) {
			$this->o_db->rollback();
			$maptable_access->free();
			$memotypetable_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S3_008_03"), E_USER_ERROR);

			return $Error;
		}

		// マップデータの登録可能メモ種類数と現メモ種類数の比較を行い
		// 登録可能数を上回る場合は、登録できない
		if ($memotypetable_access->getRecordCount() >= $data_array->getMemoTypeCount()) {
			$this->o_db->rollback();
			$maptable_access->free();
			$memotypetable_access->free();
			$this->disconnectDB();

			return $Notice;
		}

		// -----------------------------------------------
		// メモ種類テーブルへ新規にレコードを追加
		// -----------------------------------------------

		// データ追加
		$rtn = $memotypetable_access->insertData($this->o_db, $memotypeDataobj);
		if ($rtn != true) {
			$this->o_db->rollback();
			$maptable_access->free();
			$memotypetable_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S3_008_04"), E_USER_ERROR);

			return $Error;
		}

		// 解放
		$maptable_access->free();
		$memotypetable_access->free();

		// トランザクションコミット
		$this->o_db->commit();

		// DB接続の切断
		$this->disconnectDB();

		return $NoError;

	}

	/**
	 *	関数名：レイヤーテーブルデータキー更新
     * @引数：(I) targetDataobj    - 対象列(条件列/更新列)指定用データオブジェクト
     *        (I) valDataobj       - 設定値(条件値/更新値)指定用データオブジェクト
     * @戻値：bool			- true  正常更新
     *                      - false 異常
     * @作成日/作成者：2006/09/01 (IIS) K.Shimomura
     * @更新日/更新者：
     * @使用方法：レイヤーテーブルデータをキー(ユーザID・レイヤーID)で更新を行います。
	 *            
     */
	function updateLayerDataByKey($targetDataobj, $valDataobj) {

		// DBオブジェクトの生成
		$this->createDBObject();

		// トランザクション開始
		$this->o_db->autoCommit(false);

		// レイヤーDBアクセスインスタンス生成
		$table_access = new LayerTableAccess($this->directryname);

		// レイヤーデータ更新
		$rtn = $table_access->updateByKeyData($this->o_db, $targetDataobj, $valDataobj);

		if ($rtn != true) {
			$this->o_db->rollback();
			$table_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S3_009_01"), E_USER_ERROR);
			return false;
		}

		// 解放
		$table_access->free();

		// トランザクションコミット
		$this->o_db->commit();

		// DB接続の切断
		$this->disconnectDB();
		
		return true;

	}

	/**
	 *	関数名：レイヤーユーザテーブルデータキー更新
     * @引数：(I) targetDataobj    - 対象列(条件列/更新列)指定用データオブジェクト
     *        (I) valDataobj       - 設定値(条件値/更新値)指定用データオブジェクト
     * @戻値：bool			- true  正常更新
     *                      - false 異常
     * @作成日/作成者：2006/09/01 (IIS) K.Shimomura
     * @更新日/更新者：
     * @使用方法：レイヤーユーザテーブルデータをキー(ユーザID・レイヤーID)で更新を行います。
	 *            
     */
	function updateLayerUserDataByKey($targetDataobj, $valDataobj) {

		// DBオブジェクトの生成
		$this->createDBObject();

		// トランザクション開始
		$this->o_db->autoCommit(false);

		// レイヤーユーザDBアクセスインスタンス生成
		$table_access = new LayerUserTableAccess($this->directryname);

		// レイヤーユーザデータ更新
		$rtn = $table_access->updateByKeyData($this->o_db, $targetDataobj, $valDataobj);

		if ($rtn != true) {
			$this->o_db->rollback();
			$table_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S3_010_01"), E_USER_ERROR);
			return false;
		}

		// 解放
		$table_access->free();

		// トランザクションコミット
		$this->o_db->commit();

		// DB接続の切断
		$this->disconnectDB();
		
		return true;

	}

	/**
	 *	関数名：メモ種類テーブルデータキー更新
     * @引数：(I) targetDataobj    - 対象列(条件列/更新列)指定用データオブジェクト
     *        (I) valDataobj       - 設定値(条件値/更新値)指定用データオブジェクト
     * @戻値：bool			- true  正常更新
     *                      - false 異常
     * @作成日/作成者：2006/09/01 (IIS) K.Shimomura
     * @更新日/更新者：
     * @使用方法：メモ種類テーブルデータをキー(メモ種類ID)で更新を行います。
	 *            
     */
	function updateMemoTypeDataByKey($targetDataobj, $valDataobj) {

		// DBオブジェクトの生成
		$this->createDBObject();

		// トランザクション開始
		$this->o_db->autoCommit(false);

		// メモ種類DBアクセスインスタンス生成
		$table_access = new MemoTypeTableAccess($this->directryname);

		// メモ種類データ更新
		$rtn = $table_access->updateByKeyData($this->o_db, $targetDataobj, $valDataobj);

		if ($rtn != true) {
			$this->o_db->rollback();
			$table_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S3_011_01"), E_USER_ERROR);
			return false;
		}

		// 解放
		$table_access->free();

		// トランザクションコミット
		$this->o_db->commit();

		// DB接続の切断
		$this->disconnectDB();
		
		return true;

	}

	/**
	 *	関数名：レイヤーテーブルデータキー削除
     * @引数：(I) userid    - ユーザID
     *        (I) layerid   - レイヤーID
     * @戻値：bool			- true  正常更新
     *                      - false 異常
     * @作成日/作成者：2006/09/01 (IIS) K.Shimomura
     * @更新日/更新者：
     * @使用方法：レイヤーテーブルデータをキー(ユーザID・レイヤーID)で削除を行います。
	 *            
     */
	function deleteLayerByKey($userid, $layerid) {

		// DBオブジェクトの生成
		$this->createDBObject();

		// トランザクション開始
		$this->o_db->autoCommit(false);

        // -----------------------------------------------
        // RSSファイルの削除
        // -----------------------------------------------
        $rss_control_obj = new RSSController($layerid, $this->directryname);
        $rss_control_obj->deleteRSSFile();

		// レイヤーDBアクセスインスタンス生成
		$table_access = new LayerTableAccess($this->directryname);

		// レイヤーデータ削除
		$rtn = $table_access->deleteByKeyData($this->o_db, $userid, $layerid);

		if ($rtn != true) {
			$this->o_db->rollback();
			$table_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S3_012_01"), E_USER_ERROR);
			return false;
		}

		// 意味空間で使用されているレイヤーのチェックと削除
		$rtn = $this->checkAndUpdateImiBaseInfo($layerid, null, null, 2);
		if ($rtn != true) {
			$this->o_db->rollback();
			$table_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S3_012_01"), E_USER_ERROR);
			return false;
		}
		$imilayer_access = new ImiLayerTableAccess($this->directryname);
		$rtn = $imilayer_access->deleteByKeyData($this->o_db, null,$layerid);		
		if ($rtn != true) {
			$this->o_db->rollback();
			$table_access->free();
			$imilayer_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S3_012_01"), E_USER_ERROR);
			return false;
		}
		$imilayer_access->free();
		$imimemoposition_access = new ImiMemoPositionTableAccess($this->directryname);
		$rtn = $imimemoposition_access->deleteDataByLayer($this->o_db, $layerid);		
		if ($rtn != true) {
			$this->o_db->rollback();
			$table_access->free();
			$imimemoposition_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S3_012_01"), E_USER_ERROR);
			return false;
		}
		$imimemoposition_access->free();
		
		
		// 解放
		$table_access->free();

		// トランザクションコミット
		$this->o_db->commit();

		// DB接続の切断
		$this->disconnectDB();
		
		return true;

	}

	/**
	 *	関数名：レイヤーユーザテーブルデータキー削除
     * @引数：(I) userid    - ユーザID
     *        (I) layerid   - レイヤーID
     * @戻値：bool			- true  正常更新
     *                      - false 異常
     * @作成日/作成者：2006/09/01 (IIS) K.Shimomura
     * @更新日/更新者：
     * @使用方法：レイヤーユーザテーブルデータをキー(ユーザID・レイヤーID)で削除を行います。
	 *            
     */
	function deleteLayerUserByKey($userid, $layerid) {

		// DBオブジェクトの生成
		$this->createDBObject();

		// トランザクション開始
		$this->o_db->autoCommit(false);

		// レイヤーユーザDBアクセスインスタンス生成
		$table_access = new LayerUserTableAccess($this->directryname);

		// レイヤーユーザデータ削除
		$rtn = $table_access->deleteByKeyData($this->o_db, $userid, $layerid);

		if ($rtn != true) {
			$this->o_db->rollback();
			$table_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S3_013_01"), E_USER_ERROR);
			return false;
		}

		// 解放
		$table_access->free();

		// トランザクションコミット
		$this->o_db->commit();

		// DB接続の切断
		$this->disconnectDB();
		
		return true;

	}

	/**
	 *	関数名：メモ種類テーブルデータキー削除
     * @引数：(I) memotypeid  - メモ種類ID
     *        (I) icontype    - アイコン種別
     *        (I) icon        - アイコン
     * @戻値：bool			- true  正常更新
     *                      - false 異常
     * @作成日/作成者：2006/09/01 (IIS) K.Shimomura
     * @更新日/更新者：
     * @使用方法：レイヤーユーザテーブルデータをキー(ユーザID・レイヤーID)で削除を行います。
	 *            
     */
	function deleteMemoTypeByKey($memotypeid, $icontype, $icon) {

		// DBオブジェクトの生成
		$this->createDBObject();

		// トランザクション開始
		$this->o_db->autoCommit(false);

		// メモ種別DBアクセスインスタンス生成
		$table_access = new MemoTypeTableAccess($this->directryname);

		// メモ種類データ削除
		$rtn = $table_access->deleteByKeyData($this->o_db, $memotypeid);

		if ($rtn != true) {
			$this->o_db->rollback();
			$table_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S3_014_01"), E_USER_ERROR);
			return false;
		}
		
		// 意味空間で使用されているメモ種類のチェックと削除
		$rtn = $this->checkAndUpdateImiBaseInfo(null, $memotypeid, null, 2);
		if ($rtn != true) {
			$this->o_db->rollback();
			$table_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S3_014_01"), E_USER_ERROR);
			return false;
		}
		$imimemoposition_access = new ImiMemoPositionTableAccess($this->directryname);
		$rtn = $imimemoposition_access->deleteDataByMemotype($this->o_db, $memotypeid);		
		if ($rtn != true) {
			$this->o_db->rollback();
			$table_access->free();
			$imimemoposition_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S3_014_01"), E_USER_ERROR);
			return false;
		}
		$imimemoposition_access->free();


		// 解放
		$table_access->free();

		// アップロードファイル削除
		$this->deleteUploadFile($icontype, $icon);

		// トランザクションコミット
		$this->o_db->commit();

		// DB接続の切断
		$this->disconnectDB();
		
		return true;

	}

    /**
     * 関数概要：レイヤーID指定レイヤーデータ取得
     * 処理詳細：レイヤーテーブルからレイヤーID指定のデータの取得を
     *         ：行うクエリ作成を呼び出す
     *
     * @引数： (I) layerid  - string レイヤーID
     * @戻値： array   レイヤーデータクラス
     *      ：         DBエラーの場合はnull
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
	function getLayerDataFindByLayerID($layerid) {

		// DBオブジェクトの生成
		$this->createDBObject();

		// レイヤーテーブルDBアクセス
		$table_access = new LayerTableAccess($this->directryname);

		// クエリー発行
		$rtn = $table_access->selectLayerIDKeyData($this->o_db, $layerid);
		if ($rtn != true) {
			$table_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S3_015_01"), E_USER_ERROR);
			return null;
		}

		if ($table_access->getRecordCount() == 0){
			return null;
		}

		// データオブジェクトの取得
		$array_data = $table_access->getOneData();

		// 解放
		$table_access->free();

		// DB接続の切断
		$this->disconnectDB();

		return $array_data;

	}

    /**
     * 関数概要：レイヤー情報マップ詳細データ取得
     * 処理詳細：ベースマップテーブルとマップ詳細テーブルをリンクしたデータの取得を
     *         ：行うクエリ作成を呼び出す
     *
     * @引数： 
     * @戻値： array   ベースマップリンクマップ詳細データクラス
     *      ：         DBエラーの場合はnull
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
	function getLayerInfoMapDetailData() {

		// DBオブジェクトの生成
		$this->createDBObject();

		// ベースマップジョインマップ詳細テーブルDBアクセス
		$table_access = new BaseMapJoinMapDetailTableAccess();

		// クエリー発行
		$rtn = $table_access->selectMapLinkMapDetailListData($this->o_db, $this->directryname);
		if ($rtn != true) {
			$table_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S3_016_01"), E_USER_ERROR);
			return null;
		}

		// データオブジェクトの取得
		$array_data = $table_access->getDataList();

		// DB接続の切断
		$this->disconnectDB();

		return $array_data;

	}

    /**
     * 関数概要：ユーザ情報データ取得
     * 処理詳細：ユーザテーブルデータの取得を
     *         ：行うクエリ作成を呼び出す
     *
     * @引数： (I) userid  - string ユーザID
     * @戻値： array   ユーザデータクラス
     *      ：         DBエラーの場合はnull
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
	function getUserInfoData($userid) {

		// DBオブジェクトの生成
		$this->createDBObject();

		// ユーザ管理テーブルDBアクセス
		$table_access = new UserTableAccess($this->directryname);

		// クエリー発行
		$rtn = $table_access->selectFindByKeyData($this->o_db, $userid);
		if ($rtn != true) {
			$table_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S3_017_01"), E_USER_ERROR);
			return null;
		}

		if ($table_access->getRecordCount() == 0){
			return null;
		}

		// データオブジェクトの取得
		$array_data = $table_access->getOneData();

		// 解放
		$table_access->free();

		// DB接続の切断
		$this->disconnectDB();

		return $array_data;

	}

    /**
     * 関数概要：メール送信者情報取得
     *
     * @引数： (I) layerid  	- string レイヤーID
     *         (O) userName  	- ユーザ名
     *         (O) mailAddress  - アドレス
     *         (O) contentsName - コンテンツ名
     * @戻値： true/false /null  
     * @作成日/作成者：2006/09/01 (IIS) N.Soma
     * @更新日/更新者：
     * @更新内容：
     */
	function getMailUserInfo($layerId, &$userName, &$mailAddress, &$contentsName) {

		// DBオブジェクトの生成
		$this->createDBObject();

		//マップ管理者情報を取得
		$adminuser_access = new AdminUserJoinMapTableAccess();
		$rtn = $adminuser_access->selectMapUserDataByDirectoryname($this->o_db,$this->directryname);		
		if ($rtn != true) {
			$adminuser_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S3_019_01"), E_USER_ERROR);
			return false;
		}
		if ($adminuser_access->getRecordCount() == 0){
			$adminuser_access->free();
			$this->disconnectDB();
			return null;
		}
		// データオブジェクトの取得		
		$adminuser_data = $adminuser_access->getOneData();
		$adminuser_access->free();
		
		$contentsName = $adminuser_data->getContentsName_Map();
		if (is_null($layerId) == true){
			//マップ管理者情報を取得
			$userName = $adminuser_data->getUserName_AdminUser();
			$mailAddress = $adminuser_data->getMailAddress_AdminUser();
		}else{
			//レイヤー管理者情報を取得
			$userjoinlayer_access = new UserJoinLayerTableAccess($this->directryname);
			$rtn = $userjoinlayer_access->selectLayerAndLayerMaker($this->o_db, $layerId);
			if ($rtn != true) {
				$userjoinlayer_access->free();
				$this->disconnectDB();
				trigger_error($this->o_resource->getValue("S3_019_01"), E_USER_ERROR);
				return false;
			}
			if ($userjoinlayer_access->getRecordCount() == 0){
				$userjoinlayer_access->free();
				$this->disconnectDB();
				return null;
			}
			// データオブジェクトの取得			
			$userjoinlayerdata = $userjoinlayer_access->getOneData();
			$userName = $userjoinlayerdata->getUserName_User();
			$mailAddress = $userjoinlayerdata->getMailAddress_User();
			// 解放
			$userjoinlayer_access->free();
			
		}
		// DB接続の切断
		$this->disconnectDB();

		return true;

	}


//#################################################################################################
//                                        以下、サービス内使用関数定義
//#################################################################################################

	/**
	* 関数概要：アップロードファイル削除処理
	* 処理詳細：アップロードされたファイルの削除を行う
	*
	* @引数： (I) layerId   - アイコン種別
	*      ： (I) icon      - イメージアイコン
	* @戻値： なし
	* @作成日/作成者：2006/09/01 (IIS)K.Shimomura
	* @更新日/更新者：
	* @更新内容：
	*/
	function deleteUploadFile($icontype, $icon) {
	
	    // アップロードファイルがある場合は削除
		if ($icontype == KAKIKOMAP_ICONTYPE_USER) {
		    if (file_exists("../img/icon/".$icon)) {
			    $ret = unlink("../img/icon/".$icon);
			}
		}

	}

	/**
	 * 関数名：意味空間情報変更通知の更新処理
     * 処理詳細：引数の何れかを元に意味空間の関連テーブルを精査し意味空間の警告表示フラグを立てる
     * @引数：(I) layerId    	- 対象レイヤーID
     *        (I) memoTypeId    - 対象メモ種類ID
     *        (I) memoId   		- 対象メモID
     *        (I) $flgType		- フラグの種類(1:メモ増加フラグ、2:メモ削除変更フラグ)
     * @戻値：bool			- true  正常終了
     *                      - false 異常終了
     * @作成日/作成者：2006/09/01 (IIS)N.Soma
     * @更新日/更新者：
	 * @注意事項：当関数は当サービス関数内からの呼び出しを前提とし、DBの接続・切断は行っていない。
	 *          ：また、トランザクションも制御していないので、必要に応じてコール元で制御すること。
     */
	function checkAndUpdateImiBaseInfo($layerId, $memoTypeId, $memoId, $flgType){

		// 意味空間基本情報テーブルアクセス
		$imibase_access = new ImiBaseInfoTableAccess($this->directryname);

		//レイヤーIDに対するチェック
		if (is_null($layerId) == false){
			// 意味空間レイヤーテーブルDBアクセス
			$imilayer_access = new ImiLayerTableAccess($this->directryname);
			$rtn = $imilayer_access->selectFindByKeyData($this->o_db, null, $layerId);
			if ($rtn != true) {
				$imilayer_access->free();
				trigger_error($this->o_resource->getValue("S3_018_01"), E_USER_ERROR);
				return false;
			}
			$imilayerlist = $imilayer_access->getDataList();
			$imispaceid_array = array();
			foreach($imilayerlist as $currentimilayer){
				array_push($imispaceid_array, $currentimilayer->getImiSpaceID());
			}
			if (count($imispaceid_array) > 0){
				//意味空間の警告表示フラグを立てる
				if ($flgType == 1){
					$rtn = $imibase_access->updateWarnFlgLump($this->o_db, $imispaceid_array, KAKIKOMAP_IMIBASE_CHANGE_ON, $flgType);
					
				}elseif ($flgType == 2){
					$rtn = $imibase_access->updateWarnFlgLump($this->o_db, $imispaceid_array, KAKIKOMAP_IMIBASE_WARN_ON, $flgType);					
				}
				if ($rtn != true) {
					$imibase_access->free();
					$imilayer_access->free();
					trigger_error($this->o_resource->getValue("S3_018_01"), E_USER_ERROR);
					return false;
				}
			}
			$imilayer_access->free();
		}
		
		//メモ種類IDに対するチェック
		if (is_null($memoTypeId) == false){
			// 意味空間メモ位置テーブルDBアクセス
			$imimemoposition_access = new ImiMemoPositionTableAccess($this->directryname);
			$rtn = $imimemoposition_access->selectFindDataByMemoTypeID($this->o_db, $memoTypeId);
			if ($rtn != true) {
				$imibase_access->free();
				$imimemoposition_access->free();
				trigger_error($this->o_resource->getValue("S3_018_01"), E_USER_ERROR);
				return false;
			}
			$imimemopositionlist = $imimemoposition_access->getDataList();
			$imispaceid_array = array();
			foreach($imimemopositionlist as $currentimimemoposition){
				array_push($imispaceid_array, $currentimimemoposition->getImiSpaceID());
			}
			
			if (count($imispaceid_array) > 0){				
				//意味空間の警告表示フラグを立てる
				if ($flgType == 1){
					$rtn = $imibase_access->updateWarnFlgLump($this->o_db, $imispaceid_array, KAKIKOMAP_IMIBASE_CHANGE_ON, $flgType);
					
				}elseif ($flgType == 2){
					$rtn = $imibase_access->updateWarnFlgLump($this->o_db, $imispaceid_array, KAKIKOMAP_IMIBASE_WARN_ON, $flgType);
				}
				if ($rtn != true) {
					$imibase_access->free();
					$imimemoposition_access->free();
					trigger_error($this->o_resource->getValue("S3_018_01"), E_USER_ERROR);
					return false;
				}
			}
			$imimemoposition_access->free();
		}

		//メモIDに対するチェック
		if (is_null($memoId) == false){
			// 意味空間メモ位置テーブルDBアクセス
			$imimemoposition_access = new ImiMemoPositionTableAccess($this->directryname);
			$rtn = $imimemoposition_access->selectFindByKeyData($this->o_db, null, $memoId);
			if ($rtn != true) {
				$imibase_access->free();
				$imimemoposition_access->free();
				trigger_error($this->o_resource->getValue("S3_018_01"), E_USER_ERROR);
				return false;
			}
			$imimemopositionlist = $imimemoposition_access->getDataList();
			$imispaceid_array = array();
			foreach($imimemopositionlist as $currentimimemoposition){
				array_push($imispaceid_array, $currentimimemoposition->getImiSpaceID());
			}
			if (count($imispaceid_array) > 0){
				//意味空間の警告表示フラグを立てる
				if ($flgType == 1){
					$rtn = $imibase_access->updateWarnFlgLump($this->o_db, $imispaceid_array, KAKIKOMAP_IMIBASE_CHANGE_ON, $flgType);
					
				}elseif ($flgType == 2){
					$rtn = $imibase_access->updateWarnFlgLump($this->o_db, $imispaceid_array, KAKIKOMAP_IMIBASE_WARN_ON, $flgType);		
				}
				if ($rtn != true) {
					$imibase_access->free();
					$imimemoposition_access->free();
					trigger_error($this->o_resource->getValue("S3_018_01"), E_USER_ERROR);
					return false;
				}
			}
			$imimemoposition_access->free();
		}
		$imibase_access->free();
		

		return true;
	
	}

    /**
     * 関数概要：ベースマップ情報データ取得
     * 処理詳細：ベースマップテーブルとマップ詳細テーブルを
     *         ：リンクしたデータの取得を行うクエリ作成を呼び出す
     *
     * @引数： 
     * @戻値： array   ベースマップリンクマップ詳細データクラス
     *      ：         DBエラーの場合はnull
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
	function getBaseMapInfoData() {

		// DBオブジェクトの生成
		$this->createDBObject();

		// ベースマップジョインマップ詳細テーブルDBアクセス
		$table_access = new BaseMapJoinMapDetailTableAccess();

		// クエリー発行
		$rtn = $table_access->selectMapLinkMapDetailListData($this->o_db, $this->directryname);
		if ($rtn != true) {
			$table_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S2_012_01"), E_USER_ERROR);
			return null;
		}

		// データオブジェクトの取得
		$array_data = $table_access->getDataList();

		// DB接続の切断
		$this->disconnectDB();

		return $array_data;
	}
	
    /**
    * 関数名：レイヤー１データ取得
    * @引数：(I) $layerId 	- レイヤーID
    * @戻値：LayerTableData	- レイヤーデータ
    *      ：null			- データが存在しない場合
    *      ：false			- 異常終了
    * @作成日/作成者：2006/09/01 (IIS) N.Soma
    * @更新日/更新者：
    * @概要：指定のレイヤーIDからレイヤーテーブルデータを取得する。
    */
    function getLayerOneData($layerId){
		// DBオブジェクトの生成
		$this->createDBObject();

		// 最優先レイヤー情報を取得
		$table_access = new LayerTableAccess($this->directryname);
    	$rtn = $table_access->selectLayerIDKeyData($this->o_db, $layerId);
 		if ($rtn != true) {
			$table_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S4_001_39"), E_USER_ERROR);
			return false;
		}
		if ($table_access->getRecordCount() == 0){
			return null;
		}
		$layerdata = $table_access->getOneData();

		$table_access->free();
		// DB接続の切断
		$this->disconnectDB();

		return $layerdata;
    }

}
?>
